package com.xforceplus.business.client.service;

import com.google.common.collect.Lists;
import com.xforceplus.api.model.ClientModel;
import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.dao.ClientDao;
import com.xforceplus.entity.Client;
import io.geewit.core.exception.ProcessedException;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.utils.uuid.UUID;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;


@Service
@CacheConfig(cacheNames = {"clientInfo"})
public class ClientSecretService {
    private final static Logger logger = LoggerFactory.getLogger(ClientSecretService.class);

    @Resource
    private ClientDao clientDao;

    @Resource
    private ClientTenantRelService clientTenantRelService;
    @Resource
    UserService userService;

    @Transactional(rollbackFor = Exception.class)
    public Client create(Client client) {
        //clientId查询是否已存在记录，存在不能添加
        if (this.existsByClientId(client.getClientId())) {
            throw new IllegalArgumentException("重复的clientId");
        }
        //校验userId
        if (client.getUserId() != null) {
            userService.findById(client.getUserId(), 0);
        }
        client.setSecret(UUID.randomUUID().toString());
        client = clientDao.saveAndFlush(client);
        logger.info(client.toString());
        if (!CollectionUtils.isEmpty(client.getTenantIds())) {
            clientTenantRelService.createBatch(client.getClientId(), client.getTenantIds());
        }
        return client;
    }

    /**
     * 根据clientId查询是否已存在记录
     *
     * @param clientId 客户端ID
     * @return Boolean true，表示已存在，false表示可以新建
     */
    private boolean existsByClientId(String clientId) {
        List<Client> clients = clientDao.findByClientId(clientId);
        if (clients == null || clients.isEmpty()) {
            return false;
        }
        return true;
    }


    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = "clientInfo", key = "#clientId")
    public Client update(String clientId, Client client) {
        Client existClient = this.findById(clientId);
        if (client.getUserId() != null) {
            userService.findById(client.getUserId());
        }
        BeanUtils.copyProperties(client, existClient, new String[]{"id"});

        if (!CollectionUtils.isEmpty(client.getTenantIds())) {
            clientTenantRelService.createBatch(clientId, client.getTenantIds());
        }

        return clientDao.saveAndFlush(existClient);
    }

    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(cacheNames = "clientInfo", key = "#clientId")
    public void delete(String clientId) {
        clientDao.deleteByClientId(clientId);
        clientTenantRelService.deleteByClientId(clientId);
    }

    /**
     * 根据Client查询数据
     *
     * @param clientId clientId
     * @return Client  Client
     */
    @Cacheable(cacheNames = "clientInfo", key = "#clientId")
    public Client findById(String clientId) {
        List<Client> clients = clientDao.findByClientId(clientId);
        if (clients == null || clients.isEmpty()) {
            throw new ProcessedException("未找到合法的client", HttpStatus.NOT_FOUND.value());
        }
        Client client = clients.stream().findFirst().get();
        List<Long> tenantIds = clientTenantRelService.findTenantIdByClientId(clientId);
        client.setTenantIds(tenantIds);
        return client;
    }

    public Client findOne(String clientId) {
        return this.findById(clientId);
    }

    public boolean existsByClientIdAndSecret(String clientId, String secret) {
        Client client = this.getByClientIdAndSecret(clientId, secret);
        return client != null;
    }

    public Client getByClientIdAndSecret(String clientId, String secret) {
        Client client = this.findById(clientId);
        if (client == null) {
            logger.warn("clientId not found");
            return null;
        }
        if (secret == null || !secret.equals(client.getSecret())) {
            logger.warn("wrong secret");
            return null;
        }

        if (client.getStatus() != null && client.getStatus()) {
            return client;
        } else {
            return null;
        }
    }

    public Client findByClientIdAndSecret(String clientId, String secret) {
        Client client = this.getByClientIdAndSecret(clientId, secret);
        if (client == null) {
            throw new ProcessedException("client不合法。", HttpStatus.NOT_FOUND.value());
        }
        return client;
    }

    public List<Client> findAll() {
        return Lists.newArrayList(clientDao.findAll());
    }

    public Page<Client> findAll(Pageable pageable) {
        return clientDao.findAll(pageable);
    }

    public List<Client> findBySecret(String secret) {
        Specification<Client> specification = (Root<Client> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) -> criteriaBuilder.and(criteriaBuilder.equal(root.<String>get("secret"), secret));
        return clientDao.findAll(specification);
    }


    public Page<Client> findByQuery(ClientModel.Request.Query query, Pageable pageable) {
        Specification<Client> specification = (Root<Client> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getAppId() != null && query.getAppId() > 0) {
                predicates.add(builder.equal(root.<Integer>get("appId"), query.getAppId()));
            }
            if (StringUtils.isNotBlank(query.getClientId())) {
                predicates.add(builder.like(root.<String>get("clientId"), "%" + query.getClientId() + "%"));
            }

            if (StringUtils.isNotBlank(query.getClientName())) {
                predicates.add(builder.like(root.<String>get("clientName"), "%" + query.getClientName() + "%"));
            }
            if (StringUtils.isNotBlank(query.getUserId())) {
                predicates.add(builder.equal(root.<String>get("userId"), query.getUserId()));
            }

            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return clientDao.findAll(specification, pageable);
    }

    public boolean existsBySecret(String secret) {
        Specification<Client> specification = (Root<Client> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) -> criteriaBuilder.and(criteriaBuilder.equal(root.<String>get("secret"), secret));
        return clientDao.count(specification) > 0L;
    }

    /**
     * client绑定用户信息
     *
     * @param clientId
     * @param userId
     * @return
     */
    @CacheEvict(cacheNames = "clientInfo", key = "#clientId")
    public Client bindUserId(String clientId, Long userId) {
        Client client = this.findById(clientId);
        if (client.getUserId() != null) {
            throw new ProcessedException("client已经绑定过用户", Rep.CommonCode.PARAM_FORMAT_WRONG, HttpStatus.BAD_REQUEST.value());
        }
        userService.findById(userId, 0);
        client.setUserId(userId);
        return clientDao.save(client);
    }

    /**
     * client解绑用户信息
     *
     * @param clientId
     * @return
     */
    @CacheEvict(cacheNames = "clientInfo", key = "#clientId")
    public Client unbindUserId(String clientId) {
        Client client = this.findById(clientId);
        if (client.getUserId() == null) {
            throw new ProcessedException("client未绑定用户", Rep.CommonCode.PARAM_FORMAT_WRONG, HttpStatus.BAD_REQUEST.value());
        }
        client.setUserId(null);
        return clientDao.save(client);
    }
}
