package com.xforceplus.business.company.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.company.CompanyApi;
import com.xforceplus.api.model.CompanyModel.Request;
import com.xforceplus.api.model.CompanyModel.Response;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.company.controller.vo.CompanyImportExcelRespVo;
import com.xforceplus.business.company.dto.CompanyTaxwareDto;
import com.xforceplus.business.company.service.CompanyExtensionService;
import com.xforceplus.business.company.service.CompanyProcess;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelService;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.tenant.service.TenantService;
import com.xforceplus.business.tenant.service.WrapperOrgService;
import com.xforceplus.domain.company.CompanyPackage;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.Tenant;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.utils.DateUtils;
import io.geewit.core.jackson.view.View;
import io.geewit.data.jpa.envers.domain.ComparedRevision;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.api.global.company.CompanyApi.Path.*;
import static com.xforceplus.api.utils.Separator.COMMA;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.COMPANY_CONFIG_EXCEL_FILE;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.COMPANY_EXCEL_FILE;

@SuppressWarnings("all")
@Api(value = "公司相关接口", description = "公司相关接口")
@Validated
@Controller
public class CompanyController extends AbstractController implements CompanyApi, com.xforceplus.api.tenant.company.CompanyApi, com.xforceplus.api.current.company.CompanyApi {
    private final static Logger logger = LoggerFactory.getLogger(CompanyController.class);

    private final WrapperOrgService orgService;

    private final TenantService tenantService;

    private final ExcelService excelService;

    private final CompanyExtensionService companyExtensionService;

    private final CompanyProcess companyProcess;


    private final ImportFileService importFileService;
    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;

    @Value("${company.move.hour:19}")
    private Integer companyMoveHour;
    @Value("${xforce.it.callback.enabled:false}")
    private boolean callbackEnabled;

    public CompanyController(WrapperOrgService orgService,
                             TenantService tenantService,
                             ExcelService excelService,
                             CompanyExtensionService companyExtensionService,
                             ImportFileService importFileService,
                             ExportFileService exportFileService,
                             CompanyProcess companyProcess) {
        this.orgService = orgService;
        this.tenantService = tenantService;
        this.excelService = excelService;
        this.importFileService = importFileService;
        this.companyExtensionService = companyExtensionService;
        this.exportFileService = exportFileService;
        this.companyProcess = companyProcess;
    }

    /**
     * 获取公司分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @ApiIgnore
    @JsonView(View.Page.class)
    @ApiOperation(value = "获取公司分页", notes = "通过分页参数，获取公司分页")
    @ResponseBody
    @RequestMapping(name = "公司分页列表查询", value = {API_PREFIX + "/query"}, method = RequestMethod.GET)
    public Page<Company> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<Company> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<Company> page = companyService.page(specification, pageable);
        return page;
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @JsonView(View.Page.class)
    @ApiOperation(value = "获取公司分页", notes = "通过分页参数，获取公司分页")
    @Override
    public ResponseEntity<Page<Company>> page(@ApiParam(value = "request") Request.Query query,
                                              @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        //公司查询默认查询所有状态的公司，否则管理端租户管理看到的公司数跟公司管理页面看到的公司数对不上
//        if (query.getStatus() == null) {
//            query.setStatus(1);
//        }
        Page<Company> result = companyService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    /**
     * 公司信息批量导出（异步）
     *
     * @param query 查询条件
     * @param sort  排序条件
     * @return ResponseEntity<String>
     */
    @ResponseBody
    @ApiOperation(value = "公司信息批量导出（异步）", notes = "公司信息批量导出（异步）")
    @RequestMapping(name = "公司信息批量导出（异步）", value = BATCH_EXPORT, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> exportCompany(@ApiParam(value = "request") @RequestBody Request.Query query,
                                                          @ApiParam(value = "sort") Sort sort) {
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .param(ExcelFile.PARAMS_SORT, sort)
                .build();
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), COMPANY_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(COMPANY_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @ResponseBody
    @ApiOperation(value = "批量服务配置异步导出", notes = "批量服务配置异步导出")
    @RequestMapping(name = "批量服务配置异步导出", value = BATCH_EXPORT_CONFIGURATION, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> exportCompanyConfig(@ApiParam(value = "request") @RequestBody Request.Query query,
                                                                @ApiParam(value = "sort") Sort sort) {
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .param(ExcelFile.PARAMS_SORT, sort)
                .build();
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), COMPANY_CONFIG_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(COMPANY_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @ApiIgnore
    @JsonView(View.List.class)
    @ApiOperation(value = "获取公司列表", notes = "通过参数，获取公司列表")
    @ResponseBody
    @RequestMapping(name = "公司分页列表", value = CompanyApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Company>> list(Request.Query query, Sort sort) {
        List<Company> result = companyService.list(query, sort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "获取公司详情", notes = "通过id，获取公司详情")
    @Override
    public ResponseEntity<Company> info(@ApiParam(value = "companyKey", required = true) String companyKey,
                                        @ApiParam(value = "byTaxNum", defaultValue = "false") boolean byTaxNum,
                                        @ApiParam(value = "revisionDate") Date revisionDate,
                                        @ApiParam(value = "withExtendParams") String withExtendParams) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        String[] withExtendParamArray = StringUtils.split(withExtendParams, COMMA);
        Set<String> withExtendParamSet;
        if (ArrayUtils.isEmpty(withExtendParamArray)) {
            withExtendParamSet = Collections.emptySet();
        } else {
            withExtendParamSet = Arrays.stream(withExtendParamArray).filter(Objects::nonNull).map(String::trim).collect(Collectors.toSet());
        }
        Company result = companyService.findByIdAndStatusAndRevisionDate(companyId, 1, revisionDate, withExtendParamSet);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "公司审计表更新是否有效")
    @Override
    public ResponseEntity<String> updateRevisionEffective(@ApiParam(value = "companyKey", required = true) String companyKey,
                                                          @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                          long revision,
                                                          Boolean effective) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        boolean result = companyService.updateAuditEffective(companyId, revision, effective);
        if (result) {
            return ResponseEntity.ok("更新成功");
        } else {
            return ResponseEntity.ok("更新失败");
        }
    }

    @Override
    public ResponseEntity<Page<ComparedRevision<Company, String>>> histories(@ApiParam(value = "companyKey", required = true) String companyKey,
                                                                             @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                                             Pageable pageable) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Page<ComparedRevision<Company, String>> page = companyService.findCompanyHistories(companyId, pageable);

        return ResponseEntity.ok(page);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "更新公司信息", notes = "通过参数，更新公司信息")
    @Override
    public ResponseEntity<Company> update(@ApiParam(value = "companyKey", required = true) String companyKey,
                                          @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                          @ApiParam(value = "model", required = true) Request.Save model) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Company result = companyService.update(companyId, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "更新公司状态", notes = "通过参数，更新公司信息")
    @Override
    public ResponseEntity<String> updateStatus(@ApiParam(value = "companyKey", required = true) String companyKey,
                                               @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                               @ApiParam(value = "status", required = true) int status) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        companyService.updateStatus(companyId, status);
        return ResponseEntity.ok("更新成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @JsonView(View.Page.class)
    @ApiOperation(value = "获取公司分页", notes = "通过分页参数，获取公司分页")
    @Override
    public ResponseEntity<Page<Company>> page(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                              @ApiParam(value = "request") Request.Query query,
                                              @ApiParam(value = "pageable") Pageable pageable) {
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<Company> result = companyService.page(tenantId, query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @ApiIgnore
    @JsonView(View.List.class)
    @ApiOperation(value = "获取公司列表", notes = "通过参数，获取公司列表")
    @ResponseBody
    @RequestMapping(name = "公司列表", value = com.xforceplus.api.tenant.company.CompanyApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Company>> list(@PathVariable(value = "tenantKey") String tenantKey,
                                              Request.Query query,
                                              Sort sort) {
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "createTime"));
        List<Company> result = companyService.list(tenantId, query, currentSort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "获取公司详情")
    @Override
    public ResponseEntity<Company> info(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                        @ApiParam(value = "companyKey", required = true) String companyKey,
                                        @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                        @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                        @ApiParam(value = "revisionDate", required = false) Date revisionDate,
                                        @ApiParam(value = "withExtendParams") String withExtendParams) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        String[] withExtendParamArray = StringUtils.split(withExtendParams, COMMA);
        Set<String> withExtendParamSet;
        if (ArrayUtils.isEmpty(withExtendParamArray)) {
            withExtendParamSet = Collections.emptySet();
        } else {
            withExtendParamSet = Arrays.stream(withExtendParamArray).filter(Objects::nonNull).map(String::trim).collect(Collectors.toSet());
        }
        Company result = companyService.revisionInfo(tenantId, companyId, revisionDate, withExtendParamSet);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<Page<ComparedRevision<Company, String>>> histories(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                                                             @ApiParam(value = "companyKey", required = true) String companyKey,
                                                                             @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                             @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                                             Pageable pageable) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Company company = companyService.findByTenantIdAndId(tenantId, companyId);
        Page<ComparedRevision<Company, String>> page = companyService.findCompanyHistories(companyId, pageable);

        return ResponseEntity.ok(page);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "在租户下创建公司")
    @Override
    public ResponseEntity<Company> create(String tenantKey,
                                          @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                          Request.Save model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        Tenant tenant = tenantService.findById(tenantId);
        //检查公司税号是否已经创建公司
        Optional<Company> optionalCompany = companyService.findOneByTaxNum(model.getTaxNum());
        if (optionalCompany.isPresent()) {
            throw new IllegalArgumentException("税号为：" + model.getTaxNum() + "已存在，请检查公司税号");
        }
        Company result = orgService.saveCompany(tenant, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "修改公司详情")
    @Override
    public ResponseEntity<Company> update(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                          @ApiParam(value = "companyKey", required = true) String companyKey,
                                          @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                          @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                          @ApiParam(value = "model", required = true) @Validated(ValidationGroup.OnUpdate.class) Request.Save model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Company result = companyService.update(tenantId, companyId, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:read"})
    @ApiOperation(value = "公司服务包分页查询")
    @Override
    public ResponseEntity<Page<CompanyPackage>> packages(Request.CompanyPackageQuery query, Pageable pageable) {
        Page<CompanyPackage> result = companyService.packagesByQuery(query, pageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:read"})
    @ApiOperation(value = "判断是否历史上存在该公司名及税号")
    @Override
    public ResponseEntity<Boolean> existInHistory(String taxNum, String companyName, Date effectiveDate) {
        boolean exist = companyService.existInHistory(taxNum, companyName, effectiveDate);
        return ResponseEntity.ok(exist);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "公司绑定服务包")
    @Override
    public ResponseEntity<String> bindPackages(String tenantKey,
                                               String companyKey,
                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                               Request.BindPackages bindPackage) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        companyService.bindPackages(tenantId, companyId, bindPackage);
        return ResponseEntity.ok("绑定成功");
    }

    /**
     * 替换为 {@link CompanyController#batchAsnycImport}
     *
     * @param file
     * @return ResponseEntity
     */
    @Deprecated
    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @ApiOperation(value = "批量导入企业excel")
    @ResponseBody
    @RequestMapping(name = "批量导入企业excel", value = CompanyApi.Path.BATCH_IMPORT, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity batchImport(@RequestParam("file") MultipartFile file) {
        return excelService.batchImportCompanies(file);
    }


    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @ApiOperation(value = "批量导入企业服务配置excel")
    @ResponseBody
    @RequestMapping(name = "批量导入企业服务配置excel", value = CompanyApi.Path.BATCH_IMPORT_CONFIGURATION, produces = {"application/json"}, method = RequestMethod.POST)
    @Deprecated
    public ResponseEntity<Long> batchImportConfiguration(@RequestParam("file") MultipartFile file) {
        return companyService.batchImportConfiguration(file);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @ApiOperation(value = "修复租户公司关系")
    @ResponseBody
    @RequestMapping(name = "修复租户公司关系", value = CompanyApi.Path.FIX_COMPANIES, method = RequestMethod.POST)
    public ResponseEntity<String> fixTenantCompanies() {
        companyService.fixTenantCompanies();
        return ResponseEntity.ok("修复成功");
    }

    /**
     * 异步批量导入企业excel
     *
     * @param file 文件
     * @return ResponseEntity<CompanyImportExcelRespVo>
     */
    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @ApiOperation(value = "异步批量导入企业excel")
    @ResponseBody
    @RequestMapping(name = "异步导入企业excel", value = CompanyApi.Path.BATCH_ASNYC_IMPORT, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<CompanyImportExcelRespVo> batchAsnycImport(@RequestParam("file") MultipartFile file) {

        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.create(file, COMPANY_EXCEL_FILE, userId, tenantId);
        CompanyImportExcelRespVo respVo = new CompanyImportExcelRespVo();
        respVo.setBusinessType(COMPANY_EXCEL_FILE.getName());
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @ApiOperation(value = "当前登录用户公司信息")
    @Override
    public ResponseEntity<Company> currentInfo(String companyKey,
                                               @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                               Date revisionDate,
                                               String withExtendParams) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        String[] withExtendParamArray = StringUtils.split(withExtendParams, COMMA);
        Set<String> withExtendParamSet;
        if (ArrayUtils.isEmpty(withExtendParamArray)) {
            withExtendParamSet = Collections.emptySet();
        } else {
            withExtendParamSet = Arrays.stream(withExtendParamArray).filter(Objects::nonNull).map(String::trim).collect(Collectors.toSet());
        }
        Company result = companyService.currentInfo(companyId, 1, revisionDate, withExtendParamSet);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<Page<ComparedRevision<Company, String>>> currentHistories(@ApiParam(value = "companyKey", required = true) String companyKey,
                                                                                    @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                                                    Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.histories(String.valueOf(tenantId), companyKey, false, byTaxNum, pageable);
    }

    @Override
    public ResponseEntity<Page<CompanyPackage>> currentPackages(Request.CompanyPackageQuery query, Pageable pageable) {
        Page<CompanyPackage> result = companyService.currentPackagesByQuery(query, pageable);
        return ResponseEntity.ok(result);
    }


    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @ApiOperation(value = "公司服务配置文件导入上传")
    @ResponseBody
    @RequestMapping(name = "公司服务配置文件导入上传", value = {"/api/global/company/file/excel"}, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> uploadExcelFile(@RequestParam("file") MultipartFile file) {

        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }

        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file,
                COMPANY_CONFIG_EXCEL_FILE.name(), userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(COMPANY_CONFIG_EXCEL_FILE.name());
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }


    @AuthorizedDefinition(resources = {"xforce:business:company:save"})
    @ApiOperation(value = "批量修改公司（名称和税号）excel")
    @ResponseBody
    @RequestMapping(name = "批量修改公司（名称和税号）excel", value = CompanyApi.Path.BATCH_IMPORT_UPDATE_COMPANY, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> batchImportUpdate(@RequestParam("file") MultipartFile file) {
        IAuthorizedUser currentUser = UserInfoHolder.currentUser();
        Long tenantId = currentUser.getTenantId();
        Long userId = currentUser.getId();

        String businessType = ExcelConfigBusinessType.COMPANY_NAME_TAX_NUM_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file, businessType, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @ApiOperation(value = "组织公司编号导入excel")
    @ResponseBody
    @RequestMapping(name = "组织公司编号批量导入excel", value = com.xforceplus.api.current.company.CompanyApi.Path.BATCH_IMPORT_ORG_COM_EXCEL, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> batchImportOrgCompanyNumber(@RequestParam("file") MultipartFile file) {

        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }

        String businessType = ExcelConfigBusinessType.ORG_COMPANY_NUM_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file, businessType, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:move"})
    @Override
    public ResponseEntity<Map<String, Object>> move(String companyKey,
                                                    Request.Move move) {
        long companyId = super.toCompanyId(companyKey, move.getByTaxNum());
        move.setCompanyId(companyId);
        long startTime = DateUtils.getCustomTime(companyMoveHour, 0, 0);
        long currentTime = System.currentTimeMillis();
        if (currentTime < startTime) {
            throw new IllegalArgumentException("请在" + companyMoveHour + ":00:00至23:59:59发起请求！");
        }
        Map<String, Object> result = companyService.move(move);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company:move"})
    @Override
    public ResponseEntity<Map<String, Object>> moveAll(String companyKey,
                                                       Request.Move move) {
        long companyId = super.toCompanyId(companyKey, move.getByTaxNum());
        move.setCompanyId(companyId);
        long startTime = DateUtils.getCustomTime(companyMoveHour, 0, 0);
        long currentTime = System.currentTimeMillis();
        if (currentTime < startTime) {
            throw new IllegalArgumentException("请在" + companyMoveHour + ":00:00至23:59:59发起请求！");
        }
        Map<String, Object> result = companyService.moveAll(move);
        return ResponseEntity.ok(result);
    }

    @Override
    @ApiOperation(value = "公司关联租户详情列表")
    public ResponseEntity<Response.CompanyInfo> tenantRels(String companyKey,
                                                           @ApiParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum) {
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Response.CompanyInfo result = companyService.findCompanyInfoByCompanyId(companyId);
        return ResponseEntity.ok(result);
    }

    @Override
    @ApiOperation(value = "修改公司名称")
    public ResponseEntity<String> updateCompanyName(String tenantKey,
                                                    Boolean byTenantCode,
                                                    Request.CompanyName companyName) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        if (null != UserInfoHolder.get()) {
            Long currentTenantId = UserInfoHolder.get().getTenantId();
            if (currentTenantId == null || currentTenantId != tenantId) {
                throw new IllegalArgumentException("租户信息与当前用户不匹配！");
            }
        }
        companyService.updateCompanyName(tenantId, companyName);
        return ResponseEntity.ok("Success");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:org:terminal"})
    @Override
    @ApiOperation(value = "设置或者更新公司终端鉴权")
    public ResponseEntity<Boolean> setCompanyTerminalAuthSwitch(@ApiParam(name = "tenantKey", value = "租户ID(tenantId)") String tenantKey,
                                                                @ApiParam(name = "companyKey", value = "公司ID或者税号") String companyKey,
                                                                @ApiParam(name = "byTenantCode", value = "是否按租户Code查", required = false, defaultValue = "false") Boolean byTenantCode,
                                                                @ApiParam(name = "byTaxNum", value = "根据税号还是公司ID操作，true按税号，false按公司ID", defaultValue = "false") Boolean byTaxNum,
                                                                @ApiParam(name = "terminalSwitch", value = "终端鉴权开关，1表示打开，0关闭") int terminalSwitch) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        boolean result = companyExtensionService.setTerminalSwitch(tenantId, companyId, terminalSwitch);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:org:terminal"})
    @ApiOperation(value = "查询公司终端鉴权开关状态")
    @Override
    public ResponseEntity<Integer> getCompanyTerminalAuthSwitch(
            @ApiParam(name = "tenantKey", value = "租户ID(tenantId)") String tenantKey,
            @ApiParam(name = "companyKey", value = "公司ID或者税号") String companyKey,
            @ApiParam(name = "byTenantCode", value = "是否按租户Code查", required = false, defaultValue = "false") Boolean byTenantCode,
            @ApiParam(name = "byTaxNum", value = "根据税号还是公司ID操作，true按税号，false按公司ID", defaultValue = "false") Boolean byTaxNum) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Integer termanalSwitch = companyExtensionService.getCompanyTerminalSwitch(tenantId, companyId);
        return ResponseEntity.ok(termanalSwitch);
    }

    @ApiOperation(value = "验证租户下公指定公司的多个服务包是否被角色绑定")
    @Override
    public ResponseEntity<Map<Long, Boolean>> validCompaniesPackagesBound(
            @ApiParam(name = "tenantKey", value = "租户ID(tenantId)") String tenantKey,
            @ApiParam(name = "companyKey", value = "公司ID或者税号") String companyKey,
            @ApiParam(name = "byTenantCode", value = "是否按租户Code查", required = false, defaultValue = "false") Boolean byTenantCode,
            @ApiParam(name = "byTaxNum", value = "根据税号还是公司ID操作, true按税号, false按公司ID", defaultValue = "false") Boolean byTaxNum,
            @ApiParam(name = "packageIds", value = "服务包id集合, ‘,’分割") String packageIds) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long companyId = super.toCompanyId(companyKey, byTaxNum);
        Set<Long> packageIdSet = new HashSet<>();
        for (String packageIdStr : StringUtils.split(packageIds, ",")) {
            try {
                Long packageId = Long.parseLong(packageIdStr);
                packageIdSet.add(packageId);
            } catch (NumberFormatException ignored) {
            }
        }

        Map<Long, Boolean> result = companyService.validateCompanyPackagesWereBoundByRole(tenantId, companyId, packageIdSet);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<String> changeHostTenant(Long companyId, Long tenantId) {
        companyService.changeHostTenant(companyId, tenantId);
        return ResponseEntity.ok();
    }

    @ResponseBody
    @PostMapping(name = "集成平台回调税件推送消息接口", value = TAXWARE_CALLBACK)
    @ApiOperation(value = "集成平台回调税件推送消息接口")
    public ResponseEntity<Boolean> taxwareMsgCallBack(@RequestBody CompanyTaxwareDto companyTaxwareDto) {
        if (callbackEnabled) {
            //仅当开启回调才处理消息，saas上这个功能不需要打开，仅给属地回调使用
            boolean result = companyService.handleTaxwareMessage(companyTaxwareDto);
            return ResponseEntity.ok(result);
        } else {
            logger.warn("callback not enabled would not handle this request");
            return ResponseEntity.ok(true);
        }
    }

    @Override
    public ResponseEntity<String> removeTenant(Long companyId, Long tenantId) {
        companyProcess.removeTenant(companyId, tenantId);
        return ResponseEntity.ok();
    }
}
