package com.xforceplus.business.company.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.company.CompanyServiceApi;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.api.model.CompanyModel.Request.BindCompanyPackages;
import com.xforceplus.api.model.CompanyModel.Request.BindTaxNumAndPackage;
import com.xforceplus.api.model.CompanyModel.Request.ChangePackageStatus;
import com.xforceplus.api.model.CompanyModel.Request.CompanyPackageDetail;
import com.xforceplus.api.model.CompanyServiceRelModel.Request.CompanyPackageExport;
import com.xforceplus.api.model.CompnayResourceSetModel;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.api.model.ServicePackageModel;
import com.xforceplus.business.company.service.CompanyServicePackageService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelService;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.FileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.resource.service.ServicePackageService;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.domain.company.CompanyPackageExportDto;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.entity.ServicePackage;
import com.xforceplus.tenant.core.exception.UnknownException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import io.geewit.core.jackson.view.View;
import io.geewit.web.utils.JsonUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.LinkedHashMap;
import java.util.List;

import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.COMPANY_PACKAGE_IMPORT;

@SuppressWarnings("all")
@Api(value = "公司服务包相关接口", description = "公司服务包相关接口")
@Controller
public class CompanyServiceController implements CompanyServiceApi {

    private final CompanyServicePackageService companyServicePackageService;
    private final OrgService orgService;
    private final ServicePackageService servicePackageService;
    private final ExcelService excelService;
    private final FileService fileService;
    private final ImportFileService importFileService;
    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;
    Logger logger = LoggerFactory.getLogger(CompanyServiceController.class);


    public CompanyServiceController(CompanyServicePackageService companyServicePackageService, OrgService orgService,
                                    ServicePackageService servicePackageService, ExcelService excelService, FileService fileService,
                                    ImportFileService importFileService, ExportFileService exportFileService) {
        this.companyServicePackageService = companyServicePackageService;
        this.orgService = orgService;
        this.servicePackageService = servicePackageService;
        this.excelService = excelService;
        this.fileService = fileService;
        this.importFileService = importFileService;
        this.exportFileService = exportFileService;
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "绑定公司和服务包2.0-7", notes = "如果服务包不存在就用功能集合创建服务包")
    @JsonView(View.Page.class)
    @Override
    public ResponseEntity<String> bindCompanyResourceSet(List<CompnayResourceSetModel.Request.CompanyPackageBind> companyPackageBindRequest) {
        companyServicePackageService.createServicePackageAndBindCompanyFromResourceSet(companyPackageBindRequest);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "公司绑定服务包")
    @JsonView(View.Page.class)
    @Override
    public ResponseEntity<String> bindPackages(Long tenantId, BindTaxNumAndPackage bindPackage) {
        companyServicePackageService.bindCompanyAndPackage(tenantId, bindPackage);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "公司绑定服务包")
    @Override
    public ResponseEntity<String> bindCompanyPackages(BindCompanyPackages bindCompanyPackages) {
        logger.info("bindCompanyPackages==={}", JsonUtils.toJson(bindCompanyPackages));
        if (bindCompanyPackages.getCompanyId() == null && CollectionUtils.isEmpty(bindCompanyPackages.getCompanyIds())) {
            throw new IllegalArgumentException("公司id不能为空");
        }
        companyServicePackageService.bindCompanyPackages(bindCompanyPackages);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:read"})
    @ApiOperation(value = "获取公司服务包详情")
    @Override
    public ResponseEntity<CompanyPackageDetail> info(long tenantId, long companyId) {
        OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
        orgQuery.setTenantId(tenantId);
        orgQuery.setCompanyId(companyId);
        OrgStruct org = orgService.findOne(orgQuery).orElseThrow(() -> new IllegalArgumentException("找不到对应的组织信息"));

        ServicePackageModel.Request.Query packageQuery = new ServicePackageModel.Request.Query();
        packageQuery.setTenantId(tenantId);
        packageQuery.setCompanyId(companyId);
        packageQuery.setWithExtendParams("resourcesets");
        List<ServicePackage> packages = servicePackageService.list(packageQuery, Sort.unsorted());
        CompanyPackageDetail<ServicePackage> companyPackageDetail = new CompanyPackageDetail<>();
        companyPackageDetail.setCompanyId(companyId);
        companyPackageDetail.setCompanyName(org.getCompanyName());
        companyPackageDetail.setTenantId(tenantId);
        companyPackageDetail.setTenantName(org.getTenantName());
        companyPackageDetail.setPackages(packages);
        return ResponseEntity.ok(companyPackageDetail);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "修改公司服务包状态")
    @Override
    public ResponseEntity<String> changeCompanyServiceStatus(Long companyId, Long tenantId, CompanyModel.Request.ChangeCompanyPackageStatus changePackageStatus) {
        companyServicePackageService.changeCompanyServiceStatus(
                tenantId,
                companyId,
                changePackageStatus.getServicePackageIds(),
                changePackageStatus.getStatus(),
                changePackageStatus.getOperateReason());
        return ResponseEntity.ok("修改成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "根据租户id和公司id批量修改公司服务包状态")
    @Override
    public ResponseEntity<String> changeStatus(ChangePackageStatus changePackageStatus) {
        companyServicePackageService.changeStatusByTenantIdAndCompanyId(
                changePackageStatus.getTenantId(),
                changePackageStatus.getCompanyId(),
                changePackageStatus.getStatus(),
                changePackageStatus.getOperateReason());
        return ResponseEntity.ok("修改成功");
    }

    @ResponseBody
    @ApiOperation(value = "解绑公司服务包")
    @Override
    public ResponseEntity<String> delete(long id) {
        companyServicePackageService.unbindCompanyPackages(id);
        return ResponseEntity.ok("解绑成功");
    }

    @ResponseBody
    @ApiOperation(value = "删除公司服务包")
    @Override
    public ResponseEntity<String> deleteByCompanyIdAndTenantId(Long companyId, Long tenantId) {
        companyServicePackageService.deleteByCompanyIdAndTenantId(companyId, tenantId);
        return ResponseEntity.ok();
    }

    /**
     * 导入公司服务包excel
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:save"})
    @ApiOperation(value = "导入公司服务包excel")
    @RequestMapping(name = "导入公司服务包excel",
            value = CompanyServiceApi.Path.BATCH_IMPORT,
            produces = {"application/json"},
            method = RequestMethod.POST)
    @ResponseBody
    @Deprecated
    public ResponseEntity batchImport(@RequestParam("file") MultipartFile file) {
        return excelService.batchImportCompanyPackages(file);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:read"})
    @Override
    @Deprecated
    public ResponseEntity export(@RequestBody CompanyPackageExport export) {
        if (CollectionUtils.isEmpty(export.getCompanyIds())) {
            throw new UnknownException("未选择公司");
        }

        if (CollectionUtils.isEmpty(export.getTenantIds())) {
            throw new UnknownException("未选择租户");
        }
        List<CompanyPackageExportDto> companyPackageExportDtos =
                companyServicePackageService.getExportData(export.getCompanyIds(), export.getTenantIds());
        if (CollectionUtils.isEmpty(companyPackageExportDtos)) {
            throw new UnknownException("导出失败，没有获取到公司");
        }

        LinkedHashMap<String, String> rules = new LinkedHashMap<>();

        rules.put("tenantCode", "租户代码");
        rules.put("taxNum", "公司税号");
        rules.put("packageCodes", "产品服务包代码");
        rules.put("statusName", "状态");

        Long fileId;
        try {
            String json = JSON.toJSONString(companyPackageExportDtos, SerializerFeature.WriteMapNullValue);
            fileId = fileService.exportDataExcel(json, rules, "公司服务包导出列表");
        } catch (Exception e) {
            logger.error("上传文件失败", e);
            throw new UnknownException("公司服务包导出失败");
        }

        return ResponseEntity.ok(fileId);
    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:read"})
    @ApiOperation(value = "公司服务包批量异步导出", notes = "公司服务包批量异步导出")
    @Override
    public ResponseEntity batchAsyncExport(@RequestBody CompanyPackageExport export) {
        if (ArrayUtils.isEmpty(export.getSheets())) {
            throw new UnknownException("请选择SheetName");
        }

        if (CollectionUtils.isEmpty(export.getCompanyIds())) {
            throw new UnknownException("未选择公司");
        }

        if (CollectionUtils.isEmpty(export.getTenantIds())) {
            throw new UnknownException("未选择租户");
        }

        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, export)
                .build();
        BusinessType businessType = COMPANY_PACKAGE_IMPORT;
        ExcelFileStore excelFileStore = exportFileService.asyncExcelExport(queryParam.params(), businessType);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(businessType.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);

    }

    @AuthorizedDefinition(resources = {"xforce:business:company-service-package:read"})
    @ApiOperation(value = "公司服务包批量导入上传")
    @ResponseBody
    @RequestMapping(name = "公司服务包批量导入上传", value = {"/api/global/companypackage/file/excel"}, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> uploadExcelFile(@RequestParam("file") MultipartFile file) {

        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }
        String businessTypeName = ExcelConfigBusinessType.COMPANY_PACKAGE_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file,
                businessTypeName, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessTypeName);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

}
