package com.xforceplus.business.company.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.current.company.CompanyTenantRelOperationApi;
import com.xforceplus.api.model.CompanyTenantRelOperationModel.*;
import com.xforceplus.business.company.service.CompanyTenantRelOperationProcess;
import com.xforceplus.business.company.service.CompanyTenantRelOperationService;
import com.xforceplus.domain.company.CompanyTenantRelOperationDto;
import com.xforceplus.entity.CompanyTenantRelOperation;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.domain.company.CompanyTenantRelOperationDto.*;


/**
 * @author geewit
 */
@SuppressWarnings("all")
@Api(value = "公司租户关联邀请相关接口", description = "公司租户关联邀请相关接口")
@Validated
@Controller
public class CompanyTenantRelOperationController implements CompanyTenantRelOperationApi {
    private final static Logger logger = LoggerFactory.getLogger(CompanyTenantRelOperationController.class);

    private final static Set<String> WITH_EXTEND_PARAM_SET = Stream.of("hostTenantName").collect(Collectors.toSet());

    private final CompanyTenantRelOperationService operationService;

    @Autowired
    private CompanyTenantRelOperationProcess operationProcess;

    public CompanyTenantRelOperationController(CompanyTenantRelOperationService operationService) {
        this.operationService = operationService;
    }

    @ApiOperation(value = "发起邀请公司租户关联")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:relationship:save"})
    @Override
    public ResponseEntity<CompanyTenantRelOperation> invite(Request.Invitation model) {
        CompanyTenantRelOperation result = operationService.invite(model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "我邀请的公司租户关联邀请分页列表")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:read"})
    @Override
    public ResponseEntity<Page<CompanyTenantRelOperation>> invitedPage(Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTenantId(authorizedUser.getTenantId());
        query.setOperationType(OPERATION_TYPE_INVITE);
        if(StringUtils.isNotBlank(query.getWithExtendParams())) {
            Set<String> withExtendParamSet = Stream.of(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet());
            withExtendParamSet.addAll(withExtendParamSet);
            query.setWithExtendParamSet(withExtendParamSet);
        } else {
            query.setWithExtendParamSet(WITH_EXTEND_PARAM_SET);
        }
        Page<CompanyTenantRelOperation> page = operationService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }


    @ApiOperation(value = "被取消的授权分页列表")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:read"})
    @Override
    public ResponseEntity<Page<CompanyTenantRelOperation>> myCanceledPage(Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTenantId(authorizedUser.getTenantId());
        query.setOperationType(OPERATION_TYPE_UNRELATE);
        if(StringUtils.isNotBlank(query.getWithExtendParams())) {
            Set<String> withExtendParamSet = Stream.of(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet());
            withExtendParamSet.addAll(withExtendParamSet);
            query.setWithExtendParamSet(withExtendParamSet);
        } else {
            query.setWithExtendParamSet(WITH_EXTEND_PARAM_SET);
        }
        Page<CompanyTenantRelOperation> page = operationService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @ApiOperation(value = "被取消的关联分页列表")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:read"})
    @Override
    public ResponseEntity<Page<CompanyTenantRelOperation>> canceledPage(Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setRelatedTenantId(authorizedUser.getTenantId());
        query.setOperationType(OPERATION_TYPE_UNRELATE);
        if(StringUtils.isNotBlank(query.getWithExtendParams())) {
            Set<String> withExtendParamSet = Stream.of(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet());
            withExtendParamSet.addAll(withExtendParamSet);
            query.setWithExtendParamSet(withExtendParamSet);
        } else {
            query.setWithExtendParamSet(WITH_EXTEND_PARAM_SET);
        }
        Page<CompanyTenantRelOperation> page = operationService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @ApiOperation(value = "待处理的公司租户关联邀请分页列表")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:read"})
    @Override
    public ResponseEntity<Page<CompanyTenantRelOperation>> processingPage(Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        query.setRelatedTenantId(authorizedUser.getTenantId());
        if(StringUtils.isNotBlank(query.getWithExtendParams())) {
            Set<String> withExtendParamSet = Stream.of(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet());
            withExtendParamSet.addAll(withExtendParamSet);
            query.setWithExtendParamSet(withExtendParamSet);
        } else {
            query.setWithExtendParamSet(WITH_EXTEND_PARAM_SET);
        }
        query.setOperationType(0);
        query.setWithoutStatus(CompanyTenantRelOperationDto.Status.CANCELED.value());
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<CompanyTenantRelOperation> page = operationService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @ApiOperation(value = "公司租户关联邀请信息")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:read"})
    @Override
    public ResponseEntity<CompanyTenantRelOperation> info(long operationId, String withExtendParams) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Set<String> withExtendParamSet;
        if(StringUtils.isNotBlank(withExtendParams)) {
            withExtendParamSet = Stream.of(StringUtils.split(withExtendParams, ",")).collect(Collectors.toSet());
            withExtendParamSet.addAll(WITH_EXTEND_PARAM_SET);
        } else {
            withExtendParamSet = WITH_EXTEND_PARAM_SET;
        }
        CompanyTenantRelOperation result = operationService.findById(operationId, withExtendParamSet);
        if(!authorizedUser.getTenantId().equals(result.getCreaterTenantId()) && !authorizedUser.getTenantId().equals(result.getRelatedTenantId()) && !authorizedUser.getTenantId().equals(result.getTenantId())) {
            throw new IllegalArgumentException("当前账户所在租户没有查看该记录信息的权限");
        }
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "公司租户关联邀请接受")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:relationship:save"})
    @Override
    public ResponseEntity<String> accept(long operationId, String remark) {
        operationProcess.accept(operationId, remark);
        return ResponseEntity.ok("邀请接受成功");
    }

    @ApiOperation(value = "公司租户关联邀请拒绝")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:relationship:save"})
    @Override
    public ResponseEntity<String> refuse(long operationId, String remark) {
        operationProcess.refuse(operationId, remark);
        return ResponseEntity.ok("邀请拒绝成功");
    }

    @ApiOperation(value = "公司租户关联邀请撤销")
    @AuthorizedDefinition(resources = {"xforce:ucenter:multitanent:relationship:save"})
    @Override
    public ResponseEntity<String> cancel(long operationId) {
        operationService.cancel(operationId);
        return ResponseEntity.ok("邀请撤销成功");
    }
}
