package com.xforceplus.business.company.service;

import com.xforceplus.api.model.CompanyApplyModel;
import com.xforceplus.api.model.CompanyApplyModel.Request.ApplicationProcess;
import com.xforceplus.api.model.CompanyApplyModel.Request.Query;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.dao.CompanyApplyDao;
import com.xforceplus.dao.CompanyDao;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.CompanyApply;
import com.xforceplus.entity.Tenant;
import com.xforceplus.query.CompanyApplyQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.utils.uuid.UUID;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

@SuppressWarnings("all")
@Service
public class CompanyApplyService {
    private final static Logger logger = LoggerFactory.getLogger(CompanyApplyService.class);

    private final CompanyApplyDao companyApplyDao;

    private final OrgStructDao orgStructDao;

    private final CompanyDao companyDao;

    private final CompanyService companyService;

    private final TenantDao tenantDao;


    public CompanyApplyService(CompanyApplyDao companyApplyDao, OrgStructDao orgStructDao, CompanyDao companyDao, CompanyService companyService, TenantDao tenantDao) {
        this.companyApplyDao = companyApplyDao;
        this.orgStructDao = orgStructDao;
        this.companyDao = companyDao;
        this.companyService = companyService;
        this.tenantDao = tenantDao;
    }

    public CompanyApply apply(long tenantId, long orgId, CompanyModel.Request.Save model) {
        CompanyApply entity = new CompanyApply();
        BeanUtils.copyProperties(
                model, entity
                , null
                , Stream.of("cquota", "squota", "ceQuota", "juQuota", "seQuota", "vehicleLimit").toArray(String[]::new));
        if (model.getHostTenantId() != null && model.getHostTenantId() > 0) {
            entity.setTenantId(model.getHostTenantId());
            entity.setHostTenantId(model.getHostTenantId());
        } else {
            entity.setTenantId(tenantId);
            entity.setHostTenantId(tenantId);
        }
        entity.setOrgStructId(orgId);
        return companyApplyDao.saveAndFlush(entity);
    }

    public Page<CompanyApply> page(Query query, Pageable pageable) {
        Specification<CompanyApply> specification = CompanyApplyQueryHelper.querySpecification(query);
        Page<CompanyApply> page = companyApplyDao.findAll(specification, pageable);
        return page;
    }

    public Page<CompanyApply> page(Specification<CompanyApply> specification, Pageable pageable) {
        return companyApplyDao.findAll(specification, pageable);
    }

    public List<CompanyApply> list(Query query, Sort sort) {
        Specification<CompanyApply> specification = CompanyApplyQueryHelper.querySpecification(query);
        List<CompanyApply> list = companyApplyDao.findAll(specification, sort);
        return list;
    }

    public List<CompanyApply> list(Specification<CompanyApply> specification, Sort sort) {
        return companyApplyDao.findAll(specification, sort);
    }


    public CompanyApply findById(long applyId) {
        return companyApplyDao.findById(applyId).orElseThrow(() -> new IllegalArgumentException("未找到公司申请实体"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long applicationId, int status) {
        CompanyApply existEntity = this.findById(applicationId);
        existEntity.setStatus(status);
        companyApplyDao.saveAndFlush(existEntity);
    }

    public Company initCompany(CompanyApply companyApply, String[] ignoreProperties, String reason, int status) {
        Company company = new Company();
        BeanUtils.copyProperties(
                companyApply, company, ignoreProperties,
                Stream.of("cquota", "squota", "ceQuota", "juQuota", "seQuota", "vehicleLimit").toArray(String[]::new));
        BeanUtils.copyProperties(companyApply, company);
        company.setOperateReason(reason);
        if (StringUtils.isBlank(company.getCompanyCode())) {
            String companyCode = UUID.randomUUID().toString();
            company.setCompanyCode(companyCode);
        }
        company.setStatus(status);
        return company;
    }

    @Transactional(rollbackFor = Exception.class)
    public void approve(long applicationId, String reason) {

        CompanyApply companyApply = this.findById(applicationId);
        final Long orgId = companyApply.getOrgStructId();
        if (orgId == null || orgId <= 0) {
            throw new IllegalArgumentException("审核信息中的组织id不能空");
        }

        final Long tenantId = companyApply.getTenantId();
        if (tenantId == null || tenantId <= 0) {
            throw new IllegalArgumentException("审核信息中的租户id不能空");
        }
        Tenant tenant = tenantDao.findById(tenantId).orElseThrow(() -> new IllegalArgumentException("错误的租户id(" + tenantId + ")"));
        //判断（代码、税号、名称） 已经有重复的税号或公司名称或代码 如存在则throw new "已经有重复的税号或公司名称或代码"
        List<Company> existCompanies = companyDao.findByTaxNum(companyApply.getTaxNum());
        Company company;
        if (!existCompanies.isEmpty()) {
            company = existCompanies.stream().findFirst().get();
        } else {
            company = this.initCompany(companyApply, null, reason, 1);
            company.setHostTenantId(tenantId);
            company = companyDao.saveAndFlush(company);
        }

        orgStructDao.approve(orgId, company.getCompanyId(), 1, 1);

        companyService.saveTenantCompany(tenant, company);
        companyApplyDao.deleteById(applicationId);
        List<CompanyApply> companyApplies = companyApplyDao.findByTaxNum(company.getTaxNum());
        if (!companyApplies.isEmpty()) {
            for (CompanyApply remainApply : companyApplies) {
                orgStructDao.approve(remainApply.getOrgStructId(), company.getCompanyId(), 1, 1);
                Tenant existTenant = tenantDao.findById(remainApply.getTenantId()).get();
                Company comppany = new Company();
                company.setCompanyName(remainApply.getCompanyName());
                company.setCompanyCode(remainApply.getCompanyCode());
                company.setCompanyId(remainApply.getCompanyId());
                company.setTaxNum(remainApply.getTaxNum());
                companyService.saveTenantCompany(existTenant, comppany);
                companyApplyDao.deleteById(remainApply.getCompanyApplyId());
            }
        }
    }

    /**
     * 判断是否已经存在
     *
     * @param companyCode 公司代码
     * @param taxNum      税号
     * @param companyName 公司名称
     */
    public void isCompanyExists(String companyCode, String taxNum) {
        if (companyDao.validateExistCompany(companyCode, taxNum) > 0) {
            throw new IllegalArgumentException("已经有重复的税号或代码");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void decline(long applicationId, String reason) {
        CompanyApply existEntity = this.findById(applicationId);
        companyApplyDao.deleteById(existEntity.getCompanyApplyId());
        if (existEntity.getOrgStructId() != null && existEntity.getOrgStructId() > 0) {
            orgStructDao.deleteById(existEntity.getOrgStructId());
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public CompanyApplyModel.Response.ApplicationProcessResult approve(ApplicationProcess applicationProcess) {
        return this.processApplications(applicationProcess, 3);
    }

    @Transactional(rollbackFor = Exception.class)
    public CompanyApplyModel.Response.ApplicationProcessResult decline(ApplicationProcess applicationProcess) {
        return this.processApplications(applicationProcess, 2);
    }

    private CompanyApplyModel.Response.ApplicationProcessResult processApplications(ApplicationProcess applicationProcess, int status) {
        List<Long> applicationIds = applicationProcess.getApplicationIds();
        Iterable<CompanyApply> existEntities = companyApplyDao.findAllById(applicationIds);
        int total = applicationIds.size();
        int success = 0;
        for (CompanyApply existEntity : existEntities) {
            if (existEntity.getStatus() != null && existEntity.getStatus() == status) {
                continue;
            }
            existEntity.setStatus(status);
            existEntity.setOperateReason(applicationProcess.getReason());
            try {
                if (status == 3) {
                    final Long orgId = existEntity.getOrgStructId();
                    if (orgId == null || orgId <= 0) {
                        /**
                         * 此日志打印无法提供排查问题有效信息
                         */
                      //  logger.warn("审核信息中的组织id不能空");
                        continue;
                    }

                    final Long tenantId = existEntity.getTenantId();
                    if (tenantId == null || tenantId <= 0) {
                        /**
                         * 此日志打印无法提供排查问题有效信息
                         */
                       // logger.warn("审核信息中的租户id不能空");
                        continue;
                    }
                    List<Company> existCompanies = companyDao.findByTaxNum(existEntity.getTaxNum());
                    Company company;
                    if (!existCompanies.isEmpty()) {
                        company = existCompanies.stream().findFirst().get();
                    } else {
                        company = this.initCompany(existEntity, Stream.of("orgs", "companyNos", "hostTenant", "tenantRels").toArray(String[]::new), applicationProcess.getReason(), 1);
                        company.setHostTenantId(tenantId);
                        company = companyDao.saveAndFlush(company);
                    }

                    try {
                        orgStructDao.approve(orgId, company.getCompanyId(), 1, 1);
                        Tenant tenant = tenantDao.findById(existEntity.getTenantId()).orElseThrow(() -> new IllegalArgumentException("错误的租户id(" + existEntity.getTenantId() + ")"));
                        companyService.saveTenantCompany(tenant, company);
                        companyApplyDao.deleteById(existEntity.getCompanyApplyId());
                    } catch (Exception e) {
                        logger.warn(e.getMessage(), e);
                        continue;
                    }
                } else if (status == 2) {
                    if (existEntity.getOrgStructId() != null && existEntity.getOrgStructId() > 0) {
                        orgStructDao.deleteById(existEntity.getOrgStructId());
                    }
                }
                try {
                    companyApplyDao.deleteById(existEntity.getCompanyApplyId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }

                success++;
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
            }
        }
        int fail = total - success;
        CompanyApplyModel.Response.ApplicationProcessResult result = new CompanyApplyModel.Response.ApplicationProcessResult();
        result.setSuccess(success);
        result.setFail(fail);
        return result;
    }

    /**
     * 根据租户ID和税号ID查询是否已经申请
     *
     * @param tenantId 根据租户ID和税号ID查询是否已经申请
     * @param taxNum   税号
     */
    public Optional<CompanyApply> findByTenantIdAndTaxNum(long tenantId, String taxNum) {
        List<CompanyApply> companyApplyList = this.companyApplyDao.findByTenantIdAndTaxNum(tenantId, taxNum);
        if (CollectionUtils.isEmpty(companyApplyList)) {
            return Optional.empty();
        }
        return Optional.of(companyApplyList.get(0));
    }
}
