/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyExcelExportServiceImpl.java   2020-10-15 14-28-36
 * Author: Evan
 */
package com.xforceplus.business.company.service;

import com.xforceplus.api.model.CompanyModel.Request.Query;
import com.xforceplus.business.company.dto.TenantCompany;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.company.CompanyExcelDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.entity.Company;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.xforceplus.business.company.service.CompanyConfigExcel.SORT_FIELD;
import static com.xforceplus.business.company.service.CompanyConfigExcel.TAX_NUM;
import static com.xforceplus.business.company.service.CompanyExcel.SN_COMPANY;
import static com.xforceplus.business.excel.ExcelFile.*;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * <p>
 * Title:  CompanyExcelExportServiceImpl
 * </p>
 * <p>
 * Description: CompanyExcelExportServiceImpl
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-15 14-28-36
 * @since V1.0
 */
@Service
public class CompanyExcelExportServiceImpl implements ExcelWriteService {

    private final CompanyService companyService;

    public CompanyExcelExportServiceImpl(CompanyService companyService) {
        this.companyService = companyService;
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.COMPANY_EXCEL_FILE;
    }


    /**
     * 导入定义
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public void write(Context context) {

        Query query = context.getParam(PARAMS_QUERY, Query.class);
        Sort sort = context.getParam(PARAMS_SORT, Sort.class);
        List<String> sheetNames = Arrays.asList(query.getSheets());
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        Integer maxPageSize = context.getMaxPageSize();
        for (String sheetName : sheetNames) {
            //等于公司服务配置信息，则分页读取公司信息
            if (SN_COMPANY.equals(sheetName)) {
                this.writeCompanySheet(query, sort, simpleExcelWriter, sheetName, maxPageSize);
            }
        }
    }

    /**
     * 写入公司信息
     *
     * @param query             查询
     * @param sort              排序
     * @param simpleExcelWriter Excel写入方法
     * @param sheetName         SheetName
     * @param maxPageSize       限制导出最大页数
     */
    private void writeCompanySheet(Query query, Sort sort, SimpleExcelWriter simpleExcelWriter, String sheetName, Integer maxPageSize) {
        Page<Company> page;
        //如果指定page 不是从0开始，则在计算导出成功数时，需要指定起始页，即this.calcSuccessSize(page,)
        Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
        Map<Long, TenantCompany> tenantCompanyMap = this.companyService.findTenantCompanyAll();
        do {
            page = this.companyService.page(query, pageable);
            try {
                //对象转换
                Page<CompanyExcelDTO> companyExcelPage = page.map(companyExcel -> {
                    CompanyExcelDTO dto = new CompanyExcelDTO();
                    //处理
                    BeanUtils.copyProperties(companyExcel, dto, new String[]{TAX_NUM});
                    dto.setTaxNum(StringUtils.trimToEmpty(companyExcel.getTaxNum()));
                    if (tenantCompanyMap.containsKey(companyExcel.getCompanyId())) {
                        TenantCompany tenantCompany = tenantCompanyMap.get(companyExcel.getCompanyId());
                        dto.setTenantName(tenantCompany.getTenantName());
                        dto.setTenantCode(tenantCompany.getTenantCode());
                    }
                    return dto;
                });
                simpleExcelWriter.fill(sheetName, companyExcelPage.getContent());
            } catch (Exception e) {
                simpleExcelWriter.setException(e);
                break;
            }
            pageable = pageable.next();
        } while (page.hasNext() && page.getNumber() < maxPageSize);
        //计算当前导出数据  this.calcSuccessSize(page,)
        simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
    }
}
