package com.xforceplus.business.company.service;

import com.xforceplus.api.model.CompanyExtensionModel.Request.Query;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.constants.ExtensionStatus;
import com.xforceplus.dao.CompanyDao;
import com.xforceplus.dao.CompanyExtensionDao;
import com.xforceplus.domain.company.CompanyExtensionDto;
import com.xforceplus.entity.CompanyExtension;
import io.geewit.core.utils.reflection.BeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;

import javax.annotation.Resource;
import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.stream.Collectors;

@Validated
@Service
@Slf4j
public class CompanyExtensionService {
    private final static Logger logger = LoggerFactory.getLogger(CompanyExtensionService.class);
    private static final String AUTH_SWITCH_KEY = "terminal_";
    private static final Integer AUTH_SWITCH_ENABLED = 1;
    @Resource
    private CompanyExtensionDao companyExtensionDao;
    @Resource
    private CompanyDao companyDao;


    private Specification<CompanyExtension> querySpecification(Query query, Integer status) {
        return (Specification<CompanyExtension>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getCompanyId() != null && query.getCompanyId() > 0) {
                predicates.add(builder.equal(root.get("companyId"), query.getCompanyId()));
            }
            if (StringUtils.isNotBlank(query.getExtensionKey())) {
                predicates.add(builder.equal(root.get("extensionKey"), query.getExtensionKey()));
            }
            if (StringUtils.isNotBlank(query.getExtensionValue())) {
                predicates.add(builder.equal(root.get("extensionValue"), query.getExtensionValue()));
            }
            if (status != null && status > 0) {
                predicates.add(builder.equal(root.get("status"), status));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
    }

    public Page<CompanyExtension> page(Query query, Pageable pageable) {
        Specification<CompanyExtension> specification = querySpecification(query, ExtensionStatus.ENABLE);
        return companyExtensionDao.findAll(specification, pageable);
    }

    public List<CompanyExtension> list(Query query, Sort sort) {
        Specification<CompanyExtension> specification = querySpecification(query, ExtensionStatus.ENABLE);
        return companyExtensionDao.findAll(specification, sort);
    }

    @Transactional(rollbackFor = Exception.class)
    public CompanyExtension save(CompanyExtensionDto companyExtensionDto) {
        List<CompanyExtension> list = companyExtensionDao.findByCompanyIdAndKey(companyExtensionDto.getCompanyId(), companyExtensionDto.getExtensionKey());
        if (CollectionUtils.isEmpty(list)) {
            CompanyExtension entity = new CompanyExtension();
            BeanUtils.copyProperties(companyExtensionDto, entity);
            entity.setStatus(ExtensionStatus.ENABLE);
            entity.setCreateTime(new Date());
            entity.setUpdateTime(new Date());
            return companyExtensionDao.saveAndFlush(entity);
        } else {
            CompanyExtension entity = list.get(0);
            entity.setExtensionValue(companyExtensionDto.getExtensionValue());
            entity.setStatus(ExtensionStatus.ENABLE);
            entity.setUpdateTime(new Date());
            return companyExtensionDao.saveAndFlush(entity);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Set<CompanyExtension> batchSave(long companyId, List<CompanyModel.Request.Extension> extensions, boolean isOverwrite) {
        if (companyId <= 0 || CollectionUtils.isEmpty(extensions)) {
            return null;
        }
        List<CompanyExtension> existExtensions = this.findByComapnyId(companyId);
        Set<CompanyExtension> savingExtensions = extensions.stream().filter(Objects::nonNull).map(extension -> {
            List<CompanyExtension> exists = companyExtensionDao.findByCompanyIdAndKey(companyId, extension.getExtensionKey());
            CompanyExtension savingExtension;
            if (CollectionUtils.isEmpty(exists)) {
                savingExtension = new CompanyExtension();
                savingExtension.setCompanyId(companyId);
                savingExtension.setExtensionKey(extension.getExtensionKey());
            } else {
                savingExtension = exists.get(0);
            }
            savingExtension.setExtensionValue(extension.getExtensionValue());
            return savingExtension;
        }).collect(Collectors.toSet());
        if (!savingExtensions.isEmpty()) {
            savingExtensions.forEach(companyExtensionDao::saveAndFlush);
        }
        Set<CompanyExtension> result = new HashSet<>(savingExtensions);
        result.addAll(existExtensions);
        if (isOverwrite) {
            existExtensions.stream().filter(existExtension -> extensions.stream().noneMatch(extension -> extension.getExtensionKey().equals(existExtension.getExtensionKey()))).forEach(extension -> {
                logger.info("deleting CompanyExtension record, {}", extension);
                try {
                    companyExtensionDao.deleteById(extension.getCompanyExtensionId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
        } else {
            result.addAll(existExtensions);
        }
        return result;
    }

    public List<CompanyExtension> findByComapnyId(long companyId) {
        Query query = new Query();
        query.setCompanyId(companyId);
        Specification<CompanyExtension> specification = this.querySpecification(query, null);
        return companyExtensionDao.findAll(specification, Sort.unsorted());
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByCompanyId(long companyId) {
        companyExtensionDao.deleteByCompanyId(companyId);
    }

    @Transactional(rollbackFor = Exception.class)
    public Boolean setTerminalSwitch(Long tenantId, Long companyId, Integer terminalSwitch) {
        List<CompanyExtension> extensionList = this.findByComapnyId(companyId);
        String key = AUTH_SWITCH_KEY + tenantId;
        CompanyExtension companyExtension = new CompanyExtension();
        //公司没有扩展信息，直接插入即可
        if (CollectionUtils.isEmpty(extensionList)) {
            companyExtension = this.buildCompanyExtention(companyId, tenantId, key, String.valueOf(terminalSwitch));
        } else {
            //更新现有扩展信息
            Optional<CompanyExtension> companyExtensionOptional = extensionList.stream().filter(item -> item.getExtensionKey().equals(key) && item.getStatus().equals(1)).findAny();
            if (companyExtensionOptional.isPresent()) {
                //原来有开关设置，直接更新
                companyExtension = companyExtensionOptional.get();
                companyExtension.setExtensionValue(String.valueOf(terminalSwitch));
                companyExtension.setStatus(1);
            } else {
                companyExtension = this.buildCompanyExtention(companyId, tenantId, key, String.valueOf(terminalSwitch));
            }
        }
        log.info("companyId {} tenantId {} 端鉴权开关变更为{}", companyId, tenantId, terminalSwitch);
        companyExtensionDao.saveAndFlush(companyExtension);
        return true;
    }

    @Transactional(rollbackFor = Exception.class)
    public Integer getCompanyTerminalSwitch(Long tenantId, Long companyId) {
        String key = AUTH_SWITCH_KEY + tenantId;
        List<CompanyExtension> extensionList = this.findByComapnyId(companyId);
        if (CollectionUtils.isEmpty(extensionList)) {
            //没有扩展字段，系统自动插入并设置为启用状态
            this.addDefaultTerminalSwitch(companyId, tenantId, key, String.valueOf(AUTH_SWITCH_ENABLED));
            log.info("companyId {} tenantId {} 没有公司扩展信息，系统默认添加终端鉴权开关并设置为开", companyId, tenantId);
            return AUTH_SWITCH_ENABLED;
        }
        Optional<CompanyExtension> companyExtensionOptional = extensionList.stream().filter(item -> item.getExtensionKey().equals(key) && item.getStatus().equals(1)).findAny();
        if (companyExtensionOptional.isPresent()) {
            return Integer.parseInt(companyExtensionOptional.get().getExtensionValue());
        } else {
            log.info("companyId {} tenantId {} 没有设置终端鉴权开关，系统默认添加开关并设置为开", companyId, tenantId);
            this.addDefaultTerminalSwitch(companyId, tenantId, key, String.valueOf(AUTH_SWITCH_ENABLED));
            return AUTH_SWITCH_ENABLED;
        }
    }

    private CompanyExtension buildCompanyExtention(Long companyId, Long tenantId, String key, String value) {
        CompanyExtension companyExtension = new CompanyExtension();
        companyExtension.setCompanyId(companyId);
        companyExtension.setExtensionKey(key);
        companyExtension.setExtensionValue(value);
        companyExtension.setStatus(ExtensionStatus.ENABLE);
        return companyExtension;
    }

    private void addDefaultTerminalSwitch(Long companyId, Long tenantId, String key, String value) {
        CompanyExtension companyExtension = this.buildCompanyExtention(companyId, tenantId, key, value);
        companyExtensionDao.saveAndFlush(companyExtension);
    }

    /**
     * 根据税号来设置终端鉴权
     *
     * @param taxNum       税号
     * @param tenantId     租户ID
     * @param terminalAuth 终端鉴权
     * @return
     */
    public Boolean setTerminalAuthByTaxNum(String taxNum, Long tenantId, Integer terminalAuth) {
        Optional<Long> optionalLong = companyDao.findIdByTaxNum(taxNum);
        if (optionalLong.isPresent()) {
            Long companyId = optionalLong.get();
            List<CompanyExtension> extensionList = this.findByComapnyId(companyId);
            String key = AUTH_SWITCH_KEY + tenantId;
            Optional<CompanyExtension> companyExtensionOptional = extensionList.stream().filter(item -> item.getExtensionKey().equals(key) && item.getStatus().equals(1)).findFirst();
            CompanyExtension companyExtension = new CompanyExtension();
            if (companyExtensionOptional.isPresent()) {
                companyExtension = companyExtensionOptional.get();
                companyExtension.setExtensionValue(String.valueOf(terminalAuth));
            } else {
                companyExtension = this.buildCompanyExtention(companyId, tenantId, key, String.valueOf(terminalAuth));
            }
            companyExtensionDao.saveAndFlush(companyExtension);
            return true;
        }
        logger.warn("税号{}对应的公司不存在，无法设置终端鉴权", taxNum);
        return false;
    }
}
