package com.xforceplus.business.company.service;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.CompanyModel.Request;
import com.xforceplus.api.model.CompanyModel.Response;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.api.model.TenantModel;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.company.dto.CompanyTaxwareDto;
import com.xforceplus.business.company.dto.LimitInfo;
import com.xforceplus.business.company.dto.TenantCompany;
import com.xforceplus.business.file.service.FileService;
import com.xforceplus.business.log.service.LogService;
import com.xforceplus.business.messagebus.CompanyPubService;
import com.xforceplus.business.messagebus.OrgPubService;
import com.xforceplus.constants.BusinessTypeEnum;
import com.xforceplus.dao.*;
import com.xforceplus.domain.company.*;
import com.xforceplus.domain.log.SystemLogDTO;
import com.xforceplus.dto.company.CompanyServicePackageDTO;
import com.xforceplus.dto.org.CompanyTenantDto;
import com.xforceplus.entity.*;
import com.xforceplus.enums.ActionCodeEnum;
import com.xforceplus.query.CompanyQueryHelper;
import com.xforceplus.query.OrgQueryHelper;
import com.xforceplus.query.TenantQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.utils.RandomUtils;
import com.xforceplus.utils.excel.ExcelUtils;
import com.xforceplus.utils.excel.exception.ImportException;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.envers.domain.ComparedRevision;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import io.geewit.utils.uuid.UUID;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.*;
import org.springframework.data.history.RevisionSort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.persistence.Tuple;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.validation.Validator;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.api.utils.Separator.COMMA;
import static java.util.stream.Collectors.*;

@Service
public class CompanyService {
    private final static Logger logger = LoggerFactory.getLogger(CompanyService.class);
    private final CompanyDao companyDao;
    private final CompanyServiceRelDao companyServiceRelDao;
    private final TenantCompanyRelDao tenantCompanyRelDao;
    private final TenantDao tenantDao;
    private final ServicePackageDao servicePackageDao;
    private final OrgCompanynoDao orgCompanynoDao;
    private final Validator validator;
    private final FileService fileService;
    private final OrgStructDao orgStructDao;
    private final UserDao userDao;
    private final RoleDao roleDao;
    private final RoleUserRelDao roleUserRelDao;
    private final OrgUserRelDao orgUserRelDao;

    private final int MAX_LENT = 64;

    private final CompanyExtensionService companyExtensionService;

    private final CompanyPubService companyPubService;

    private final OrgPubService orgPubService;

    private final CompanyTenantRelDao companyTenantRelDao;

    private final CompanyTenantRelDao relDao;

    private final CompanyTenantRelAuditDao relAuditDao;

    private final LogService logService;

    private final JdbcTemplate jdbcTemplate;


    private static final String ADD_TENANT_COMPANY_REL = "bind";
    private static final String DELETE_TENANT_COMPANY_REL = "unbind";

    /**
     * 一般纳税人
     */
    private static final String NORMAL_TAX_PAYER = "NormalTaxPayer";

    /**
     * 小规模纳税人
     */
    private static final String SMALL_TAX_PAYER = "SmallTaxPayer";


    public CompanyService(CompanyDao companyDao, CompanyServiceRelDao companyServiceRelDao,
                          TenantCompanyRelDao tenantCompanyRelDao,
                          TenantDao tenantDao, ServicePackageDao servicePackageDao,
                          OrgCompanynoDao orgCompanynoDao, Validator validator,
                          FileService fileService, OrgStructDao orgStructDao, UserDao userDao,
                          RoleDao roleDao, RoleUserRelDao roleUserRelDao,
                          OrgUserRelDao orgUserRelDao, CompanyExtensionService companyExtensionService,
                          CompanyPubService companyPubService, OrgPubService orgPubService,
                          CompanyTenantRelDao companyTenantRelDao, CompanyTenantRelDao relDao,
                          CompanyTenantRelAuditDao relAuditDao, LogService logService,
                          JdbcTemplate jdbcTemplate) {
        this.companyDao = companyDao;
        this.companyServiceRelDao = companyServiceRelDao;
        this.tenantCompanyRelDao = tenantCompanyRelDao;
        this.tenantDao = tenantDao;
        this.servicePackageDao = servicePackageDao;
        this.orgCompanynoDao = orgCompanynoDao;
        this.validator = validator;
        this.fileService = fileService;
        this.orgStructDao = orgStructDao;
        this.userDao = userDao;
        this.roleDao = roleDao;
        this.roleUserRelDao = roleUserRelDao;
        this.orgUserRelDao = orgUserRelDao;
        this.companyExtensionService = companyExtensionService;
        this.companyPubService = companyPubService;
        this.orgPubService = orgPubService;
        this.companyTenantRelDao = companyTenantRelDao;

        this.relDao = relDao;
        this.relAuditDao = relAuditDao;
        this.logService = logService;
        this.jdbcTemplate = jdbcTemplate;
    }

    public Page<Company> page(Request.Query query, Pageable pageable) {
        Page<Company> page;
        if (query.getMultipleTenants() != null) {
            Page<Tuple> tuples = companyDao.findTuples(query, pageable);
            List<Company> contents = tuples.getContent().stream().map(tuple -> {
                Company company = tuple.get("company", Company.class);
                return company;
            }).collect(Collectors.toList());
            page = new PageImpl<>(contents, pageable, tuples.getTotalElements());
        } else {
            Specification<Company> specification = CompanyQueryHelper.querySpecification(query);
            page = companyDao.findAll(specification, pageable, EntityGraphs.named(Company.NAMED_ENTITY_GRAPH_DEFAULT));
        }

        boolean enableTenant = query.getTenantId() != null && query.getTenantId() > 0;

        if (StringUtils.isNotBlank(query.getWithExtendParams()) || enableTenant) {
            String[] withExtendParamArray = StringUtils.split(query.getWithExtendParams(), COMMA);
            boolean enableExtensions = false;
            boolean enableRelations = false;
            boolean enableHostOrgs = false;
            boolean enableTenants = false;
            if (ArrayUtils.isNotEmpty(withExtendParamArray)) {
                for (String withExtendParam : withExtendParamArray) {
                    withExtendParam = withExtendParam.trim();
                    if (!enableExtensions && "extensions".equalsIgnoreCase(withExtendParam)) {
                        enableExtensions = true;
                    } else if (!enableRelations && "relations".equalsIgnoreCase(withExtendParam)) {
                        enableRelations = true;
                    } else if (!enableHostOrgs && "hostOrgs".equalsIgnoreCase(withExtendParam)) {
                        enableHostOrgs = true;
                    } else if (!enableTenants && "tenants".equalsIgnoreCase(withExtendParam)) {
                        enableTenants = true;
                    }
                }
            }
            if (enableExtensions || enableRelations || enableHostOrgs || enableTenants || enableTenant) {
                Tenant tenant = null;
                if (enableTenant) {
                    tenant = tenantDao.findById(query.getTenantId()).orElse(null);
                }
                for (Company company : page) {
                    if (enableExtensions) {
                        this.fillCompanyExtentions(company);
                    }
                    if (enableRelations) {
                        this.fillCompanyRelations(company);
                    }
                    if (enableHostOrgs) {
                        this.fillHostOrgs(company);
                    }
                    if (enableTenants) {
                        this.fillTenants(company);
                    }
                    if (enableTenant) {
                        this.fillTenant(company, tenant);
                    }
                }
            }
        }
        return page;
    }

    public Page<Company> page(Specification<Company> specification, Pageable pageable) {
        return companyDao.findAll(specification, pageable);
    }

    public List<Company> list(Request.Query query, Sort sort) {
        Specification<Company> specification = CompanyQueryHelper.querySpecification(query);
        List<Company> list = companyDao.findAll(specification, sort);
        return list;
    }

    public List<Company> findByTaxNum(String taxNum) {
        Request.Query query = new Request.Query();
        query.setTaxNum(taxNum);
        query.setStatus(1);
        Specification<Company> specification = CompanyQueryHelper.querySpecification(query);
        List<Company> companies = companyDao.findAll(specification);
        if (!companies.isEmpty()) {
            Company company = companies.get(0);
            this.fillTenants(company);
            return Stream.of(company).collect(Collectors.toList());
        } else {
            return new ArrayList<>();
        }
    }

    private void fillHostOrgs(CompanyDto company) {
        if (company.getHostTenantId() != null && company.getHostTenantId() > 0 && company.getCompanyId() != null && company.getCompanyId() > 0) {
            OrgModel.Request.Query query = new OrgModel.Request.Query();
            query.setCompanyId(company.getCompanyId());
            query.setTenantId(company.getHostTenantId());
            query.setStatus(1);
            List<OrgStruct> hostOrgEntities = orgStructDao.findAll(OrgQueryHelper.querySpecification(query));
            List<OrgStruct> hostOrgs = new ArrayList<>(hostOrgEntities.size());
            for (OrgStruct hostOrgEntity : hostOrgEntities) {
                OrgStruct hostOrg = new OrgStruct();
                BeanUtils.copyProperties(hostOrgEntity, hostOrg, Stream.of("company", "tenant", "packages", "orgUserRels", "orgCompanyRels").toArray(String[]::new));
                hostOrgs.add(hostOrg);
            }
            company.setHostOrgs(hostOrgs);
        }
    }

    private void fillCompany(Company company, Set<String> withExtendParams) {
        if (company == null) {
            return;
        }
        if (withExtendParams == null || withExtendParams.isEmpty()) {
            return;
        }
        boolean enableExtensions = false;
        boolean enableRelations = false;
        boolean enableHostOrgs = false;
        boolean enableTenants = false;
        for (String withExtendParam : withExtendParams) {
            withExtendParam = withExtendParam.trim();
            if (!enableExtensions && "extensions".equalsIgnoreCase(withExtendParam)) {
                enableExtensions = true;
            } else if (!enableRelations && "relations".equalsIgnoreCase(withExtendParam)) {
                enableRelations = true;
            } else if (!enableHostOrgs && "hostOrgs".equalsIgnoreCase(withExtendParam)) {
                enableHostOrgs = true;
            } else if (!enableTenants && "tenants".equalsIgnoreCase(withExtendParam)) {
                enableTenants = true;
            }
        }
        if (enableExtensions) {
            this.fillCompanyExtentions(company);
        }
        if (enableRelations) {
            this.fillCompanyRelations(company);
        }
        if (enableHostOrgs) {
            this.fillHostOrgs(company);
        }
        if (enableTenants) {
            this.fillTenants(company);
        }
    }

    private void fillTenants(Company company) {
        if (company == null) {
            return;
        }
        TenantModel.Request.Query tenantQuery = new TenantModel.Request.Query();
        tenantQuery.setCompanyId(company.getCompanyId());
        tenantQuery.setStatus(1);
        Specification<Tenant> tenantSpecification = TenantQueryHelper.querySpecification(tenantQuery);
        List<Tenant> tenants = tenantDao.findAll(tenantSpecification);
        company.setTenants(tenants.stream().filter(Objects::nonNull).collect(toList()));
    }

    private void fillCompanyExtentions(CompanyDto company) {
        if (company == null) {
            return;
        }
        List<CompanyExtension> extensions = companyExtensionService.findByComapnyId(company.getCompanyId());
        if (!CollectionUtils.isEmpty(extensions)) {
            List<CompanyExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(toList());
            company.setExtensions(extensionDtos);
        }
    }

    private void fillCompanyRelations(Company company) {
        if (company == null || company.getCompanyId() == null || company.getCompanyId() == 0) {
            return;
        }
        OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
        orgQuery.setCompanyId(company.getCompanyId());
        orgQuery.setStatus(1);
        Specification<OrgStruct> orgSpecification = OrgQueryHelper.querySpecification(orgQuery);
        List<OrgStruct> orgs = orgStructDao.findAll(orgSpecification);
        if (orgs == null || orgs.isEmpty()) {
            return;
        }
        Long[] tenantIdArray = orgs.stream().map(OrgStruct::getTenantId).toArray(Long[]::new);
        if (ArrayUtils.isEmpty(tenantIdArray)) {
            return;
        }
        TenantModel.Request.Query tenantQuery = new TenantModel.Request.Query();
        tenantQuery.setTenantIds(tenantIdArray);
        tenantQuery.setStatus(1);
        Specification<Tenant> tenantSpecification = TenantQueryHelper.querySpecification(tenantQuery);
        List<Tenant> tenants = tenantDao.findAll(tenantSpecification);
        Set<Long> tenantIds = new HashSet<>();
        Set<Long> relatedTenantIds = new HashSet<>();
        Set<Relation<OrgStruct>> relations = new HashSet<>();
        List<CompanyTenantRel> rels;
        if (company.getTenantId() != null && company.getTenantId() > 0) {
            rels = companyTenantRelDao.findCompanyIdAndRelatedTenantId(company.getCompanyId(), company.getTenantId());
        } else {
            rels = companyTenantRelDao.findByCompanyId(company.getCompanyId());
        }
        for (CompanyTenantRel rel : rels) {
            tenantIds.add(rel.getTenantId());
            relatedTenantIds.add(rel.getRelatedTenantId());
            Relation<OrgStruct> relation = new Relation<>();
            relation.setTenantId(rel.getRelatedTenantId());
            Tenant tenant = tenants.stream().filter(t -> t.getTenantId().equals(relation.getTenantId())).findAny().orElse(null);
            if (tenant == null) {
                continue;
            }
            relation.setTenantName(tenant.getTenantName());
            relation.setTenantCode(tenant.getTenantCode());
            Set<OrgStruct> relatedOrgs = orgs.stream().filter(o -> o.getTenantId().equals(relation.getTenantId())).collect(toSet());
            relation.setOrgs(relatedOrgs);
            relations.add(relation);
        }
        company.setRelations(relations);
    }

    private void fillTenant(Company company, Tenant tenant) {
        company.setTenantId(tenant.getTenantId());
        company.setTenantName(tenant.getTenantName());
        company.setTenantCode(tenant.getTenantCode());
    }

    public List<Company> list(Specification<Company> specification, Sort sort) {
        return companyDao.findAll(specification, sort);
    }

    public Optional<Company> findOne(Request.Query query) {
        Specification<Company> specification = CompanyQueryHelper.queryOneSpecification(query);
        return companyDao.findOne(specification);
    }

    /**
     * 查询并返回 Company  (taxNum是唯一索引)
     *
     * @param taxNum 税号
     * @return Optional<Company>
     */
    public Optional<Company> findOneByTaxNum(String taxNum) {
        if (StringUtils.isBlank(taxNum)) {
            throw new IllegalArgumentException("公司税号不能为空");
        }
        List<Company> list = this.companyDao.findByTaxNum(taxNum);
        if (CollectionUtils.isEmpty(list)) {
            return Optional.empty();
        }
        return Optional.of(list.get(0));
    }

    /**
     * 根据税号查询公司列表信息
     *
     * @param taxNum 税号
     * @return List<Company>
     */
    public List<Company> findListByTaxNum(String taxNum) {
        return this.companyDao.findByTaxNum(taxNum);
    }

    public List<Company> findList(Request.Query query) {
        Specification<Company> specification = CompanyQueryHelper.queryOneSpecification(query);
        return companyDao.findAll(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public Company create(Request.Save model) {
        this.checkExtensions(model.getExtensions());
        Company company = this.save(model, false);
        return company;
    }

    @Transactional(rollbackFor = Exception.class)
    public <C extends Request.Save> Company update(long companyId, C model) {
        this.checkExtensions(model.getExtensions());
        model.setCompanyId(companyId);
        boolean isOverwrite = true;
        Company company = this.save(model, isOverwrite);
        Set<CompanyExtension> extensions = companyExtensionService.batchSave(company.getCompanyId(), model.getExtensions(), isOverwrite);
        if (!CollectionUtils.isEmpty(extensions)) {
            company.setExtensions(new ArrayList<>(extensions));
        }
        return company;
    }

    @Transactional(rollbackFor = Exception.class)
    public <C extends Request.Save> Company save(C model, boolean isOverwrite) {

        this.checkExtensions(model.getExtensions());

        Optional<Company> optionalCompany = Optional.empty();
        //如果CompanyId不能等于null
        if (model.getCompanyId() != null && model.getCompanyId() > 0) {
            optionalCompany = companyDao.findById(model.getCompanyId());
        }
        //判断税号是否为空
        if (!optionalCompany.isPresent() && StringUtils.isNotBlank(model.getTaxNum())) {
            optionalCompany = this.findCompanyByTaxNum(model.getTaxNum());
        }
        //根据companyId查询数据
        Company companyEntity = null;
        if (optionalCompany.isPresent()) {
            if (isOverwrite) {
                if (StringUtils.isBlank(model.getCompanyName())) {
                    model.setCompanyName(null);
                }
                if (StringUtils.isBlank(model.getCompanyCode())) {
                    model.setCompanyCode(null);
                }
                if (StringUtils.isBlank(model.getTaxNum())) {
                    model.setTaxNum(null);
                }
            }
            companyEntity = optionalCompany.get();
        } else {
            logger.info("未根据参数, companyId = {}, companyCode = {}, taxNum = {}, companyName = {}, 查到实体", model.getCompanyId(), model.getCompanyCode(), model.getTaxNum(), model.getCompanyName());
            companyEntity = new Company();
        }

        BeanUtils.copyProperties(model, companyEntity,
                Stream.of("companyId", "orgs", "tenants", "companyNos", "hostTenantId").toArray(String[]::new),
                Stream.of("cquota", "squota", "ceQuota", "juQuota", "seQuota", "vehicleLimit", "hostTenantId").toArray(String[]::new));
        if (companyEntity.getHostTenantId() == null && model.getHostTenantId() != null) {
            companyEntity.setHostTenantId(model.getHostTenantId());
        }
        companyEntity = this.saveAndFlush(companyEntity);

        companyExtensionService.batchSave(companyEntity.getCompanyId(), model.getExtensions(), isOverwrite);
        return companyEntity;
    }


    /**
     * 新增情况校验是否已经存在,如果存在则回写错误信息 (只判断公司税号)
     *
     * @param dto 新增对象
     * @return Boolean true 已存在，false 则不存在
     */
    protected Optional<Company> findCompanyByTaxNum(String taxNum) {
        List<Company> companies = this.findListByTaxNum(taxNum);
        //如果不存在就直接返回
        if (CollectionUtils.isEmpty(companies)) {
            return Optional.empty();
        }
        //如果超过两个则报错
        if (companies.size() > 1) {
            String message = "已存在2个公司税号(" + taxNum + ")的公司";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        return Optional.of(companies.get(0));
    }

    public Company findByTenantIdAndId(long tenantId, long companyId) {
        Request.Query query = new Request.Query();
        query.setTenantId(tenantId);
        query.setCompanyId(companyId);
        return this.findOne(query).orElseThrow(() -> new IllegalArgumentException("租户(" + tenantId + ")下, 未找到公司(" + companyId + ") !"));
    }

    public Company findById(long companyId) {
        Company company = companyDao.findById(companyId).orElseThrow(() -> new IllegalArgumentException("未找到公司实体(" + companyId + ")"));
        return company;
    }

    public Company findByIdAndStatusAndRevisionDate(long companyId, Integer status, Date revisionDate, Set<String> withExtendParams) {
        Company company = companyDao.findById(companyId).orElseThrow(() -> new IllegalArgumentException("未找到公司实体(" + companyId + ")"));
        if (revisionDate != null) {
            try {
                Company historyCompany = companyDao.findRevisionByLastUpdateTime(companyId, revisionDate);
                if (historyCompany != null) {
                    BeanUtils.copyProperties(historyCompany, company, Stream.of("orgs", "companyNos", "hostTenant", "tenantRels").toArray(String[]::new));
                }
            } catch (Exception e) {
                logger.warn(e.getMessage());
            }
        }
        this.fillCompany(company, withExtendParams);
        return company;
    }

    public Page<Company> page(long tenantId, Request.Query query, Pageable pageable) {
        query.setTenantId(tenantId);
        return this.page(query, pageable);
    }

    public List<Company> list(long tenantId, Request.Query query, Sort sort) {
        query.setTenantId(tenantId);
        return this.list(query, sort);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long companyId) {
        Company company = this.findById(companyId);
        companyDao.deleteById(company.getCompanyId());
        companyExtensionService.deleteByCompanyId(company.getCompanyId());
    }

    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    public Page<CompanyPackage> packagesByQuery(Request.CompanyPackageQuery query, Pageable pageable) {
        Page<CompanyPackage> page = companyDao.findPackages(query, pageable);
        Set<Long> companyIds = page.getContent()
                .stream()
                .map(CompanyPackage::getCompanyId)
                .collect(toSet());
        List<CompanyServicePackageDTO> companyServicePackageDTOs;
        if (!CollectionUtils.isEmpty(companyIds)) {
            companyServicePackageDTOs = companyDao.findCompanyServicePackageByCompanyIds(companyIds);
        } else {
            companyServicePackageDTOs = Collections.emptyList();
        }
        //以CompanyId为判断判断
        Map<Long, Set<String>> companyPackageNameMap = companyServicePackageDTOs
                .stream()
                .collect(groupingBy(CompanyServicePackageDTO::getCompanyId,
                        mapping(CompanyServicePackageDTO::getServicePackageName, toSet())));
        for (CompanyPackage companyPackage : page) {
            //如果没有数据则返回为空
            if (!companyPackageNameMap.containsKey(companyPackage.getCompanyId())) {
                continue;
            }
            Set<String> packageNames = companyPackageNameMap.get(companyPackage.getCompanyId());
            companyPackage.setPackageNames(new ArrayList<>(packageNames));
        }
        return page;
    }

    /**
     * 校验公司是否正确
     *
     * @param companyId   公司id
     * @param companyCode 公司代码
     * @param taxNum      公司税号
     */
    private void validateCompany(long companyId, String companyCode, String taxNum) {
        long count = companyDao.validateExistCompany(companyId, companyCode, taxNum);
        if (count > 0) {
            throw new IllegalArgumentException("已经有重复的税号或公司名称或代码");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Company update(long tenantId, long companyId, Request.Save model) {
        this.checkExtensions(model.getExtensions());
        Request.Query query = new Request.Query();
        query.setTenantId(tenantId);
        query.setCompanyId(companyId);
        Optional<Company> companyOptional = this.findOne(query);
        if (!companyOptional.isPresent()) {
            throw new IllegalArgumentException("未找到公司实体tenantId:" + tenantId + ",companyId:" + companyId);
        }
        this.validateCompany(companyId, model.getCompanyCode(), model.getTaxNum());
        Company existModel = companyOptional.get();
        if (StringUtils.isBlank(model.getCompanyName())) {
            model.setCompanyName(null);
        }
        if (StringUtils.isBlank(model.getCompanyCode())) {
            model.setCompanyCode(null);
        }
        if (StringUtils.isBlank(model.getTaxNum())) {
            model.setTaxNum(null);
        }

        BeanUtils.copyProperties(model, existModel);
        existModel = this.saveAndFlush(existModel);
        companyExtensionService.batchSave(companyId, model.getExtensions(), true);
        return existModel;
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long companyId, int status) {
        Company company = this.findById(companyId);
        company.setStatus(status);
        this.saveAndFlush(company);
    }

    /**
     * 公司绑定服务包
     *
     * @param tenantId
     * @param companyId
     * @param bindPackage
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindPackages(long tenantId, long companyId, Request.BindPackages bindPackage) {
        if (bindPackage == null) {
            String message = "请求参数错误, bindPackage == null";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        List<Long> packageIds = bindPackage.getPackageIds();
        if (packageIds == null) {
            String message = "请求参数错误, packageIds == null";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        Request.Query query = new Request.Query();
        query.setTenantId(tenantId);
        query.setCompanyId(companyId);
        Company existEntity = this.findOne(query).orElseThrow(() -> new IllegalArgumentException("未找到公司(tenantId: " + tenantId + ",companyId:" + companyId + ")实体"));
        Specification<CompanyServiceRel> specification = (Specification<CompanyServiceRel>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.<Long>get("tenantId"), tenantId));
            predicates.add(builder.equal(root.<Long>get("companyId"), existEntity.getCompanyId()));
            if (!predicates.isEmpty()) {
                criteriaQuery.where(builder.and(predicates.toArray(new Predicate[0])));
            }
            return criteriaQuery.getRestriction();
        };
        List<CompanyServiceRel> existRels = companyServiceRelDao.findAll(specification, Sort.unsorted());

        Set<String> errors = new HashSet<>();
        Set<CompanyServiceRel> insertingRels = packageIds.stream().filter(Objects::nonNull).filter(packageId -> existRels.stream()
                .map(CompanyServiceRel::getServicePackageId)
                .noneMatch(relPackageId -> relPackageId.equals(packageId)))
                .map(packageId -> {
            Optional<ServicePackage> servicePackageOptional = servicePackageDao.findById(packageId);
            if (servicePackageOptional.isPresent()) {
                CompanyServiceRel rel = new CompanyServiceRel();
                rel.setTenantId(tenantId);
                rel.setCompanyId(companyId);
                rel.setServicePackageId(packageId);
                rel.setStatus(1);
                rel.setOperateReason(StringUtils.EMPTY);
                rel.setRemarks(StringUtils.EMPTY);
                return rel;
            } else {
                errors.add("不存在服务包(" + packageId + ")");
                return null;
            }
        }).filter(Objects::nonNull).collect(toSet());
        if (!errors.isEmpty()) {
            throw new IllegalArgumentException(errors.stream().collect(joining(",")));
        }
        if (!insertingRels.isEmpty()) {
            insertingRels.forEach(companyServiceRelDao::saveAndFlush);
        }

        if (bindPackage.isOverwrite()) {
            existRels.stream().filter(Objects::nonNull).filter(existRel -> packageIds.stream().noneMatch(packageId -> packageId != null && packageId.equals(existRel.getServicePackageId()))).forEach(existRel -> {
                logger.info("删除公司服务包, companyId = {}, servicePackageId = {}", existRel.getCompanyId(), existRel.getServicePackageId());
                try {
                    companyServiceRelDao.deleteById(existRel.getId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
        }

    }

    /**
     * TODO 废弃的方法
     *
     * @param tenantId
     * @param companyId
     */
    @Deprecated
    @Transactional(rollbackFor = Exception.class)
    public TenantCompanyRel saveTenantCompany(Tenant tenant, Company company) {
        //region TODO delete it!
        Optional<TenantCompanyRel> tenantCompanyRelOptional = tenantCompanyRelDao.findOne((Specification<TenantCompanyRel>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.<Long>get("tenantId"), tenant.getTenantId()));
            predicates.add(builder.equal(root.<Long>get("companyId"), company.getCompanyId()));
            criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            return criteriaQuery.getRestriction();
        });
        TenantCompanyRel tenantCompanyRel;
        if (tenantCompanyRelOptional.isPresent()) {
            tenantCompanyRel = tenantCompanyRelOptional.get();
        } else {
            tenantCompanyRel = new TenantCompanyRel();
            tenantCompanyRel.setTenantId(tenant.getTenantId());
            tenantCompanyRel.setCompanyId(company.getCompanyId());
            tenantCompanyRel = tenantCompanyRelDao.saveAndFlush(tenantCompanyRel);
            companyPubService.sendTenantCompanyRelMsg(ADD_TENANT_COMPANY_REL, company, tenant);
        }
        return tenantCompanyRel;
        //endregion
    }

    public ResponseEntity<Long> batchImportConfiguration(MultipartFile file) {

        String template = "companyConfiguration.json";
        List<CompanyConfigurationDto> list = ExcelUtils.list(file, template, CompanyConfigurationDto.class);
        int success = 0;
        int fail = 0;
        List<String> errorMsg = new ArrayList<>();
        for (int i = 0; i < list.size(); i++) {
            CompanyConfigurationDto companyConfigurationDomain = list.get(i);
            try {
                ExcelUtils.validField(companyConfigurationDomain, validator);
                this.saveConfigurationImport(companyConfigurationDomain);
                success++;
            } catch (ImportException e) {
                fail++;
                for (String msg : e.getMsgError()) {
                    String message = "导入第 " + (i + 1) + " 行失败," + msg;
                    logger.warn(message);
                    errorMsg.add(message);
                }
            }
        }
        ResponseEntity<Long> restResponse = new ResponseEntity<>();
        restResponse.setCode("1");
        if (!CollectionUtils.isEmpty(errorMsg)) {
            Long fileId = fileService.uploadMsgExcel(errorMsg);
            restResponse.setResult(fileId);
            restResponse.setCode(null == fileId ? "1" : "0");
        }
        //上传oss
        restResponse.setMessage("导入成功 " + success + " 条, 导入失败 " + fail + " 条");
        return restResponse;
    }

    @Transactional(rollbackFor = Exception.class)
    public void saveConfigurationImport(CompanyConfigurationDto companyConfigurationDto) throws ImportException {

        Company company = this.getByTaxNum(companyConfigurationDto.getCompanyTax());
        if (company == null) {
            throw new ImportException(Stream.of("税号【" + companyConfigurationDto.getCompanyTax() + "】对应公司不存在").collect(toList()));
        }

        int traditionAuthenFlag = Integer.parseInt(companyConfigurationDto.getTraditionAuthenFlag());
        int inspectionChannelFlag = Integer.parseInt(companyConfigurationDto.getInspectionServiceFlag());
        int speedInspectionChannelFlag = Integer.parseInt(companyConfigurationDto.getSpeedInspectionChannelFlag());

        if (inspectionChannelFlag == 0 && speedInspectionChannelFlag == 1) {
            throw new ImportException(Stream.of("未开启查验服务，无法开其极速查验通道").collect(toList()));
        }
        company.setTraditionAuthenFlag(traditionAuthenFlag);
        company.setInspectionServiceFlag(inspectionChannelFlag);
        company.setSpeedInspectionChannelFlag(speedInspectionChannelFlag);
        this.saveAndFlush(company);
    }

    public Company getByTaxNum(String taxNum) {
        Request.Query query = new Request.Query();
        query.setTaxNum(taxNum);
        Optional<Company> optionalCompany = this.findOne(query);
        return optionalCompany.orElse(null);
    }

    @Transactional(rollbackFor = Exception.class)
    public void saveCompanyNos(long orgId, Set<String> companyNos, boolean isOverwrite) {
        if (CollectionUtils.isEmpty(companyNos)) {
            if (companyNos != null && isOverwrite) {
                logger.info("companyNos is empty, isOverwrite = true, clean all and return");
                orgCompanynoDao.deleteByOrgId(orgId);
            } else {
                logger.info("companyNos == null, isOverwrite = false, return");
            }
            return;
        }
        List<OrgCompanyRel> existRels = orgCompanynoDao.findByOrgId(orgId);
        Set<OrgCompanyRel> insertingRels = companyNos.stream()
                .filter(companyNo -> existRels.stream()
                        .map(OrgCompanyRel::getCompanyNo)
                        .filter(StringUtils::isNotBlank)
                        .noneMatch(companyNo::equals))
                .map(companyNo -> {
                    OrgCompanyRel rel = new OrgCompanyRel();
                    rel.setOrgStructId(orgId);
                    rel.setCompanyNo(companyNo);
                    return rel;
                }).collect(toSet());
        if (!CollectionUtils.isEmpty(insertingRels)) {
            insertingRels.forEach(orgCompanynoDao::saveAndFlush);
        }

        if (isOverwrite) {
            //region 从数据库中解绑不在报文 roleIds 中的用户-角色关系
            existRels.stream().filter(rel -> companyNos.stream()
                            .filter(Objects::nonNull)
                            .noneMatch(companyNo -> companyNo.equals(rel.getCompanyNo())))
                    .forEach(rel -> {
                        try {
                            orgCompanynoDao.deleteById(rel.getId());
                        } catch (Exception e) {
                            logger.warn(e.getMessage(), e);
                        }
                    });
            //endregion
        }
    }

    @Async("threadPoolExecutor")
    public void fixTenantCompanies() {
        List<Map<String, Object>> fixingCompanies = companyDao.findFixingCompanies();
        if (fixingCompanies.isEmpty()) {
            return;
        }
        fixingCompanies.stream().filter(Objects::nonNull).forEach(org -> {
            logger.debug("org: {}", JsonUtils.toJson(org));
            TenantCompanyRel rel = new TenantCompanyRel();
            rel.setTenantId(((BigInteger) org.get("tenant_id")).longValue());
            rel.setCompanyId(((BigInteger) org.get("company_id")).longValue());
            tenantCompanyRelDao.saveAndFlush(rel);

            Optional<Tenant> tenantOptional = tenantDao.findById(rel.getTenantId());
            Optional<Company> companyOptional = companyDao.findById(rel.getCompanyId());
            companyPubService.sendTenantCompanyRelMsg(ADD_TENANT_COMPANY_REL, companyOptional.get(), tenantOptional.get());

        });
    }

    /**
     * 检查扩展信息
     *
     * @param extensions List<Extension>
     */
    private void checkExtensions(List<Request.Extension> extensions) {
        if (CollectionUtils.isEmpty(extensions)) {
            return;
        }
        int maxExtnesion = 10;
        if (extensions.size() > maxExtnesion) {
            throw new IllegalArgumentException("扩展标签个数不能超过【" + maxExtnesion + "】");
        }
        boolean checkResult = extensions.stream().anyMatch(p -> StringUtils.isBlank(p.getExtensionKey()) || StringUtils.isBlank(p.getExtensionValue()));
        if (checkResult) {
            throw new IllegalArgumentException("扩展标签字段不能为空】");
        }
        checkResult = extensions.stream().anyMatch(p -> p.getExtensionKey().length() > MAX_LENT || p.getExtensionValue().length() > MAX_LENT);
        if (checkResult) {
            throw new IllegalArgumentException("扩展标签最大长度不能超过【" + MAX_LENT + "】");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Company saveAndFlush(Company company) {
        // PLAT-5231：修改公司名称， 同时修改组织名称
//        this.updateOrgName(company);
        if (company.getCompanyId() != null) {
            orgStructDao.updateOrgNameByCompanyId(company.getCompanyId(), company.getCompanyName());
        }
        Company saveObj = companyDao.saveAndFlush(company);
        companyPubService.pub(saveObj.getCompanyId());
        return saveObj;
    }

    @SuppressWarnings("all")
    @Deprecated
    private void updateOrgName(Company company) {
        if (company != null && company.getCompanyId() != null) {
            //使用sql 批量更新所有的关联的组织信息
            Long userId = 0L;
            if (UserInfoHolder.get() != null) {
                userId = UserInfoHolder.get().getId();
            }

            //注意：由于公司名称已经在JPA托管态中修改了，其他方式拿不到旧的公司名称，只能使用jdbcTemplate方式!!!
            String sql = "update sys_org_struct so"
                    + " inner join bss_company bc on bc.company_id = so.company_id "
                    + " set so.org_name  = ? ,  so.update_time = current_timestamp, so.update_user_id = ?  "
                    + " where so.company_id = ? "
                    + " and so.org_name = bc.company_name and  bc.company_name!= ? ";
            int updateCount = jdbcTemplate.update(sql, company.getCompanyName(), userId, company.getCompanyId(), company.getCompanyName());

            logger.warn("更新公司：{}, 批量更新组织名称数量：{}", company.getCompanyId(), updateCount);
        }
    }

    public Company info(long companyId, Set<String> withExtendParams) {
        Company company = this.findById(companyId);
        this.fillCompany(company, withExtendParams);
        return company;
    }

    public Company currentInfo(long companyId, Integer status, Date revisionDate, Set<String> withExtendParams) {
        Company company = this.findByIdAndStatusAndRevisionDate(companyId, status, revisionDate, withExtendParams);
        this.fillCompany(company, withExtendParams);
        return company;
    }

    public Page<CompanyPackage> currentPackagesByQuery(Request.CompanyPackageQuery query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        query.setTenantId(authorizedUser.getTenantId());
        return this.packagesByQuery(query, pageable);
    }

    /**
     * 查询数据
     * @return Map<Long, TenantCompany>
     */
    public Map<Long, TenantCompany> findTenantCompanyAll() {
        List<Object[]> companyList = tenantCompanyRelDao.findTenantCommpanyRelAll();
        if (CollectionUtils.isEmpty(companyList)) {
            return Collections.emptyMap();
        }
        Map<Long, TenantCompany> tenantCompanyMap = new HashMap<>(companyList.size());
        for (Object[] objects : companyList) {
            TenantCompany tenantCompany = new TenantCompany();
            tenantCompany.setTenantName(String.valueOf(objects[1]));
            tenantCompany.setTenantCode(String.valueOf(objects[0]));
            tenantCompany.setCompanyId(Long.valueOf(objects[2].toString()));
            tenantCompanyMap.put(tenantCompany.getCompanyId(), tenantCompany);
        }
        return tenantCompanyMap;
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteOrgComRelById(long orgCompanyNumberRelId) {
        orgCompanynoDao.deleteById(orgCompanyNumberRelId);
    }

    public Page<ComparedRevision<Company, String>> findCompanyHistories(long companyId, Pageable pageable) {
        pageable = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), pageable.getSort().and(RevisionSort.desc()));
        Page<ComparedRevision<Company, String>> page = companyDao.findComparedRevisions(companyId, pageable);
        return page;
    }

    public Company revisionInfo(long tenantId, long companyId, Date revision, Set<String> withExtendParams) {
        Company company = this.findByTenantIdAndId(tenantId, companyId);
        if (revision != null) {
            try {
                Company historyCompany = companyDao.findRevisionByLastUpdateTime(companyId, revision);
                if (historyCompany != null) {
                    BeanUtils.copyProperties(historyCompany, company, Stream.of("orgs", "companyNos", "hostTenant", "tenantRels").toArray(String[]::new));
                }
            } catch (Exception e) {
                logger.warn(e.getMessage());
            }
        }
        Tenant tenant = tenantDao.findById(tenantId).orElseThrow(() -> new IllegalArgumentException("非法的租户id(" + tenantId + ")"));
        this.fillTenant(company, tenant);
        this.fillCompany(company, withExtendParams);
        return company;
    }

    public boolean existInHistory(String taxNum, String companyName, Date effectiveDate) {
        long count;
        if (effectiveDate == null) {
            count = companyDao.countAuditByTaxNumAndCompanyName(taxNum, companyName);
        } else {
            count = companyDao.countAuditByTaxNumAndCompanyNameAndEffectiveDate(taxNum, companyName, effectiveDate);
        }
        return count > 0;
    }

    /**
     * 公司审计表更新是否有效
     *
     * @param companyId 公司id
     * @param revision  版本号
     * @param effective 是否有效
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean updateAuditEffective(long companyId, long revision, boolean effective) {
        return companyDao.updateAuditEffective(companyId, revision, effective) > 0;
    }

    @Transactional(rollbackFor = Exception.class)
    public synchronized Map<String, Object> move(Request.Move move) {
        String batchId = UUID.randomUUID().toString();
        SystemLogDTO systemLogDTO = new SystemLogDTO()
                .batchId(batchId)
                .businessType(BusinessTypeEnum.COMPANY_MOVE_TENANT.name());

        Long companyId = move.getCompanyId();
        Long newTenantId = move.getTenantId();

        Map<String, Object> result = new HashMap<>();

        Company company = this.findById(companyId);
        Tenant tenant = tenantDao.findById(newTenantId).orElseThrow(() -> new IllegalArgumentException("未找到租户实体(" + newTenantId + ")"));

        //校验
        List<TenantCompanyRel> tenantCompanyRels = tenantCompanyRelDao.findByCompanyId(companyId);
        if (CollectionUtils.isEmpty(tenantCompanyRels)) {
            throw new IllegalArgumentException("未找到当前公司对应租户");
        }
        if (tenantCompanyRels.size() > 1) {
            List<Long> ortherCompanyIds = tenantCompanyRels.stream().filter(rel -> !rel.getCompanyId().equals(companyId)).map(TenantCompanyRel::getCompanyId).collect(Collectors.toList());
            throw new IllegalArgumentException("只能迁移独立租户的公司，当前公司对应租户下存在其他公司(" + ortherCompanyIds + "), 请确认!");
        }

        Long oldTenantId = null;
        List<OrgStruct> oldCompanyOrgs = orgStructDao.findByCompanyId(companyId);
        if (!CollectionUtils.isEmpty(oldCompanyOrgs)) {
            oldTenantId = oldCompanyOrgs.get(0).getTenantId();
            if (newTenantId.equals(oldTenantId)) {
                throw new IllegalArgumentException("当前公司租户与目标租户一致");
            }
            List<OrgStruct> oldOrgs = orgStructDao.findByTenantId(oldTenantId);
            oldOrgs = oldOrgs.stream().filter(orgStruct -> (null != orgStruct.getParentId() && 0L != orgStruct.getParentId())).collect(Collectors.toList());
            if (oldOrgs.size() > 1) {
                List<Long> ortherOrgIds = oldOrgs.stream()
                        .filter(org -> null != org.getCompanyId())
                        .filter(org -> !org.getCompanyId().equals(companyId))
                        .map(OrgStruct::getOrgId).collect(Collectors.toList());
                throw new IllegalArgumentException("只能迁移独立租户的公司，当前公司对应租户下存在其他组织(" + ortherOrgIds + "), 请确认!");
            }
        }

        //迁移租户与公司关系
        List<Long> oldTenantIds = tenantCompanyRels.stream().map(rel -> rel.getTenantId()).collect(Collectors.toList());
        if (!oldTenantId.equals(oldTenantIds.get(0))) {
            throw new IllegalArgumentException("公司对应组织租户id(" + oldTenantIds.get(0) + ")与公司对应租户id(" + oldTenantId + ")不一致");
        }

        if (newTenantId.equals(oldTenantId)) {
            throw new IllegalArgumentException("当前公司租户与移动租户一致");
        }

        List<User> users = userDao.findByTenantId(oldTenantId);
        List<User> repeUsers = userDao.findByAccountIdIn(users.stream().map(User::getAccountId).collect(Collectors.toList()));

        List<User> errorUsers = repeUsers.stream().filter(user -> user.getTenantId().equals(newTenantId)).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(errorUsers)) {
            throw new IllegalArgumentException("用户(" + errorUsers.stream().map(User::getId).collect(Collectors.toList()) + ")已存在目标租户，请确认");
        }

        List<Pair> logList = new ArrayList<>();

        for (TenantCompanyRel rel : tenantCompanyRels) {
            Tenant oldTenant = tenantDao.findById(rel.getTenantId()).get();
            TenantCompanyRel old = new TenantCompanyRel();
            BeanUtils.copyProperties(rel, old);
            rel.setTenantId(newTenantId);
            TenantCompanyRel ret = tenantCompanyRelDao.saveAndFlush(rel);
            Pair<TenantCompanyRel, TenantCompanyRel> pair = Pair.of(old, ret);
            logList.add(pair);

            companyPubService.sendTenantCompanyRelMsg(DELETE_TENANT_COMPANY_REL, company, oldTenant);
            companyPubService.sendTenantCompanyRelMsg(ADD_TENANT_COMPANY_REL, company, tenant);
        }

        //迁移组织
        List<OrgStruct> rootOrgs = orgStructDao.findRootsByTenantId(newTenantId);
        if (CollectionUtils.isEmpty(rootOrgs)) {
            throw new IllegalArgumentException("未找到根组织实体(" + newTenantId + ")");
        }
        OrgStruct rootOrg = rootOrgs.get(0);

        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setCompanyId(companyId);
        query.setTenantId(oldTenantId);
        query.setStatus(1);
        List<OrgStruct> orgStructs = orgStructDao.findAll(OrgQueryHelper.querySpecification(query));
        orgStructs.forEach(org -> {
            OrgStruct old = new OrgStruct();
            BeanUtils.copyProperties(org, old);
            org.setTenantId(newTenantId);
            org.setParentId(rootOrg.getOrgId());
            OrgStruct ret = orgStructDao.saveAndFlush(org);
            orgPubService.sendOrgMsg("save", ret);
            Pair<OrgStruct, OrgStruct> pair = Pair.of(old, ret);
            logList.add(pair);
        });

        //迁移公司服务包
        List<CompanyServiceRel> companyServiceRels = companyServiceRelDao.findByTenantIdEqualsAndCompanyIdEquals(oldTenantId, companyId);
        List<Long> companyPackageIds = new ArrayList<>();
        companyServiceRels.forEach(old -> {
            CompanyServiceRel newObj = new CompanyServiceRel();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            companyPackageIds.add(old.getId());
            Pair<CompanyServiceRel, CompanyServiceRel> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(companyPackageIds)) {
            companyServiceRelDao.batchUpdateTenant(companyPackageIds, newTenantId);
        }

        //迁移人员
        List<Long> userIds = new ArrayList<>();
        users.forEach(old -> {
            User newObj = new User();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            userIds.add(old.getId());
            Pair<User, User> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(userIds)) {
            userDao.batchUpdateTenant(userIds, newTenantId);
        }

        //迁移人员组织关联
        List<OrgUserRel> orgUserRels = orgUserRelDao.findByTenantId(oldTenantId);
        List<Long> orgUserRelIds = new ArrayList<>();
        orgUserRels.forEach(old -> {
            OrgUserRel newObj = new OrgUserRel();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            orgUserRelIds.add(old.getId());
            Pair<OrgUserRel, OrgUserRel> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(orgUserRelIds)) {
            orgUserRelDao.batchUpdateTenant(orgUserRelIds, newTenantId);
        }

        //修改公司主租户
        company.setHostTenantId(newTenantId);
        companyDao.saveAndFlush(company);

        if (move.isMoveRole()) {
            //迁移角色
            List<Role> roles = roleDao.findByTenantId(oldTenantId);
            List<Long> roleIds = new ArrayList<>();
            List<Role> updateRoles = new ArrayList<>();
            String suffix = RandomUtils.getRandomNum(6);

            roles.forEach(role -> {
                Role old = new Role();
                BeanUtils.copyProperties(role, old);
                role.setName(role.getName() + Separator.UNDERLINE + "copy" + suffix);
                role.setCode(role.getCode() + Separator.UNDERLINE + "copy" + suffix);
                role.setTenantId(newTenantId);
                roleIds.add(role.getId());
                updateRoles.add(role);
                Pair<Role, Role> pair = Pair.of(old, role);
                logList.add(pair);
            });

            if (!CollectionUtils.isEmpty(updateRoles)) {
                roleDao.saveAll(updateRoles);
            }

            //迁移人员角色关联
            List<RoleUserRel> roleUserRels = roleUserRelDao.findByTenantId(oldTenantId);
            List<Long> roleUserRelIds = new ArrayList<>();
            roleUserRels.forEach(old -> {
                RoleUserRel newObj = new RoleUserRel();
                if (1L == old.getRoleId()) {
                    roleUserRelDao.delete(old);
                    Pair<RoleUserRel, RoleUserRel> pair = Pair.of(old, newObj);
                    logList.add(pair);
                } else {
                    BeanUtils.copyProperties(old, newObj);
                    newObj.setTenantId(newTenantId);
                    roleUserRelIds.add(old.getId());
                    Pair<RoleUserRel, RoleUserRel> pair = Pair.of(old, newObj);
                    logList.add(pair);
                }
            });
            if (!CollectionUtils.isEmpty(roleUserRelIds)) {
                roleUserRelDao.batchUpdateTenant(roleUserRelIds, newTenantId);
            }
            result.put("sys_role", roleIds);
            result.put("sys_role_user_rel", roleUserRelIds);
        } else {
            //删除人员角色关联
            List<RoleUserRel> roleUserRels = roleUserRelDao.findByTenantId(oldTenantId);
            roleUserRels.forEach(rel -> {
                roleUserRelDao.delete(rel);
                RoleUserRel newObj = new RoleUserRel();
                Pair<RoleUserRel, RoleUserRel> pair = Pair.of(rel, newObj);
                logList.add(pair);
            });
        }

        systemLogDTO.setActionCode(ActionCodeEnum.U);
        logService.insertLogBatch(logList, systemLogDTO);

        result.put("oldTenantId", oldTenantId);
        result.put("newTenantId", newTenantId);

        result.put("bss_tenant_company_rel", tenantCompanyRels.stream().map(rel -> rel.getId()).collect(Collectors.toList()));
        result.put("sys_org_struct", orgStructs.stream().map(rel -> rel.getOrgId()).collect(Collectors.toList()));
        result.put("bss_company_service_rel", companyPackageIds);
        result.put("sys_user", userIds);
        result.put("sys_org_user_rel", orgUserRelIds);
        result.put("batchId", batchId);

        return result;
    }

    @Transactional(rollbackFor = Exception.class)
    public synchronized Map<String, Object> moveAll(Request.Move move) {

        String batchId = UUID.randomUUID().toString();
        SystemLogDTO systemLogDTO = new SystemLogDTO()
                .batchId(batchId)
                .businessType(BusinessTypeEnum.COMPANY_MOVE_TENANT.name());

        Long companyId = move.getCompanyId();
        Long newTenantId = move.getTenantId();

        Map<String, Object> result = new HashMap<>();

        Company company = this.findById(companyId);
        Tenant tenant = tenantDao.findById(newTenantId).orElseThrow(() -> new IllegalArgumentException("未找到租户实体(" + newTenantId + ")"));

        //校验
        List<TenantCompanyRel> tenantCompanyRels = tenantCompanyRelDao.findByCompanyId(companyId);
        if (CollectionUtils.isEmpty(tenantCompanyRels)) {
            throw new IllegalArgumentException("未找到当前公司对应租户");
        }

        Long oldTenantId = null;
        List<OrgStruct> oldCompanyOrgs = orgStructDao.findByCompanyId(companyId);
        if (!CollectionUtils.isEmpty(oldCompanyOrgs)) {
            oldTenantId = oldCompanyOrgs.get(0).getTenantId();
            if (newTenantId.equals(oldTenantId)) {
                throw new IllegalArgumentException("当前公司租户与目标租户一致");
            }
        }

        //迁移租户与公司关系
        List<Long> oldTenantIds = tenantCompanyRels.stream().map(rel -> rel.getTenantId()).collect(Collectors.toList());
        if (!oldTenantId.equals(oldTenantIds.get(0))) {
            throw new IllegalArgumentException("公司对应组织租户id(" + oldTenantIds.get(0) + ")与公司对应租户id(" + oldTenantId + ")不一致");
        }

        if (newTenantId.equals(oldTenantId)) {
            throw new IllegalArgumentException("当前公司租户与移动租户一致");
        }

        List<User> users = userDao.findByTenantId(oldTenantId);
        List<User> repeUsers = userDao.findByAccountIdIn(users.stream().map(User::getAccountId).collect(Collectors.toList()));

        List<User> errorUsers = repeUsers.stream().filter(user -> user.getTenantId().equals(newTenantId)).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(errorUsers)) {
            throw new IllegalArgumentException("用户(" + errorUsers.stream().map(User::getId).collect(Collectors.toList()) + ")已存在目标租户，请确认");
        }

        List<Pair> logList = new ArrayList<>();

        for (TenantCompanyRel rel : tenantCompanyRels) {
            TenantCompanyRel old = new TenantCompanyRel();
            Tenant oldTenant = tenantDao.findById(rel.getTenantId()).get();
            BeanUtils.copyProperties(rel, old);
            rel.setTenantId(newTenantId);
            TenantCompanyRel ret = tenantCompanyRelDao.saveAndFlush(rel);
            Pair<TenantCompanyRel, TenantCompanyRel> pair = Pair.of(old, ret);
            logList.add(pair);

            companyPubService.sendTenantCompanyRelMsg(DELETE_TENANT_COMPANY_REL, company, oldTenant);
            companyPubService.sendTenantCompanyRelMsg(ADD_TENANT_COMPANY_REL, company, tenant);
        }

        //迁移组织
        List<OrgStruct> rootOrgs = orgStructDao.findRootsByTenantId(newTenantId);
        if (CollectionUtils.isEmpty(rootOrgs)) {
            throw new IllegalArgumentException("未找到根组织实体(" + newTenantId + ")");
        }
        OrgStruct rootOrg = rootOrgs.get(0);

        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setCompanyId(companyId);
        query.setTenantId(oldTenantId);
        query.setStatus(1);
        List<OrgStruct> orgStructs = orgStructDao.findAll(OrgQueryHelper.querySpecification(query));
        orgStructs.forEach(org -> {
            OrgStruct old = new OrgStruct();
            BeanUtils.copyProperties(org, old);
            org.setTenantId(newTenantId);
            org.setParentId(rootOrg.getOrgId());
            OrgStruct ret = orgStructDao.saveAndFlush(org);
            orgPubService.sendOrgMsg("save", ret);
            Pair<OrgStruct, OrgStruct> pair = Pair.of(old, ret);
            logList.add(pair);
        });


        //迁移公司服务包
        List<CompanyServiceRel> companyServiceRels = companyServiceRelDao.findByTenantIdEqualsAndCompanyIdEquals(oldTenantId, companyId);
        List<Long> companyPackageIds = new ArrayList<>();
        companyServiceRels.forEach(old -> {
            CompanyServiceRel newObj = new CompanyServiceRel();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            companyPackageIds.add(old.getId());
            Pair<CompanyServiceRel, CompanyServiceRel> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(companyPackageIds)) {
            companyServiceRelDao.batchUpdateTenant(companyPackageIds, newTenantId);
        }

        //迁移人员
        List<Long> userIds = new ArrayList<>();
        users.forEach(old -> {
            User newObj = new User();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            userIds.add(old.getId());
            Pair<User, User> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(userIds)) {
            userDao.batchUpdateTenant(userIds, newTenantId);
        }

        //迁移角色
        List<Role> roles = roleDao.findByTenantId(oldTenantId);
        List<Long> roleIds = new ArrayList<>();
        roles.forEach(old -> {
            Role newObj = new Role();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            roleIds.add(old.getId());
            Pair<Role, Role> pair = Pair.of(old, newObj);
            logList.add(pair);
        });

        if (!CollectionUtils.isEmpty(roleIds)) {
            roleDao.batchUpdateTenant(roleIds, newTenantId);
        }

        //迁移人员角色关联
        List<RoleUserRel> roleUserRels = roleUserRelDao.findByTenantId(oldTenantId);
        List<Long> roleUserRelIds = new ArrayList<>();
        roleUserRels.forEach(old -> {
            RoleUserRel newObj = new RoleUserRel();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            roleUserRelIds.add(old.getId());
            Pair<RoleUserRel, RoleUserRel> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(roleUserRelIds)) {
            roleUserRelDao.batchUpdateTenant(roleUserRelIds, newTenantId);
        }

        //迁移人员组织关联
        List<OrgUserRel> orgUserRels = orgUserRelDao.findByTenantId(oldTenantId);
        List<Long> orgUserRelIds = new ArrayList<>();
        orgUserRels.forEach(old -> {
            OrgUserRel newObj = new OrgUserRel();
            BeanUtils.copyProperties(old, newObj);
            newObj.setTenantId(newTenantId);
            orgUserRelIds.add(old.getId());
            Pair<OrgUserRel, OrgUserRel> pair = Pair.of(old, newObj);
            logList.add(pair);
        });
        if (!CollectionUtils.isEmpty(orgUserRelIds)) {
            orgUserRelDao.batchUpdateTenant(orgUserRelIds, newTenantId);
        }


        systemLogDTO.setActionCode(ActionCodeEnum.U);
        logService.insertLogBatch(logList, systemLogDTO);

        result.put("oldTenantId", oldTenantId);
        result.put("newTenantId", newTenantId);

        result.put("bss_tenant_company_rel", tenantCompanyRels.stream().map(rel -> rel.getId()).collect(Collectors.toList()));
        result.put("sys_org_struct", orgStructs.stream().map(rel -> rel.getOrgId()).collect(Collectors.toList()));
        result.put("bss_company_service_rel", companyPackageIds);
        result.put("sys_user", userIds);
        result.put("sys_role", roleIds);
        result.put("sys_role_user_rel", roleUserRelIds);
        result.put("sys_org_user_rel", orgUserRelIds);

        return result;

    }

    public Optional<Long> findIdByTaxNum(String taxNum) {
        return companyDao.findIdByTaxNum(taxNum);
    }

    /**
     * 按公司查询ID
     *
     * @param ids 公司Ids
     */
    public List<Company> findByIds(List<Long> ids) {
        return this.companyDao.findByIds(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateCompanyName(Long tenantId, Request.CompanyName companyName) {

        Company company = this.findByTenantIdAndId(tenantId, companyName.getCompanyId());
        if (companyName.getCompanyName().equals(company.getCompanyName())) {
            throw new IllegalArgumentException("更新公司名称与原名称一致!");
        }

        company.setCompanyName(companyName.getCompanyName());
        this.saveAndFlush(company);
    }

    public Response.CompanyInfo findCompanyInfoByCompanyId(long companyId) {
        Company company = this.findById(companyId);
        Response.CompanyInfo companyInfo = new Response.CompanyInfo();
        companyInfo.setHasRelations(false);
        if (company.getHostTenantId() != null && company.getHostTenantId() > 0) {
            Tenant hostTenant = company.getHostTenant();
            if (hostTenant == null || hostTenant.getCreateTime() == null) {
                Optional<Tenant> hostTenantOptional = tenantDao.findById(company.getHostTenantId());
                if (hostTenantOptional.isPresent()) {
                    hostTenant = hostTenantOptional.get();
                }
            }
            if (hostTenant != null && hostTenant.getTenantName() != null) {
                companyInfo.setHostTenantId(hostTenant.getTenantId());
                companyInfo.setHostTenantName(hostTenant.getTenantName());
            }
        }
        List<CompanyTenantDto> companyTenants = orgStructDao.findTenantInfoByCompanyId(companyId);

        List<Response.RelInfo> infos = new ArrayList<>();
        this.fillRelInfos(companyInfo, infos, companyTenants);
        return companyInfo;
    }

    private void fillRelInfos(Response.CompanyInfo companyInfo, List<Response.RelInfo> infos, Collection<CompanyTenantDto> companyTenants) {
        if (companyTenants != null && !companyTenants.isEmpty()) {
            List<CompanyTenantRel> hostRels = null;
            for (CompanyTenantDto companyTenant : companyTenants) {
                Response.RelInfo info = new Response.RelInfo();
                info.setTenantId(companyTenant.getTenantId());
                info.setTenantCode(companyTenant.getTenantCode());
                info.setTenantName(companyTenant.getTenantName());
                info.setRelatedType(companyTenant.getTenantId().equals(companyInfo.getHostTenantId()) ? 1 : 2);
                List<Response.RelItem> items = new ArrayList<>();
                if (hostRels == null) {
                    hostRels = relDao.findAll(new Specification<CompanyTenantRel>() {
                        @Override
                        public Predicate toPredicate(Root<CompanyTenantRel> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                            List<Predicate> predicates = new ArrayList<>();
                            predicates.add(builder.equal(root.<Long>get("companyId"), companyTenant.getCompanyId()));
                            if (!predicates.isEmpty()) {
                                query.where(predicates.stream().toArray(Predicate[]::new));
                            }
                            return query.getRestriction();
                        }
                    }, EntityGraphs.named(CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT));
                }

                this.fillRelItems(companyInfo, info, items, hostRels);
                infos.add(info);
            }
        }
        companyInfo.setInfos(infos);
    }

    private void fillRelItems(Response.CompanyInfo companyInfo, Response.RelInfo relInfo, List<Response.RelItem> items, Collection<CompanyTenantRel> rels) {
        if (rels != null && !rels.isEmpty()) {
            List<CompanyTenantRel> hostRels = rels.stream().filter(r -> r.getTenantId().equals(relInfo.getTenantId())).collect(toList());
            if (hostRels != null && !hostRels.isEmpty()) {
                companyInfo.setHasRelations(true);
                for (CompanyTenantRel rel : hostRels) {
                    Response.RelItem item = new Response.RelItem();
                    item.setTenantId(rel.getTenantId());
                    item.setTenantCode(rel.getTenantCode());
                    item.setTenantName(rel.getTenantName());
                    item.setRelatedTenantId(rel.getRelatedTenantId());
                    item.setRelatedTenantCode(rel.getRelatedTenantCode());
                    item.setRelatedTenantName(rel.getRelatedTenantName());
                    item.setRelatedCompanyId(rel.getRelatedCompanyId());
                    item.setRelatedCompanyName(rel.getRelatedCompanyName());
                    item.setRelatedCompanyCode(rel.getRelatedCompanyCode());
                    item.setRelatedTaxNum(rel.getRelatedTaxNum());
                    item.setSwitches(rel.getSwitches());
                    item.setInvoiceStartDate(rel.getInvoiceStartDate());
                    item.setStatementStartDate(rel.getStatementStartDate());
                    item.setCreateTime(rel.getCreateTime());
                    items.add(item);
                }
            }

        }
        relInfo.setItems(items);
    }

    @Transactional(rollbackFor = Exception.class)
    public void changeHostTenant(Long companyId, Long tenantId) {
        Company company = this.findById(companyId);
        List<Long> orgs = orgStructDao.findIdByTenantIdAndCompanyId(tenantId, companyId);
        if (CollectionUtils.isEmpty(orgs)) {
            throw new IllegalArgumentException("当前公司与租户没有建立所属关系！");
        }

        List<CompanyTenantRel> rels = relDao.findByCompanyId(companyId);
        if (!CollectionUtils.isEmpty(rels)) {
            throw new IllegalArgumentException("该公司开通了多租户共享业务，不可执行所属租户的变更操作！");
        }

        company.setHostTenantId(tenantId);
        companyDao.saveAndFlush(company);
    }

    public Map<Long, Boolean> validateCompanyPackagesWereBoundByRole(long tenantId, long companyId, Set<Long> packageIds) {
        List<Long> existPackageIds = companyServiceRelDao.findPackageIdsByTenantIdAndCompanyId(tenantId, companyId, packageIds);
        Set<Long> deletingPackageIds = packageIds.stream().filter(id -> existPackageIds.stream().noneMatch(existId -> id.equals(existId))).collect(Collectors.toSet());
        List<Map<String, Object>> pairs = companyServiceRelDao.countCompanyPackagesWereBoundByRole(tenantId, companyId, deletingPackageIds);
        Map<Long, Boolean> validation = pairs.stream().collect(Collectors.toMap(pair -> ((BigInteger) pair.get("packageId")).longValue(), pair -> ((BigInteger) pair.get("cnt")).longValue() > 0));
        for (Map.Entry<Long, Boolean> entry : validation.entrySet()) {
            boolean exist = existPackageIds.stream().anyMatch(id -> entry.getKey().equals(id));
            entry.setValue(exist && entry.getValue());
        }
        return validation;
    }

    @Transactional(rollbackFor = Exception.class)
    public boolean handleTaxwareMessage(CompanyTaxwareDto companyTaxwareDto) {
        //不满足条件的数据返回ture并ack，否则会一直拉到相应的数据
        if (companyTaxwareDto == null) {
            logger.warn("cannot handle empty data");
            return true;
        }
        String taxNo = companyTaxwareDto.getTaxCode();
        if (StringUtils.isBlank(taxNo)) {
            logger.warn("cannot handle message without taxNo");
            return true;
        }

        Company company = this.getByTaxNum(taxNo);
        if (company == null) {
            logger.warn("cannot handle taxNo {}, not exist in user center", taxNo);
            return true;
        }
        String natureTaxPayer = companyTaxwareDto.getNatureTaxPayer();
        if (StringUtils.isBlank(natureTaxPayer)) {
            logger.warn("cannot handle message without natureTaxPayer");
        }
        //纳税人资质类型 1:一般增值税纳税人 2:小规模纳税人
        Integer taxPayerType = company.getTaxpayerQualificationType();
        if (taxPayerType == null) {
            if (NORMAL_TAX_PAYER.equalsIgnoreCase(natureTaxPayer)) {
                company.setTaxpayerQualificationType(1);
            } else if (SMALL_TAX_PAYER.equalsIgnoreCase(natureTaxPayer)) {
                company.setTaxpayerQualificationType(2);
            }
        } else {
            switch (natureTaxPayer) {
                case NORMAL_TAX_PAYER:
                    //一般纳税人
                    if (taxPayerType != 1) {
                        company.setTaxpayerQualificationType(1);
                    }
                    break;
                case SMALL_TAX_PAYER:
                    if (taxPayerType != 2) {
                        company.setTaxpayerQualificationType(2);
                    }
                    break;
                default:
                    //其它情况不处理
                    logger.warn("illegal taxPayer type [{}] in message", natureTaxPayer);
                    break;
            }
        }
        List<LimitInfo> list = companyTaxwareDto.getLimitInfoList();
        list.forEach(element -> {
            String invoiceType = element.getInvoiceType();
            BigDecimal limit = element.getSingleAmountLimit();
            if (limit == null) {
                return;
            }
            switch (invoiceType) {
                case "c":
                    if (company.getCquota() == null || !company.getCquota().equals(limit)) {
                        company.setCquota(limit);
                    }
                    break;
                case "ce":
                    if (company.getCeQuota() == null || !company.getCeQuota().equals(limit)) {
                        company.setCeQuota(limit);
                    }
                    break;
                case "s":
                    if (company.getSquota() == null || !company.getSquota().equals(limit)) {
                        company.setSquota(limit);
                    }
                    break;
                case "se":
                    if (company.getSeQuota() == null || !company.getSeQuota().equals(limit)) {
                        company.setSeQuota(limit);
                    }
                    break;
                case "v":
                    if (company.getVehicleLimit() == null || !company.getVehicleLimit().equals(limit)) {
                        company.setVehicleLimit(limit);
                    }
                    break;
                case "ju":
                    if (company.getJuQuota() == null || !company.getJuQuota().equals(limit)) {
                        company.setJuQuota(limit);
                    }
                    break;
                case "vs":
                    //二手机动车暂时不具备，暂不处理
                    logger.warn("current not supported invoice type");
                    break;
                default:
                    logger.warn("illegal invoice type:{}", invoiceType);
            }
        });
        try {
            this.saveAndFlush(company);
            return true;
        } catch (Exception exception) {
            logger.error("error saving company:{}", exception.getMessage());
            //不ack，后续重新拉取处理
            return false;
        }
    }

}
