package com.xforceplus.business.company.service;

import com.google.common.collect.Sets;
import com.xforceplus.api.model.*;
import com.xforceplus.business.excel.company.CompanyPackageDTO;
import com.xforceplus.business.messagebus.CompanyServicePackagePubService;
import com.xforceplus.dao.*;
import com.xforceplus.dao.impl.BatchServiceDaoImpl;
import com.xforceplus.domain.company.CompanyPackageExportDto;
import com.xforceplus.entity.*;
import com.xforceplus.query.CompanyQueryHelper;
import com.xforceplus.query.CompanyServiceRelQueryHelper;
import com.xforceplus.query.TenantQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.joining;

@Service
public class CompanyServicePackageService {
    private final static Logger logger = LoggerFactory.getLogger(CompanyServicePackageService.class);
    @Autowired
    private CompanyServiceRelDao companyServiceRelDao;
    @Autowired
    private ServicePackageDao servicePackageDao;
    @Autowired
    private OrgStructDao orgStructDao;
    @Autowired
    private CompanyService companyService;
    @Autowired
    private BatchServiceDaoImpl batchServiceDaoImpl;

    @Autowired
    private CompanyServicePackagePubService companyServicePackagePubService;

    @Autowired
    private TenantDao tenantDao;

    @Autowired
    private CompanyDao companyDao;

    @Transactional(rollbackFor = Exception.class)
    public void bindCompanyAndPackage(Long tenantId, CompanyModel.Request.BindTaxNumAndPackage bindPackage) {
        if (StringUtils.isBlank(bindPackage.getTaxNum())) {
            throw new IllegalArgumentException("税号不能为空");
        }
        CompanyModel.Request.Query query = new CompanyModel.Request.Query();
        query.setTaxNum(bindPackage.getTaxNum());
        List<Company> companies = companyService.list(query, Sort.unsorted());
        if (CollectionUtils.isEmpty(companies)) {
            throw new IllegalArgumentException("公司税号(" + bindPackage.getTaxNum() + ")不存在");
        }
        Long companyId = companies.stream().map(Company::getCompanyId).filter(Objects::nonNull).findFirst().orElseThrow(() -> new IllegalArgumentException("不存在companyId"));
        companyService.bindPackages(tenantId, companyId, bindPackage);
    }

    @Transactional(rollbackFor = Exception.class)
    public void createServicePackageAndBindCompanyFromResourceSet(List<CompnayResourceSetModel.Request.CompanyPackageBind> companyPackageBindRequest) {
        List<Long> tenantIds = companyPackageBindRequest.stream().map(CompnayResourceSetModel.Request.CompanyPackageBind::getTenantId).filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(tenantIds) || companyPackageBindRequest.size() != tenantIds.size()) {
            throw new IllegalArgumentException("租户id不能为空");
        }

        List<Long> resourcesetIds = companyPackageBindRequest.stream().map(CompnayResourceSetModel.Request.CompanyPackageBind::getResourceSetId).filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(resourcesetIds) || companyPackageBindRequest.size() != resourcesetIds.size()) {
            throw new IllegalArgumentException("功能集id不能为空");
        }
        Long servicePackageId = this.createServicePackage(companyPackageBindRequest);
        this.bindServicePackageAndCompany(servicePackageId, companyPackageBindRequest);
    }

    private void bindServicePackageAndCompany(final Long servicePackageId, List<CompnayResourceSetModel.Request.CompanyPackageBind> binds) {
        final Long tenantId = binds.get(0).getTenantId();
        List<Company> companies = orgStructDao.findCompaniesByTenantId(tenantId);
        if (CollectionUtils.isEmpty(companies)) {
            return;
        }

        batchServiceDaoImpl.deleteCompanyPackageByPackageIdAndTenantId(servicePackageId, tenantId);

        List<CompanyServiceRel> companyServices = companies.stream().map(bind -> {
            Long companyId = bind.getCompanyId();
            CompanyServiceRel companyServiceRel = new CompanyServiceRel();
            companyServiceRel.setCompanyId(companyId);
            companyServiceRel.setServicePackageId(servicePackageId);
            companyServiceRel.setRemarks("数据迁移自动绑定");
            companyServiceRel.setTenantId(tenantId);
            companyServiceRel.setStatus(1);
            companyServiceRel.setOperateReason("数据迁移自动绑定" + tenantId);
            return companyServiceRel;
        }).collect(Collectors.toList());
        companyServices = companyServiceRelDao.saveAllAndFlush(companyServices);
        companyServiceRelDao.flush();

    }

    private Long createServicePackage(List<CompnayResourceSetModel.Request.CompanyPackageBind> companyPackageBinds) {
        String newPackageName = "迁移默认服务包-" + companyPackageBinds.get(0).getTenantId();
        List<ServicePackage> servicePackages = companyServiceRelDao.findByTenantId(companyPackageBinds.get(0).getTenantId());
        Optional<ServicePackage> optionalServicePackage = servicePackages.stream().filter(p -> p.getServicePackageName().equals(newPackageName)).findFirst();
        Long servicePackageId;
        if (optionalServicePackage.isPresent()) {
            ServicePackage servicePackage = optionalServicePackage.get();
            batchServiceDaoImpl.deletePackageRelResourcesetByPackageId(optionalServicePackage.get().getServicePackageId());
            servicePackageId = servicePackage.getServicePackageId();
        } else {
            ServicePackage servicePackage = new ServicePackage();
            servicePackage.setAppId(companyPackageBinds.get(0).getAppId());
            servicePackage.setStatus(1);
            servicePackage.setServicePackageName(newPackageName);
            ServicePackage entity = servicePackageDao.saveAndFlush(servicePackage);
            servicePackageId = entity.getServicePackageId();
        }
        if (CollectionUtils.isNotEmpty(companyPackageBinds)) {
            Long finalServicePackageId = servicePackageId;
            List<ServiceResourcesetRel> serviceResourcesetRels = companyPackageBinds.stream().map(bind -> {
                ServiceResourcesetRel serviceResourcesetRel = new ServiceResourcesetRel();
                serviceResourcesetRel.setResourcesetId(bind.getResourceSetId());
                serviceResourcesetRel.setServicePackageId(finalServicePackageId);
                return serviceResourcesetRel;
            }).collect(Collectors.toList());
            try {
                batchServiceDaoImpl.saveList(serviceResourcesetRels);
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
                logger.error("插入数据异常{}", companyPackageBinds.get(0).getTenantId());
            }
        }
        return servicePackageId;
    }

    @Transactional(rollbackFor = Exception.class)
    public void changeStatusByTenantIdAndCompanyId(long tenantId, long companyId, int status, String reason) {
        companyServiceRelDao.changeStatusByTenantIdAndCompanyId(tenantId, companyId, status, reason);
    }

    @Transactional(rollbackFor = Exception.class)
    public void changeCompanyServiceStatus(long tenantId, long companyId,List<Long> servicePackageIds, int status, String reason) {
        List<CompanyServiceRel> rels = companyServiceRelDao.findRelsByTenantIdAndCompanyIdAndPackageIds(tenantId,companyId,servicePackageIds);
        rels.forEach(rel -> {
            rel.setStatus(status);
            rel.setOperateReason(reason);
            companyServiceRelDao.saveAndFlush(rel);
        });
    }

    public List<CompanyServiceRel> findList(long tenantId, long companyId) {
        Specification<CompanyServiceRel> specification = (Specification<CompanyServiceRel>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (tenantId > 0 && companyId > 0) {
                predicates.add(builder.equal(root.<Long>get("tenantId"), tenantId));
                predicates.add(builder.equal(root.<Long>get("companyId"), companyId));
            }

            if (predicates.isEmpty()) {
                throw new IllegalArgumentException("缺少查询参数");
            } else {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        List<CompanyServiceRel> rels = companyServiceRelDao.findAll(specification);
        return rels;
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindCompanyPackages(CompanyModel.Request.BindCompanyPackages bindCompanyPackages) {
        if (bindCompanyPackages == null) {
            String message = "缺少参数 bindCompanyPackages";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
        Tenant tenant = tenantDao.findById(bindCompanyPackages.getTenantId()).orElseThrow(() -> new IllegalArgumentException("未查到对应的租户(" + bindCompanyPackages.getTenantId() + ")"));
        CompanyServiceRelModel.Response.TenantModel tenantModel = new CompanyServiceRelModel.Response.TenantModel();
        BeanUtils.copyProperties(tenant, tenantModel);

        List<Long> packageIds = bindCompanyPackages.getPackageIds();
        if (packageIds == null) {
            String message = "packageId是必填字段";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
        logger.info("packageIds = " + packageIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        final List<Long> companyIds = bindCompanyPackages.getCompanyIds() != null ? bindCompanyPackages.getCompanyIds() : new ArrayList<>();
        if (bindCompanyPackages.getCompanyId() != null && bindCompanyPackages.getCompanyId() > 0) {
            companyIds.add(bindCompanyPackages.getCompanyId());
        }
        CompanyModel.Request.Query query = new CompanyModel.Request.Query();
        query.setTenantId(tenant.getTenantId());
        query.setCompanyIds(companyIds.toArray(new Long[] {}));
        List<Company> companies = companyDao.findAll(CompanyQueryHelper.querySpecification(query));
        logger.info("companyIds = " + packageIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        if (companies.isEmpty()) {
            String message = "没有效的companyId";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
        List<Long> availableCompanyIds = companies.stream().map(Company::getCompanyId).collect(Collectors.toList());
        Set<Long> unavailableCompanyIds = companyIds.stream().filter(companyId -> !availableCompanyIds.contains(companyId)).collect(Collectors.toSet());
        if (!unavailableCompanyIds.isEmpty()) {
            String message = unavailableCompanyIds.stream().map(Object::toString).collect(Collectors.joining(",", "[", "]"));
            message = message + "不是有效的公司id";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }

        Map<Long, Company> companyMap = companies.stream().collect(Collectors.toMap(Company::getCompanyId, Function.identity(), (key1, key2) -> key1));

        bindCompanyPackages.setCompanyIds(availableCompanyIds);
        List<CompanyServiceRel> rels = this.findCompanyServiceRelsByTenantIdAndCompanyIds(bindCompanyPackages.getTenantId(), companyIds);
        IAuthorizedUser iAuthorizedUser = UserInfoHolder.get();
        //region 往绑定不在数据库中但是报文 resourcesetIds 中有的服务包-功能集关系
        Set<CompanyServiceRel> insertingRels = new HashSet<>();
        List<CompanyServiceRelModel.Response.CompanyServiceModel> pubRels = new ArrayList<>();
        Map<Long, ServicePackage> servicePackageMap = new HashMap<>();
        Set<String> errors = new HashSet<>();
        bindCompanyPackages.getCompanyIds().forEach(companyId -> packageIds.stream()
                .filter(Objects::nonNull)
                .filter(packageId -> rels.stream()
                        .noneMatch(companyServiceRel -> companyServiceRel.getServicePackageId().equals(packageId) && (companyId.equals(companyServiceRel.getCompanyId()))))
                .forEach(servicePackageId -> {
                    Optional<ServicePackage> packageOptional = servicePackageDao.findById(servicePackageId);
                    if (packageOptional.isPresent()) {
                        servicePackageMap.put(packageOptional.get().getServicePackageId(), packageOptional.get());
                        CompanyServiceRel rel = new CompanyServiceRel();
                        rel.setServicePackageId(servicePackageId);
                        rel.setCompanyId(companyId);
                        rel.setTenantId(bindCompanyPackages.getTenantId());
                        if (bindCompanyPackages.getStatus() != null) {
                            rel.setStatus(bindCompanyPackages.getStatus());
                        } else {
                            rel.setStatus(1);
                        }
                        rel.setRemarks(StringUtils.EMPTY);
                        rel.setOperateReason(StringUtils.EMPTY);
                        if (iAuthorizedUser != null) {
                            rel.setCreateUserName(iAuthorizedUser.getUsername());
                        }
                        insertingRels.add(rel);
                    } else {
                        errors.add("不存在服务包(" + servicePackageId + ")");
                    }
                }));

        if (!errors.isEmpty()) {
            throw new IllegalArgumentException(errors.stream().collect(joining(",")));
        }

        if (CollectionUtils.isNotEmpty(insertingRels)) {
            if (!CollectionUtils.isEmpty(insertingRels)) {
                //修改为批量保存，提高保存效率
                this.companyServiceRelDao.saveAllAndFlush(insertingRels);
            }
            List<CompanyServiceRelModel.Response.CompanyServiceModel> insertPubs =
                    companyServicePackagePubService.buildCompanyServiceMessage(servicePackageMap, companyMap, insertingRels, "I");
            pubRels.addAll(insertPubs);
        }
        //endregion
        if (bindCompanyPackages.isOverwrite()) {
            List<Long> deleteIds = new ArrayList<>();
            Set<CompanyServiceRel> deleteRels = new HashSet<>();
            //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
            rels.stream().filter(rel -> !packageIds.contains(rel.getServicePackageId())).forEach(rel -> {
                companyServiceRelDao.deleteById(rel.getId());
                deleteIds.add(rel.getId());
                deleteRels.add(rel);
                Optional<ServicePackage> packageOptional = servicePackageDao.findById(rel.getServicePackageId());
                if (packageOptional.isPresent()) {
                    servicePackageMap.put(packageOptional.get().getServicePackageId(), packageOptional.get());
                }
            });
            if (CollectionUtils.isNotEmpty(deleteIds)) {
                List<CompanyServiceRelModel.Response.CompanyServiceModel> deletePubs =
                        companyServicePackagePubService.buildCompanyServiceMessage(servicePackageMap, companyMap, deleteRels, "D");
                pubRels.addAll(deletePubs);
            }
            //endregion
        }

        if (CollectionUtils.isEmpty(pubRels)) {
            logger.info("公司服务包没有变更，不进行pub");
            return;
        }
        String channelIdStr = null == bindCompanyPackages.getChannelId() ? "" : bindCompanyPackages.getChannelId().toString();
        companyServicePackagePubService.sendMessage(channelIdStr, tenantModel, pubRels);

    }

    private List<CompanyServiceRel> findCompanyServiceRelsByTenantIdAndCompanyIds(Long tenantId, Collection<Long> companyIds) {
        if (tenantId == null || tenantId == 0 || CollectionUtils.isEmpty(companyIds)) {
            return new ArrayList<>();
        }
        Specification<CompanyServiceRel> specification = (Specification<CompanyServiceRel>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.<Long>get("tenantId"), tenantId));
            if (!CollectionUtils.isEmpty(companyIds)) {
                if (companyIds.size() == 1) {
                    predicates.add(builder.equal(root.<Long>get("companyId"), companyIds.stream().findFirst().get()));
                } else {
                    predicates.add(root.<Long>get("companyId").in(new HashSet<>(companyIds)));
                }
            }

            if (predicates.isEmpty()) {
                throw new IllegalArgumentException("缺少查询参数");
            } else {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        List<CompanyServiceRel> rels = companyServiceRelDao.findAll(specification);
        logger.info("exist CompanyServiceRels.size = " + rels.size());
        return rels;
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindCompanyPackages(long id) {
        CompanyServiceRel companyServiceRel = companyServiceRelDao.findById(id, EntityGraphs.named(CompanyServiceRel.NAMED_ENTITY_GRAPH_DEFAULT)).orElseThrow(() -> new IllegalArgumentException("未找到公司服务包(" + id + ")"));
        companyServiceRelDao.deleteById(id);
        this.pub(companyServiceRel);

    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByCompanyIdAndTenantId(Long companyId, Long tenantId) {
        CompanyServiceRelModel.Request.Query query = new CompanyServiceRelModel.Request.Query();
        query.setTenantId(tenantId);
        query.setCompanyId(companyId);
        List<CompanyServiceRel> companyServiceRels = companyServiceRelDao.findAll(CompanyServiceRelQueryHelper.querySpecification(query), EntityGraphs.named(CompanyServiceRel.NAMED_ENTITY_GRAPH_DEFAULT));
        if(CollectionUtils.isNotEmpty(companyServiceRels)){
            for (CompanyServiceRel companyServiceRel : companyServiceRels) {
                companyServiceRelDao.delete(companyServiceRel);
                this.pub(companyServiceRel);
            }
        }
    }

    private void pub(CompanyServiceRel companyServiceRel) {
        //发布订阅
        CompanyServiceRelModel.Response.TenantModel tenantModel = new CompanyServiceRelModel.Response.TenantModel();
        Tenant tenant = companyServiceRel.getTenant();
        if (tenant == null || tenant.getCreateTime() == null) {
            tenant = tenantDao.getById(companyServiceRel.getTenantId());
        }
        BeanUtils.copyProperties(tenant, tenantModel);
        Company company = companyServiceRel.getCompany();
        if (company == null || company.getTaxNum() == null) {
            company = companyDao.getById(companyServiceRel.getCompanyId());
        }
        Map<Long, Company> companyMap = new HashMap<>(1);
        companyMap.put(companyServiceRel.getCompanyId(), company);

        ServicePackage servicePackage = companyServiceRel.getServicePackage();
        if (servicePackage == null || servicePackage.getCreateTime() == null) {
            servicePackage = servicePackageDao.getById(companyServiceRel.getServicePackageId());
        }

        Map<Long, ServicePackage> servicePackageMap = new HashMap<>(1);
        servicePackageMap.put(companyServiceRel.getServicePackageId(), servicePackage);
        Set<CompanyServiceRel> relSet = Sets.newHashSet(companyServiceRel);
        List<CompanyServiceRelModel.Response.CompanyServiceModel> delData = companyServicePackagePubService.buildCompanyServiceMessage(servicePackageMap, companyMap, relSet, "D");
        companyServicePackagePubService.sendMessage(null, tenantModel, delData);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindCompanyAndPackage(String tenantCode, String taxNum, String servicePackageCode, Integer status, boolean overwrite) {
        TenantModel.Request.Query tenantQuery = new TenantModel.Request.Query();
        tenantQuery.setTenantCode(tenantCode);
        Specification<Tenant> tenantSpecification = TenantQueryHelper.queryOneSpecification(tenantQuery);
        Tenant tenant = tenantDao.findOne(tenantSpecification).orElseThrow(() -> new IllegalArgumentException("未查到对应的租户"));
        CompanyModel.Request.Query companyQuery = new CompanyModel.Request.Query();
        companyQuery.setTaxNum(taxNum);
        Specification<Company> companySpecification = CompanyQueryHelper.queryOneSpecification(companyQuery);
        Company company = companyDao.findOne(companySpecification).orElseThrow(() -> new IllegalArgumentException("未查到对应的公司"));

        // 租户和公司如果没有建立关联，不能绑定
        boolean tenantCompanyRel = company.getOrgs().stream().map(OrgStruct::getTenantId).anyMatch(tid -> tid.equals(tenant.getTenantId()));

        if (!tenantCompanyRel) {
            throw new IllegalArgumentException("当前租户未和公司建立关联，无法绑定");
        }

        List<CompanyServiceRel> companyServiceRels = companyServiceRelDao.findByTenantIdEqualsAndCompanyIdEquals(tenant.getTenantId(), company.getCompanyId());
        if (CollectionUtils.isNotEmpty(companyServiceRels) && overwrite) {
            companyServiceRelDao.deleteAll(companyServiceRels);
        }
        String[] servicePackageCodeList = servicePackageCode.split("/");
        if (servicePackageCodeList == null || servicePackageCodeList.length <= 0) {
            String message = "服务包code为空。";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
        List<ServicePackage> servicePackageArrayList = new ArrayList<>();
        for (String item : servicePackageCodeList) {
            List<ServicePackage> servicePackages = servicePackageDao.findServicePackagesByServicePackageCodeEqualsAndStatusEquals(item, 1);
            if (CollectionUtils.isEmpty(servicePackages)) {
                String message = "未找到对应服务包。" + item;
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
            if (servicePackages.size() > 1) {
                String message = "该服务包code对应多个服务包。" + item;
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
            servicePackageArrayList.add(servicePackages.get(0));
        }
        Map<String, CompanyServiceRel> existRels = companyServiceRels.stream()
                .collect(Collectors.toMap(k -> (k.getServicePackageId() + "_" + k.getCompanyId() + "_" + k.getTenantId()).toString(), part -> part));

        Map<Long, ServicePackage> servicePackageMap = new HashMap<>();
        Map<Long, Company> companyMap = new HashMap<>();
        Set<CompanyServiceRel> saveSet = new HashSet<>();
        for (ServicePackage servicePackage : servicePackageArrayList) {
            if (!overwrite && org.springframework.util.CollectionUtils.isEmpty(existRels)) {
                if (null != existRels.get(servicePackage.getServicePackageId() + "_" + company.getCompanyId() + "_" + tenant.getTenantId())) {
                    continue;
                }
            }
            CompanyServiceRel rel = new CompanyServiceRel();
            rel.setServicePackageId(servicePackage.getServicePackageId());
            rel.setCompanyId(company.getCompanyId());
            rel.setTenantId(tenant.getTenantId());
            if (status != null) {
                rel.setStatus(status);
            } else {
                rel.setStatus(1);
            }

            rel.setRemarks("excel导入");
            rel.setOperateReason(StringUtils.EMPTY);
            rel = companyServiceRelDao.saveAndFlush(rel);
            saveSet.add(rel);

            servicePackageMap.put(servicePackage.getServicePackageId(), servicePackage);
            companyMap.put(company.getCompanyId(), company);

        }

        //向集成平台发送订阅消息
        CompanyServiceRelModel.Response.TenantModel tenantModel = new CompanyServiceRelModel.Response.TenantModel();
        BeanUtils.copyProperties(tenant, tenantModel);

        List<CompanyServiceRelModel.Response.CompanyServiceModel> pubRels =
                companyServicePackagePubService.buildCompanyServiceMessage(servicePackageMap, companyMap, saveSet, "I");
        companyServicePackagePubService.sendMessage(null, tenantModel, pubRels);
    }


    public List<CompanyPackageExportDto> getExportData(List<Long> companyIds, List<Long> tenantIds) {
        List<CompanyPackageExportDto> companyPackageExportDtos = new ArrayList<>();

        for (int i = 0; i < companyIds.size(); i++) {

            Company company = companyService.findById(companyIds.get(i));
            Optional<Tenant> optionalTenant = tenantDao.findById(tenantIds.get(i));
            if (!optionalTenant.isPresent()) {
                continue;
            }
            Tenant tenant = optionalTenant.get();

            CompanyPackageExportDto dto = new CompanyPackageExportDto();
            dto.setTaxNum(company.getTaxNum());
            dto.setTenantCode(tenant.getTenantCode());
            CompanyServiceRelModel.Request.Query query = new CompanyServiceRelModel.Request.Query();
            query.setTenantId(tenant.getTenantId());
            query.setCompanyId(company.getCompanyId());
            List<CompanyServiceRel> companyServiceRels = companyServiceRelDao.findAll(CompanyServiceRelQueryHelper.querySpecification(query), EntityGraphs.named(CompanyServiceRel.NAMED_ENTITY_GRAPH_SERVICE_PACKAGE));

            if (CollectionUtils.isNotEmpty(companyServiceRels)) {
                if (1 == companyServiceRels.get(0).getStatus()) {
                    dto.setStatusName("启用");
                } else if (0 == companyServiceRels.get(0).getStatus()) {
                    dto.setStatusName("停用");
                }
                dto.setPackageCodes(companyServiceRels.stream().map(CompanyServiceRel::getServicePackage).map(ServicePackage::getServicePackageCode).collect(Collectors.joining("/")));
            }
            companyPackageExportDtos.add(dto);
        }

        return companyPackageExportDtos;
    }

    public List<CompanyPackageDTO> getBatchExportData(List<Long> companyIds, List<Long> tenantIds) {
        List<CompanyPackageDTO> companyPackageDTOList = new ArrayList<>();

        for (int i = 0; i < companyIds.size(); i++) {

            Company company = companyService.findById(companyIds.get(i));
            Optional<Tenant> optionalTenant = tenantDao.findById(tenantIds.get(i));
            if (!optionalTenant.isPresent()) {
                continue;
            }
            Tenant tenant = optionalTenant.get();
            CompanyServiceRelModel.Request.Query query = new CompanyServiceRelModel.Request.Query();
            query.setTenantId(tenant.getTenantId());
            query.setCompanyId(company.getCompanyId());
            List<CompanyServiceRel> companyServiceRels = companyServiceRelDao.findAll(CompanyServiceRelQueryHelper.querySpecification(query), EntityGraphs.named(CompanyServiceRel.NAMED_ENTITY_GRAPH_SERVICE_PACKAGE));

            if (CollectionUtils.isNotEmpty(companyServiceRels)) {
                String statusName = 1 == companyServiceRels.get(0).getStatus() ? "启用" : "停用";
                companyServiceRels.stream().map(CompanyServiceRel::getServicePackage)
                        .forEach((servicePackage) -> {
                            CompanyPackageDTO dto = new CompanyPackageDTO();
                            dto.setStatusName(statusName);
                            dto.setTaxNum(company.getTaxNum());
                            dto.setTenantName(tenant.getTenantName());
                            dto.setTenantCode(tenant.getTenantCode());
                            dto.setCompanyName(company.getCompanyName());
                            dto.setPackageCode(servicePackage.getServicePackageCode());
                            dto.setPackageName(servicePackage.getServicePackageName());
                            companyPackageDTOList.add(dto);
                        });
            }
        }
        return companyPackageDTOList;
    }

}
