package com.xforceplus.business.company.service;

import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.business.tenant.service.WrapperOrgService;
import com.xforceplus.dao.CompanyTenantRelDao;
import com.xforceplus.dao.CompanyTenantRelOperationDao;
import com.xforceplus.domain.company.CompanyTenantRelOperationDto;
import com.xforceplus.entity.CompanyTenantRel;
import com.xforceplus.entity.CompanyTenantRelOperation;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgType;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;

/**
 * 公司租户关联关系邀请处理服务
 * @author geewit
 */
@Slf4j
@Service
public class CompanyTenantRelOperationProcess {
    @Autowired
    private CompanyTenantRelOperationService operationService;
    @Autowired
    private CompanyTenantRelOperationDao operationDao;
    @Autowired
    private CompanyTenantRelService companyTenantRelService;
    @Autowired
    private CompanyTenantRelDao companyTenantRelDao;
    @Autowired
    private OrgService orgService;
    @Autowired
    private WrapperOrgService wrapperOrgService;

    /**
     * 审核通过
     * @param operationId
     * @param remark
     */
    @Transactional(rollbackFor = Exception.class)
    public void accept(long operationId, String remark) {
        CompanyTenantRelOperation operation = operationService.findById(operationId, Collections.emptySet());
        if(CompanyTenantRelOperationDto.Status.ACCEPTED.value() == operation.getStatus()) {
            throw new IllegalArgumentException("该邀请已经接受了, 请不要重复操作");
        }
        if(CompanyTenantRelOperationDto.Status.REFUSED.value() == operation.getStatus()) {
            throw new IllegalArgumentException("该邀请已经被拒绝了");
        }
        if(CompanyTenantRelOperationDto.Status.CANCELED.value() == operation.getStatus()) {
            throw new IllegalArgumentException("该邀请已经被取消了");
        }
        operation.setStatus(CompanyTenantRelOperationDto.Status.ACCEPTED.value());
        operation.setRelatedCompanyCode(remark);
        operation.setOperationType(0);
        operation = operationDao.saveAndFlush(operation);
        List<CompanyTenantRel> existRels = companyTenantRelDao.findByTenantIdAndCompanyId(operation.getTenantId(), operation.getCompanyId());
        if (existRels != null && !existRels.isEmpty()) {
            for (CompanyTenantRel existRel : existRels) {
                //region 如果有不一致的开关变化就处理, 否则就 continue
                if ((existRel.getSwitches() ^ operation.getSwitches()) > 0) {
                    companyTenantRelService.save(
                            operation.getInvitingCompanyId(),
                            operation.getInvitingCompanyName(),
                            operation.getCompanyId(),
                            operation.getTenantId(),
                            operation.getRelatedCompanyId(),
                            operation.getRelatedTenantId(),
                            operation.getSwitches(),
                            existRel.getSwitches(),
                            operation.getInvoiceStartDate(),
                            operation.getStatementStartDate(),
                            remark);
                }
                //endregion
            }
        } else {
            List<OrgStruct> parentOrgs = orgService.findByTenantIdAndCompanyId(operation.getRelatedTenantId(), operation.getCompanyId());
            OrgStruct parentOrg;
            if(parentOrgs.isEmpty()) {
                Page<OrgStruct> rootOrgsPage = orgService.findRoots(operation.getRelatedTenantId(), PageRequest.ofSize(1));
                long relatedTenantId = operation.getRelatedTenantId();
                parentOrg = rootOrgsPage.stream().findAny().orElseThrow(() -> new IllegalArgumentException("被邀请的租户(" + relatedTenantId + ")没有根组织"));
            } else {
                parentOrg = parentOrgs.stream().findAny().orElseThrow(()->new IllegalArgumentException("被邀请的租户没有父组织"));
            }
            OrgModel.Request.Save saveOrg = new OrgModel.Request.Save();
            saveOrg.setTenantId(operation.getRelatedTenantId());
            saveOrg.setOrgName(operation.getCompanyName());
            saveOrg.setOrgDesc(operation.getCompanyName() + "(外部关联的公司)");
            saveOrg.setOrgType(OrgType.COMPANY);
            saveOrg.setOverwrite(Boolean.TRUE);
            saveOrg.setParentId(parentOrg.getOrgId());
            saveOrg.setStatus(1);
            CompanyModel.Request.Save companySave = new CompanyModel.Request.Save();
            companySave.setCompanyId(operation.getCompanyId());
            companySave.setCompanyName(operation.getCompanyName());
            companySave.setTaxNum(operation.getTaxNum());
            companySave.setCompanyCode(operation.getCompanyCode());
            saveOrg.setCompany(companySave);
            saveOrg.setUniqueCompany(Boolean.TRUE);
            saveOrg.setWithApplication(Boolean.FALSE);
            saveOrg.setIsAutoBindParentOrgUsers(Boolean.TRUE);
            //关闭严格模式
            saveOrg.setIsStrict(Boolean.FALSE);
            wrapperOrgService.create(saveOrg);
            companyTenantRelService.save(
                    operation.getInvitingCompanyId(),
                    operation.getInvitingCompanyName(),
                    operation.getCompanyId(),
                    operation.getTenantId(),
                    operation.getRelatedCompanyId(),
                    operation.getRelatedTenantId(),
                    operation.getSwitches(),
                    0,
                    operation.getInvoiceStartDate(),
                    operation.getStatementStartDate(),
                    remark);
        }
    }


    /**
     * 审核拒绝
     * @param operationId
     * @param remark
     */
    @Transactional(rollbackFor = Exception.class)
    public void refuse(long operationId, String remark) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        CompanyTenantRelOperation operation = operationService.findById(operationId, Collections.emptySet());
        if(!operation.getRelatedTenantId().equals(authorizedUser.getTenantId())) {
            throw new IllegalArgumentException("当前登录账户所在租户(" + authorizedUser.getTenantName() + ")和受邀租户(" + operation.getRelatedTenantName() + ")不一致");
        }
        if(CompanyTenantRelOperationDto.Status.ACCEPTED.value() == operation.getStatus()) {
            throw new IllegalArgumentException("该邀请已经接受了");
        }
        if(CompanyTenantRelOperationDto.Status.REFUSED.value() == operation.getStatus()) {
            throw new IllegalArgumentException("该邀请已经被拒绝了, 请不要重复操作");
        }
        if(CompanyTenantRelOperationDto.Status.CANCELED.value() == operation.getStatus()) {
            throw new IllegalArgumentException("该邀请已经被取消了");
        }
        operation.setStatus(CompanyTenantRelOperationDto.Status.REFUSED.value());
        operation.setAuditRemark(remark);
        operationDao.saveAndFlush(operation);
    }

    /**
     * 解除关系
     *
     * @param relId
     */
    @Transactional(rollbackFor = Exception.class)
    public void unrelateById(long relId) {
        CompanyTenantRel existRel = companyTenantRelService.findById(relId);
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (authorizedUser != null) {
            if(!authorizedUser.getTenantId().equals(existRel.getTenantId()) && !authorizedUser.getTenantId().equals(existRel.getRelatedTenantId()) && !authorizedUser.getTenantId().equals(existRel.getHostTenantId())) {
                throw new IllegalArgumentException("没有权限执行此操作");
            }
        }
        orgService.deleteByTenantIdAndCompanyId(existRel.getRelatedTenantId(), existRel.getCompanyId());
        companyTenantRelDao.deleteById(existRel.getId());
        operationService.saveUnrelateOperation(existRel.getCompany(), existRel);
    }
}
