package com.xforceplus.business.company.service;

import com.xforceplus.api.model.CompanyTenantRelOperationModel.Request;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.dao.*;
import com.xforceplus.domain.company.CompanyTenantRelOperationDto;
import com.xforceplus.domain.company.Switch;
import com.xforceplus.entity.*;
import com.xforceplus.query.CompanyTenantRelOperationQueryHelper;
import com.xforceplus.query.OrgQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.Tuple;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 公司租户关联关系邀请 Service
 *
 * @author geewit
 */
@Service
public class CompanyTenantRelOperationService {
    private final static Logger logger = LoggerFactory.getLogger(CompanyTenantRelOperationService.class);

    private final CompanyTenantRelOperationDao operationDao;

    @Autowired
    private CompanyTenantRelService relService;

    @Autowired
    private OrgStructDao orgDao;
    @Autowired
    private TenantDao tenantDao;
    @Autowired
    private CompanyDao companyDao;
    @Autowired
    private CompanyTenantRelDao companyTenantRelDao;
    @Autowired
    private CompanyTenantRelAuditDao auditDao;
    @Autowired
    private OrgService orgService;

    public CompanyTenantRelOperationService(CompanyTenantRelOperationDao operationDao) {
        this.operationDao = operationDao;
    }

    /**
     * 邀请
     *
     * @param model
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public CompanyTenantRelOperation invite(Request.Invitation model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();

        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setCompanyId(model.getInvitingCompanyId());
        query.setTenantId(tenantId);
        query.setStatus(1);
        List<OrgStruct> invitingOrgs = orgDao.findAll(OrgQueryHelper.querySpecification(query));
        if (invitingOrgs.isEmpty()) {
            throw new IllegalArgumentException("邀请方公司在当前租户下不不存在");
        }
        Company company = null;
        Long hostTenantId = null;
        if (model.getCompanyName() != null && model.getTaxNum() != null) {
            List<Company> companies = orgDao.findCompaniesByTenantIdAndCompanyNameAndTaxNum(tenantId, model.getCompanyName(), model.getTaxNum());
            if (companies == null || companies.isEmpty()) {
                /**
                 * 低频日志 日志不做处理
                 */
                String message = "非法的公司税号(" + model.getTaxNum() + ")和名称(" + model.getCompanyName() + ")";
                logger.warn(message);
                throw new IllegalArgumentException(message);
            } else {
                company = companies.stream().findFirst().get();
                hostTenantId = company.getHostTenantId();
            }
        }
        Tenant relatedTenant = tenantDao.findByTenantCode(model.getInvitatedTenantCode());
        if (relatedTenant == null) {
            /**
             * 低频日志 日志不做处理
             */
            String message = "不存在的租户代码(" + model.getInvitatedTenantCode() + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        if (relatedTenant.getTenantId().equals(company.getHostTenantId())) {
            /**
             * 低频日志 日志不做处理
             */
            String message = "受邀方租户和公司所属租户不能为同一个";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }

        CompanyTenantRel existRel = companyTenantRelDao.findByTenantIdAndCompanyIdAndRelatedTenantId(tenantId, company.getCompanyId(), relatedTenant.getTenantId());
        if (existRel != null) {
            /**
             * 低频日志 日志不做处理
             */
            String message = "已经存在有一模一样的配置了, 不用重复提交";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }

        List<CompanyTenantRelOperation> existOperations = operationDao.findInvitionsByTenantIdAndCompanyIdAndRelatedTenantId(tenantId, company.getCompanyId(), relatedTenant.getTenantId());
        if (CollectionUtils.isNotEmpty(existOperations)) {
            String message = "已经存在有一模一样的邀请了, 不用重复提交";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }

        int switches = Switch.toValue(model.getSwitches());

        //region 尽力构造 CompanyTenantRelOperation
        CompanyTenantRelOperation operation = operationTrans(authorizedUser, company,relatedTenant);
        operation.setSwitches(switches);
        operation.setInviteRemark(model.getRemark());
        //邀请方公司id
        if (model.getInvitingCompanyId() != null && model.getInvitingCompanyId() > 0) {
            Company invitingCompany = orgDao.findCompanyByTenantIdAndCompanyId(tenantId, model.getInvitingCompanyId());
            if (invitingCompany == null) {
                String message = "找不到对应的所属租户主体公司(" + model.getInvitingCompanyId() + ")";
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
            operation.setInvitingCompanyId(invitingCompany.getCompanyId());
            operation.setInvitingCompanyName(invitingCompany.getCompanyName());
        }
        //endregion

        //region 校验共享历史发票和结算单起始日期
        this.fillSwitches(operation, switches, model.getInvoiceStartDate(), model.getStatementStartDate());
        //endregion

        //region 构造邀请信息
        OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
        orgQuery.setTenantId(relatedTenant.getTenantId());
        orgQuery.setOrgType(OrgType.COMPANY.name());
        orgQuery.setTaxNum(model.getInvitatedTaxNum());
        orgQuery.setCompanyNameEqual(model.getInvitatedCompanyName());
        List<OrgStruct> companyOrgs = orgDao.findAll(OrgQueryHelper.querySpecification(orgQuery), EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
        if (companyOrgs.isEmpty()) {
            String message = String.format("受邀方信息错误, 公司名称(" + model.getInvitatedCompanyName() + "), 税号(" + model.getInvitatedTaxNum() + ")");
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
        OrgStruct relatedOrg = companyOrgs.stream().findFirst().get();
        operation.setRelatedCompanyId(relatedOrg.getCompanyId());
        operation.setRelatedCompanyCode(relatedOrg.getCompanyCode());
        operation.setRelatedCompanyName(relatedOrg.getCompanyName());
        operation.setRelatedTaxNum(relatedOrg.getTaxNum());
        operation.setStatus(CompanyTenantRelOperationDto.Status.INITED.value());
        //endregion
        operation.setOperationType(CompanyTenantRelOperationDto.OPERATION_TYPE_INVITE);
        operation = operationDao.saveAndFlush(operation);
        if (company != null) {
            operation.setCompany(company);
        }
        return operation;
    }

    public Page<CompanyTenantRelOperation> page(Request.Query query, Pageable pageable) {
        Page<CompanyTenantRelOperation> page;
        boolean isTuple = false;
        boolean hasHostTenantId = false;
        boolean hasHostTenantCode = false;
        boolean hasHostTenantName = false;
        if (query.getWithExtendParamSet() != null && !query.getWithExtendParamSet().isEmpty()) {
            for (String withExtendParam : query.getWithExtendParamSet()) {
                if ("hostTenantId".equalsIgnoreCase(withExtendParam)) {
                    hasHostTenantId = true;
                    query.setHasHostTenantId(hasHostTenantId);
                } else if ("hostTenantCode".equalsIgnoreCase(withExtendParam)) {
                    hasHostTenantCode = true;
                    query.setHasHostTenantCode(hasHostTenantCode);
                } else if ("hostTenantName".equalsIgnoreCase(withExtendParam)) {
                    hasHostTenantName = true;
                    query.setHasHostTenantName(hasHostTenantName);
                }
            }
            isTuple = (hasHostTenantId | hasHostTenantCode | hasHostTenantName);
            query.setTuple(isTuple);
        }
        if (isTuple) {
            Page<Tuple> tuples = operationDao.findTuples(query, pageable);
            List<CompanyTenantRelOperation> contents = tuples.getContent().stream().map(tuple -> {
                CompanyTenantRelOperation operation = tuple.get("operation", CompanyTenantRelOperation.class);
                if (operation != null) {
                    if (query.isHasHostTenantId()) {
                        Long hostTenantId = tuple.get("hostTenantId", Long.class);
                        if (hostTenantId != null) {
                            operation.setHostTenantId(hostTenantId);
                        }
                    }
                    if (query.isHasHostTenantCode()) {
                        String hostTenantCode = tuple.get("hostTenantCode", String.class);
                        if (hostTenantCode != null) {
                            operation.setHostTenantCode(hostTenantCode);
                        }
                    }
                    if (query.isHasHostTenantName()) {
                        String hostTenantName = tuple.get("hostTenantName", String.class);
                        if (hostTenantName != null) {
                            operation.setHostTenantName(hostTenantName);
                        }
                    }
                }

                return operation;
            }).collect(Collectors.toList());
            page = new PageImpl<>(contents, pageable, tuples.getTotalElements());
        } else {
            Specification<CompanyTenantRelOperation> specification = CompanyTenantRelOperationQueryHelper.querySpecification(query);
            page = operationDao.findAll(specification, pageable);
        }
        return page;
    }

    public List<CompanyTenantRelOperation> list(Specification<CompanyTenantRelOperation> specification, Sort sort) {
        return operationDao.findAll(specification, sort);
    }

    public CompanyTenantRelOperation findById(long operationId, Set<String> withExtendParams) {
        boolean isTuple = false;
        boolean hasHostTenantId = false;
        boolean hasHostTenantCode = false;
        boolean hasHostTenantName = false;
        if (withExtendParams != null && !withExtendParams.isEmpty()) {
            for (String withExtendParam : withExtendParams) {
                if ("hostTenantId".equalsIgnoreCase(withExtendParam)) {
                    hasHostTenantId = true;
                } else if ("hostTenantCode".equalsIgnoreCase(withExtendParam)) {
                    hasHostTenantCode = true;
                } else if ("hostTenantName".equalsIgnoreCase(withExtendParam)) {
                    hasHostTenantName = true;
                }
            }
            isTuple = (hasHostTenantId | hasHostTenantCode | hasHostTenantName);
        }
        CompanyTenantRelOperation operation;
        if (isTuple) {
            Request.Query query = new Request.Query();
            query.setId(operationId);
            query.setTuple(isTuple);
            if (hasHostTenantId) {
                query.setHasHostTenantId(true);
            }
            if (hasHostTenantCode) {
                query.setHasHostTenantCode(true);
            }
            if (hasHostTenantName) {
                query.setHasHostTenantName(true);
            }
            Tuple tuple = operationDao.findTuple(query);
            operation = tuple.get("operation", CompanyTenantRelOperation.class);
            if (operation == null) {
                String message = "未找到公司租户关联关系(" + operationId + ")操作实体";
                logger.info(message);
                throw new IllegalArgumentException(message);
            } else {
                if (hasHostTenantId) {
                    Long hostTenantId = tuple.get("hostTenantId", Long.class);
                    if (hostTenantId != null) {
                        operation.setHostTenantId(hostTenantId);
                    }
                }
                if (hasHostTenantCode) {
                    String hostTenantCode = tuple.get("hostTenantCode", String.class);
                    if (hostTenantCode != null) {
                        operation.setHostTenantCode(hostTenantCode);
                    }
                }
                if (hasHostTenantName) {
                    String hostTenantName = tuple.get("hostTenantName", String.class);
                    if (hostTenantName != null) {
                        operation.setHostTenantName(hostTenantName);
                    }
                }
            }
        } else {
            operation = operationDao.findById(operationId).orElseThrow(() -> new IllegalArgumentException("未找到公司租户关联关系(" + operationId + ")操作实体"));
        }
        return operation;
    }

    /**
     * 审核取消
     *
     * @param operationId
     */
    @Transactional(rollbackFor = Exception.class)
    public void cancel(long operationId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        CompanyTenantRelOperation operation = this.findById(operationId, Collections.emptySet());
        if (operation.getStatus() == CompanyTenantRelOperationDto.Status.CANCELED.value()) {
            throw new IllegalArgumentException("当前邀请已经撤销了");
        }
        if (!authorizedUser.getTenantId().equals(operation.getCreaterTenantId())) {
            throw new IllegalArgumentException("当前登录用户所在租户和该邀请创建人所在租户不一致");
        }
        operation.setStatus(CompanyTenantRelOperationDto.Status.CANCELED.value());
        operationDao.saveAndFlush(operation);
    }

    private void fillSwitches(CompanyTenantRelOperation operation, int switches, Date invoiceStartDate, Date statementStartDate) {
        //region 校验共享历史发票起始日期
        if (BinaryUtils.is(Switch.INVOICE, switches)) {
            if (invoiceStartDate == null) {
                throw new IllegalArgumentException("开启包含历史发票必须填写共享历史发票起始日期");
            }
            operation.setInvoiceStartDate(invoiceStartDate);
        } else {
            operation.setInvoiceStartDate(null);
        }
        //endregion

        //region 校验共享历史结算单起始日期
        if (BinaryUtils.is(Switch.STATEMENT, switches)) {
            if (statementStartDate == null) {
                throw new IllegalArgumentException("开启包含历史结算单必须填写共享历史结算单起始日期");
            }
            operation.setStatementStartDate(statementStartDate);
        } else {
            operation.setStatementStartDate(null);
        }
        //endregion
    }

    public CompanyTenantRelOperation saveUnrelateOperation(Company company, CompanyTenantRel existRel) {
        if (existRel.getCompany() != null) {
            if (existRel.getCompany().getHostTenantId() != null && existRel.getCompany().getHostTenantId() > 0) {
                Tenant hostTenant = existRel.getCompany().getHostTenant();
                if (hostTenant == null || hostTenant.getCreateTime() == null) {
                    hostTenant = tenantDao.getById(existRel.getCompany().getHostTenantId());
                }
                if (hostTenant == null) {
                    existRel.fillHostTenant(hostTenant);
                }
            }
        }
        CompanyTenantRelOperation operation = this.operationTrans(company, existRel);
        operation.setOperationType(1);
        operation = operationDao.saveAndFlush(operation);
        return operation;
    }

    /**
     * operation 转换类
     * @param company
     * @param existRel
     * @return
     */
    private CompanyTenantRelOperation operationTrans(Company company,CompanyTenantRel existRel) {
        CompanyTenantRelOperation operation = new CompanyTenantRelOperation();
        operation.setInvitingCompanyId(existRel.getInvitingCompanyId());
        operation.setInvitingCompanyName(existRel.getInvitingCompanyName());
        operation.setTenantId(existRel.getTenantId());
        operation.setTenantCode(existRel.getTenantCode());
        operation.setTenantName(existRel.getTenantName());
        operation.setCompanyId(company.getCompanyId());
        operation.setHostTenantId(existRel.getHostTenantId());
        operation.setHostTenantName(existRel.getHostTenantName());
        operation.setHostTenantCode(existRel.getHostTenantCode());
        operation.setTaxNum(company.getTaxNum());
        operation.setCompanyName(company.getCompanyName());
        operation.setCompanyCode(company.getCompanyCode());
        operation.setSwitches(existRel.getSwitches());
        operation.setRelatedTenantId(existRel.getRelatedTenantId());
        operation.setRelatedTenantCode(existRel.getRelatedTenantCode());
        operation.setRelatedTenantName(existRel.getRelatedTenantName());
        operation.setRelatedCompanyId(existRel.getRelatedCompanyId());
        operation.setRelatedCompanyName(existRel.getRelatedCompanyName());
        operation.setRelatedTaxNum(existRel.getRelatedTaxNum());
        operation.setRelatedCompanyCode(existRel.getRelatedCompanyCode());
        return operation;
    }

    private CompanyTenantRelOperation operationTrans(IAuthorizedUser authorizedUser,Company company,Tenant relatedTenant) {
        CompanyTenantRelOperation operation = new CompanyTenantRelOperation();
        operation.setTenantId(authorizedUser.getTenantId());
        operation.setTenantCode(authorizedUser.getTenantCode());
        operation.setTenantName(authorizedUser.getTenantName());
        operation.setCompanyId(company.getCompanyId());
        operation.setTaxNum(company.getTaxNum());
        operation.setCompanyName(company.getCompanyName());
        operation.setCompanyCode(company.getCompanyCode());
        operation.setRelatedTenantId(relatedTenant.getTenantId());
        operation.setRelatedTenantCode(relatedTenant.getTenantCode());
        operation.setRelatedTenantName(relatedTenant.getTenantName());
        return operation;
    }
}
