package com.xforceplus.business.company.service;

import com.xforceplus.api.model.CompanyTenantRelModel.Request;
import com.xforceplus.api.model.CompanyTenantRelModel.Response;
import com.xforceplus.dao.*;
import com.xforceplus.domain.company.Switch;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.CompanyTenantRel;
import com.xforceplus.entity.CompanyTenantRelAudit;
import com.xforceplus.entity.Tenant;
import com.xforceplus.query.CompanyTenantRelQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * 公司租户关联关系 Service
 *
 * @author geewit
 */
@Service
public class CompanyTenantRelService {
    private final static Logger logger = LoggerFactory.getLogger(CompanyTenantRelService.class);

    private final CompanyTenantRelDao companyTenantRelDao;

    @Autowired
    private CompanyTenantRelAuditDao auditDao;
    @Autowired
    private CompanyTenantRelOperationDao operationDao;
    @Autowired
    private OrgStructDao orgDao;
    @Autowired
    private CompanyDao companyDao;
    @Autowired
    private TenantDao tenantDao;

    public CompanyTenantRelService(CompanyTenantRelDao companyTenantRelDao) {
        this.companyTenantRelDao = companyTenantRelDao;
    }

    public Page<CompanyTenantRel> page(Request.Query query, Pageable pageable) {
        Specification<CompanyTenantRel> specification = CompanyTenantRelQueryHelper.querySpecification(query);
        Page<CompanyTenantRel> page = companyTenantRelDao.findAll(specification, pageable, EntityGraphs.named(CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT));
        return page;
    }

    public List<CompanyTenantRel> list(Request.Query query, Sort sort) {
        Specification<CompanyTenantRel> specification = CompanyTenantRelQueryHelper.querySpecification(query);
        List<CompanyTenantRel> list = companyTenantRelDao.findAll(specification, sort, EntityGraphs.named(CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT));
        return list;
    }

    public List<CompanyTenantRel> list(Specification<CompanyTenantRel> specification, Sort sort) {
        return companyTenantRelDao.findAll(specification, sort, EntityGraphs.named(CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public CompanyTenantRel findById(long relId) {
        return companyTenantRelDao.findById(relId, EntityGraphs.named(CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT)).orElseThrow(() -> new IllegalArgumentException("未找到公司租户关联关系实体"));
    }

    public Tenant findHostTenantByCompanyId(long companyId) {
        Tenant hostTenant = companyDao.findHostTenantByCompanyId(companyId);
        return hostTenant;
    }

    /**
     * 根据公司租户关系id获取详情
     *
     * @param relId 公司租户关系id
     * @return 公司租户关系详情
     */
    public Response.Info info(long relId) {
        CompanyTenantRel rel = this.findById(relId);
        Company company = rel.getCompany();
        if (company != null && company.getHostTenantId() != null) {
            Optional<Tenant> hostTenantOptional = tenantDao.findById(company.getHostTenantId());
            if (hostTenantOptional.isPresent()) {
                rel.fillHostTenant(hostTenantOptional.get());
            }
        }

        Response.Info info = infoTrans(rel);
        int switches = rel.getSwitches() != null ? rel.getSwitches() : 0;
        int switchSize = BinaryUtils.fromBinary(switches, Switch.class).size();
        List<CompanyTenantRelAudit> audits = auditDao.findAuditsByRelId(rel.getId(), rel.getSwitches());
        Map<Switch, Response.History> historiesMap = new HashMap<>();
        for (CompanyTenantRelAudit audit : audits) {
            int auditSwitches = audit.getSwitches() == null ? 0 : audit.getSwitches();
            auditSwitches &= switches;
            if (auditSwitches == 0) {
                continue;
            }
            int switchAdditions = audit.getSwitchAdditions() == null ? 0 : audit.getSwitchAdditions();
            switchAdditions &= switches;
            boolean isOver = false;
            // 遍历开关枚举
            for (Switch switchMask : Switch.values()) {
                boolean res = loopSwitch(switchMask, switchAdditions, historiesMap, audit, auditSwitches, isOver, switchSize);
                if (res) {
                    continue;
                }else {
                    break;
                }
            }
            //region 判断记录是否已经设置全了, 如果是则后面的可以忽略了
            if (isOver) {
                break;
            }
            //endregion
        }
        List<Response.History> histories = new ArrayList<>(historiesMap.values());
        Collections.sort(histories, new Comparator<Response.History>() {
            @Override
            public int compare(Response.History h1, Response.History h2) {
                return h1.getOnSwitch().compareTo(h2.getOnSwitch());
            }
        });
        info.setHistories(histories);
        return info;
    }

    /**
     * 更新公司租户关联配置
     *
     * @param relId
     * @param model
     */
    @Transactional(rollbackFor = Exception.class)
    public void update(long relId, Request.Update model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        CompanyTenantRel existRel = this.findById(relId);
        Optional<Company> companyOptional = companyDao.findById(existRel.getCompanyId());
        Company company = null;
        if (!companyOptional.isPresent()) {
            throw new IllegalArgumentException("该公司(" + existRel.getCompanyId() + ")已经不存在了");
        } else {
            company = companyOptional.get();
            Long hostTenantId = company.getHostTenantId();
            if (hostTenantId != null && !hostTenantId.equals(authorizedUser.getTenantId())) {
                throw new IllegalArgumentException("该公司(" + existRel.getCompanyId() + ")的主租户已经不是您当前登录账户的租户了");
            }
        }
        int newSwitches = Switch.toValue(model.getSwitches());
        CompanyTenantRelAudit audit = this.fillRelAndAudit(existRel, newSwitches, existRel.getSwitches(), model.getInvoiceStartDate(), model.getStatementStartDate());
        existRel = companyTenantRelDao.saveAndFlush(existRel);
        //endregion

        //region 保存审计表
        audit.setRelId(existRel.getId());
        audit.setRemark(model.getRemark());
        auditDao.saveAndFlush(audit);
        //endregion
    }

    @Transactional(rollbackFor = Exception.class)
    public CompanyTenantRel save(long invitingCompanyId, String invitingCompanyName, long companyId, long tenantId, long relatedCompanyId, long relatedTenantId, int switches, int existSwitches, Date invoiceStartDate, Date statementStartDate, String remark) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        CompanyTenantRel rel = companyTenantRelDao.findByCompanyIdAndTenantIdAndRelatedTenantId(companyId, tenantId, relatedTenantId);
        if (rel == null) {
            rel = new CompanyTenantRel();
            rel.setInvitingCompanyId(invitingCompanyId);
            rel.setInvitingCompanyName(invitingCompanyName);
            rel.setCompanyId(companyId);
            rel.setTenantId(tenantId);
            rel.setRelatedCompanyId(relatedCompanyId);
            rel.setRelatedTenantId(relatedTenantId);
            rel.setSwitches(switches);
        }
        CompanyTenantRelAudit audit = this.fillRelAndAudit(rel, switches, existSwitches, invoiceStartDate, statementStartDate);
        rel = companyTenantRelDao.saveAndFlush(rel);
        //region 保存审计表
        audit.setRelId(rel.getId());
        audit.setRemark(remark);
        auditDao.saveAndFlush(audit);
        //endregion

        return rel;
    }

    public CompanyTenantRelAudit fillRelAndAudit(CompanyTenantRel rel, int switches, int existSwitches, Date invoiceStartDate, Date statementStartDate) {
        CompanyTenantRelAudit audit = new CompanyTenantRelAudit();
        int switchAdditions = 0;

        int switchDiffs = existSwitches ^ switches;
        //开关组合增加值
        switchAdditions = (switchDiffs & switches);
        rel.setSwitches(switches);
        //开关组合变化值

        Date now = null;
        //region 校验共享历史发票起始日期
        if (BinaryUtils.is(Switch.INVOICE, switches)) {
            if (invoiceStartDate == null) {
                throw new IllegalArgumentException("开启包含历史发票必须填写共享历史发票起始日期");
            }
            audit.setInvoiceStartDate(invoiceStartDate);
            if (rel.getInvoiceStartDate() == null || !DateUtils.isSameDay(invoiceStartDate, rel.getInvoiceStartDate())) {
                switchAdditions |= BinaryUtils.toBinary(Switch.INVOICE);
            }
            rel.setInvoiceStartDate(invoiceStartDate);
            if (now == null) {
                now = Calendar.getInstance().getTime();
            }
            rel.setInvoiceGrantedTime(now);
        } else {
            rel.setInvoiceStartDate(null);
        }
        //endregion

        //region 校验共享历史结算单起始日期
        if (BinaryUtils.is(Switch.STATEMENT, switches)) {
            if (statementStartDate == null) {
                throw new IllegalArgumentException("开启包含历史结算单必须填写共享历史结算单起始日期");
            }
            audit.setStatementStartDate(statementStartDate);
            if (rel.getStatementStartDate() == null || !DateUtils.isSameDay(statementStartDate, rel.getStatementStartDate())) {
                switchAdditions |= BinaryUtils.toBinary(Switch.STATEMENT);
            }
            rel.setStatementStartDate(statementStartDate);
            if (now == null) {
                now = Calendar.getInstance().getTime();
            }
            rel.setStatementGrantedTime(now);
        } else {
            rel.setStatementStartDate(null);
        }
        //endregion

        //region 构造 CompanyTenantRelAudit
        audit.setSwitches(switches);
        audit.setSwitchAdditions(switchAdditions);
        if (now == null) {
            now = Calendar.getInstance().getTime();
        }
        audit.setCreateTime(now);
        //endregion
        return audit;
    }

    private Response.Info infoTrans(CompanyTenantRel rel ) {
        Response.Info info = new Response.Info();
        info.setCompanyId(rel.getCompanyId());
        info.setCompanyName(rel.getCompanyName());
        info.setTaxNum(rel.getTaxNum());
        info.setTenantId(rel.getTenantId());
        info.setTenantName(rel.getTenantName());
        info.setHostTenantId(rel.getHostTenantId());
        info.setHostTenantName(rel.getHostTenantName());
        info.setRelatedCompanyId(rel.getRelatedCompanyId());
        info.setRelatedCompanyName(rel.getRelatedCompanyName());
        info.setRelatedTaxNum(rel.getRelatedTaxNum());
        info.setRelatedTenantId(rel.getRelatedTenantId());
        info.setRelatedTenantName(rel.getRelatedTenantName());
        return info;
    }
    private boolean loopSwitch(Switch switchMask,int switchAdditions,Map<Switch, Response.History> historiesMap,CompanyTenantRelAudit audit,int auditSwitches,boolean isOver,int switchSize) {
        //如果开关组合值的该枚举为 true
        if (BinaryUtils.is(switchMask, switchAdditions)) {
            //历史中已存在则忽略后面的逻辑
            if (historiesMap.containsKey(switchMask)) {
                return Boolean.TRUE;
            }
            Response.History history = new Response.History();
            history.setSwitches(auditSwitches);
            history.setOnSwitch(switchMask);
            //包含历史发票
            if (Switch.INVOICE.equals(switchMask)) {
                history.setStartDate(audit.getInvoiceStartDate());
            } else if (Switch.STATEMENT.equals(switchMask)) {
                //包含历史结算单
                history.setStartDate(audit.getStatementStartDate());
            }
            history.setGrantTime(audit.getCreateTime());
            history.setRemark(audit.getRemark());
            historiesMap.put(switchMask, history);
            isOver = historiesMap.size() == switchSize;
            if (isOver) {
                return Boolean.FALSE;
            }
        }
        return Boolean.TRUE;
    }
}
