package com.xforceplus.business.company.service.impl;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.enums.ActionTypeEnum;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.company.CompanyConfigDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.domain.company.CompanyConfigurationDto;
import com.xforceplus.utils.excel.exception.ImportException;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
public class CompanyConfigImportServiceImpl implements ExcelReaderService {

    @Autowired
    private CompanyService companyService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     *
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.COMPANY_CONFIG_EXCEL_FILE;
    }

    /**
     * 异步导出处理
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            //构建监听器
            AnalysisEventListener<CompanyConfigDTO> excelDataListener = SimpleDataReadListener
                    .listener(context, (rows) -> {
                        this.saveCompanyConfigData(messageRows, context, rows);
                    });
            //开始处理
            context.getSimpleExcelReader().read(CompanyConfigDTO.class, excelDataListener);
        }
        return context;
    }

    /**
     * 写入公司配置数据
     */
    public void saveCompanyConfigData(MessageRow messageRows, Context context,
                                      List<CompanyConfigDTO> list) {

        for (CompanyConfigDTO companyConfigDTO : list) {
            try {
                if (!companyConfigDTO.getValidatedStatus()) {
                    messageRows.fail(companyConfigDTO.getRowIndex(), companyConfigDTO.getValidatedMessage());
                    continue;
                }
                if (ActionTypeEnum.UPDATE.getActionCode().equals(companyConfigDTO.getAction())) {
                    CompanyConfigurationDto dto = new CompanyConfigurationDto();
                    BeanUtils.copyProperties(companyConfigDTO, dto);
                    companyService.saveConfigurationImport(dto);
                    messageRows.success(companyConfigDTO.getRowIndex());
                } else {
                    messageRows.fail(companyConfigDTO.getRowIndex(), "操作方式非法");
                }
            } catch (ImportException ex) {
                messageRows.fail(companyConfigDTO.getRowIndex(), ex.getMessage());
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

}
