/*
 * Copyright (c)  2015~2021, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyHisServiceImpl.java   2021-01-12 15-46-35
 * Author: Evan
 */
package com.xforceplus.business.company.service.impl;

import com.xforceplus.business.company.service.CompanyHistoryService;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.dao.CompanyDao;
import com.xforceplus.domain.company.CompanyHisDTO;
import com.xforceplus.entity.Company;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * <p>
 * Title:  公司履历 Service实现层
 * </p>
 * <p>
 * Description:  公司履历 Service实现层
 * </p>
 * <p>
 * Copyright: 2015~2021
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2021-01-12 15-46-35
 * @since V1.0
 */
@Service
public class CompanyHistoryServiceImpl implements CompanyHistoryService {

    private final CompanyDao companyDao;
    /**
     * 公司服务
     */
    private final CompanyService companyService;

    public CompanyHistoryServiceImpl(CompanyDao companyDao, CompanyService companyService) {
        this.companyDao = companyDao;
        this.companyService = companyService;
    }

    /**
     * 根据公司履历的税号查询当前公司信息
     *
     * @param taxNum 税号
     * @return Optional<Company>
     */
    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    @Override
    public Optional<Company> findByHisTaxNum(String taxNum) {
        return companyDao.findCompanyHistoryByTaxNum(taxNum);
    }

    /**
     * 分页查询
     *
     * @param taxNum   税号
     * @param pageable 分页查询
     * @return Page<CompanyHisDTO>
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    public Page<CompanyHisDTO> pageHistory(String taxNum, Pageable pageable) {

        Page<CompanyHisDTO> pageHistory = companyDao.pageHistoryByTaxNum(taxNum, pageable);
        Set<Long> companyIds = pageHistory.getContent()
                .stream()
                .map(CompanyHisDTO::getCompanyId)
                .collect(Collectors.toSet());
        //在关系当前公司信息
        List<Company> companyList = companyService.findByIds(new ArrayList<>(companyIds));
        //转换
        Map<Long, Company> companyMap = companyList.stream()
                .collect(Collectors.toMap(Company::getCompanyId, Function.identity(), (key1, key2) -> key1));
        pageHistory = pageHistory.map(e -> {
            if (!companyMap.containsKey(e.getCompanyId())) {
                return e;
            }
            Company company = companyMap.get(e.getCompanyId());
            //公司名称
            if (StringUtils.isEmpty(e.getCompanyName()) && StringUtils.isNotEmpty(company.getCompanyName())) {
                e.setCompanyName(e.getCompanyName());
            }
            //公司税号处理
            if (StringUtils.isEmpty(e.getTaxNum()) && StringUtils.isNotEmpty(company.getTaxNum())) {
                e.setStatus(company.getStatus());
            }
            //公司代码
            if (StringUtils.isNotEmpty(company.getCompanyCode())) {
                e.setCompanyCode(company.getCompanyCode());
            }
            //公司状态
            if (null != company.getStatus()) {
                e.setStatus(company.getStatus());
            }
            //更新时间
            if (null != company.getUpdateTime()) {
                e.setUpdateTime(company.getUpdateTime());
            }
            return e;
        });
        return pageHistory;
    }
}
