/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyImportServiceImpl.java   2020-09-24 09-31-40
 * Author: Evan
 */
package com.xforceplus.business.company.service.impl;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.enums.ActionTypeEnum;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.company.CompanyExcelImportDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.FileController;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.entity.Company;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.List;

/**
 * <p>
 * Title: CompanyImportServiceImpl
 * </p>
 * <p>
 * Description: CompanyImportService实现类
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-24 09-31-40
 * @since V1.0
 */
@Service
public class CompanyImportServiceImpl implements ExcelReaderService {
    /**
     * 新增
     */
    public static final String ACTION_ADD =ActionTypeEnum.ADD.getActionCode();

    /**
     * 修改
     */
    public static final String ACTION_UPDATE = ActionTypeEnum.MODIFY.getActionCode();

    /**
     * 操作列表
     */
    public static final List<String> ACTION_LIST = Arrays.asList(ACTION_ADD, ACTION_UPDATE);
    /**
     * 日志
     */
    private final Logger log = LoggerFactory.getLogger(FileController.class);

    /**
     * 日志
     */
    private final TenantDao tenantDao;

    /**
     * 保存数据
     */
    private final OrgService orgService;
    /**
     * 公司信息
     */
    private final CompanyService companyService;

    /**
     * 默认构建函数
     * @param tenantDao  tenantDao
     * @param orgService orgService
     * @param companyService companyService
     */
    public CompanyImportServiceImpl(TenantDao tenantDao, OrgService orgService, CompanyService companyService) {
        this.tenantDao = tenantDao;
        this.orgService = orgService;
        this.companyService = companyService;
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.COMPANY_EXCEL_FILE;
    }

    /**
     * 导入定义
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            //构建监听器
            AnalysisEventListener<CompanyExcelImportDTO> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> this.saveImport(messageRows, context, rows));
            //开始处理
            context.getSimpleExcelReader().read(CompanyExcelImportDTO.class, excelDataListener);
        }
        return context;
    }

    /**
     * 写入数据
     *
     * @param messageRow MessageRow
     * @param context    上下文
     * @param list       数据列表
     */
    public void saveImport(MessageRow messageRow, Context context, List<CompanyExcelImportDTO> list) {
        //为空则不处理
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (CompanyExcelImportDTO dto : list) {

            if (!dto.getValidatedStatus()) {
                messageRow.fail(dto.getRowIndex(), dto.getValidatedMessage());
                continue;
            }
            //检查操作类型
            if (!ACTION_LIST.contains(dto.getAction())) {
                messageRow.fail(dto.getRowIndex(), "操作类型不正确，操作类型应该为:{" + ACTION_ADD + "," + ACTION_UPDATE + "}");
                continue;
            }
            //检查tenantCode是否存在，如果存在的话，则获取tenantId;
            if (!this.isExistedTenantByTenantCode(dto)) {
                messageRow.fail(dto.getRowIndex(), "租户code(" + dto.getTenantCode() + ")找不到对应租户");
                log.warn("sheetName:{},rowIndex:{},租户code:{}:找不到对应租户", messageRow.getSheetName(), dto.getRowIndex(), dto.getTenantCode());
                continue;
            }
            //判断如果新增时，不能有重复添加,如果存在则结束本次操作
            if (ACTION_ADD.equals(dto.getAction()) && this.isExistedCompanyByTaxNum(dto)) {
                messageRow.fail(dto.getRowIndex(), "已存公司税号：(" + dto.getTaxNum() + ")的公司");
                log.warn("sheetName:{},rowIndex:{},已存在个公司名称为:({}),税号({})的公司", messageRow.getSheetName(), dto.getRowIndex(), dto.getCompanyName(), dto.getTaxNum());
                continue;
            }

            //判断如果新增时，不能有重复添加,如果存在则结束本次操作
            if (ACTION_UPDATE.equals(dto.getAction()) && !this.isExistedCompanyByTaxNum(dto)) {
                messageRow.fail(dto.getRowIndex(), "未找到更新公司税号(" + dto.getTaxNum() + ")的公司");
                log.warn("sheetName:{},rowIndex:{},未找到更新公司名称为：({}),税号({})的公司", messageRow.getSheetName(), dto.getRowIndex(), dto.getCompanyName(), dto.getTaxNum());
                continue;
            }
            CompanyModel.Request.Save model = new CompanyModel.Request.Save();
            //复制数据
            BeanUtils.copyProperties(dto, model);
            //保存数据或更新数据
            Company company;
            try {
                company = orgService.saveCompanyAsync(dto.getTenantId(), null, model);
            } catch (RuntimeException e) {
                log.warn("sheetName:{},保存数据失败:tenantCode:{},companyName:{}", messageRow.getSheetName(), dto.getTenantCode(), dto.getCompanyName());
                log.error(e.getMessage(), e);
                messageRow.fail(dto.getRowIndex(), dto.getAction() + "数据异常");
                continue;
            }
            //如果更新数据则失败；
            if (company == null) {
                messageRow.fail(dto.getRowIndex(), dto.getAction() + "数据失败");
                log.warn("sheetName:{},保存数据失败:tenantCode:{},companyName:{}", messageRow.getSheetName(), dto.getTenantCode(), dto.getCompanyName());
                continue;
            }
            messageRow.success(dto.getRowIndex());
        }
        context.messageRow(messageRow.getSheetName(), messageRow);
    }


    /**
     * 检查租户code是否存在，找到对应tenantCode
     * @param dto CompanyExcelImportDTO
     * @return True 已经存在，false 不存在
     */
    protected Boolean isExistedTenantByTenantCode(CompanyExcelImportDTO dto) {
        Long tenantId = tenantDao.findTenantIdByTenantCode(dto.getTenantCode());
        if (tenantId == null) {
            return Boolean.FALSE;
        }
        //设置租户ID
        dto.setTenantId(tenantId);
        return Boolean.TRUE;
    }

    /**
     * 新增情况校验是否已经存在,如果存在则回写错误信息 (只判断公司税号)
     *
     * @param dto 新增对象
     * @return Boolean true 已存在，false 则不存在
     */
    protected Boolean isExistedCompanyByTaxNum(CompanyExcelImportDTO dto) {
        List<Company> companies = this.companyService.findListByTaxNum(dto.getTaxNum());
        //如果不存在就直接返回
        if (CollectionUtils.isEmpty(companies)) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }
}