package com.xforceplus.business.company.service.impl;

import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.fastjson.JSON;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.company.CompanyUpdateImportDto;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.entity.Company;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;


/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author dhy
 * <b>Creation Time:</b> 2020-09-24 14-03-25
 * @since V1.0
 */
@Service
public class CompanyUpdateImportServiceImpl implements ExcelReaderService {
    /**
     * 公司名称
     */
    public static final String ACTION_COMPANY_NAME = "公司名称";
    /**
     * 统一社会信用代码
     */
    public static final String ACTION_TAX_NUM = "统一社会信用代码";
    /**
     * 操作列表
     */
    public static final List<String> ACTION_LISTS = Arrays.asList(ACTION_COMPANY_NAME, ACTION_TAX_NUM);
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(CompanyUpdateImportServiceImpl.class);
    @Resource
    private CompanyService companyService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.COMPANY_NAME_TAX_NUM_IMPORT;
    }

    /**
     * 异步导出处理
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            //构建监听器
            AnalysisEventListener<CompanyUpdateImportDto> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                this.saveData(messageRows, context, rows);
            });
            //开始处理
            context.getSimpleExcelReader().read(CompanyUpdateImportDto.class, excelDataListener);
        }
        return context;
    }

    /**
     * 写入数据
     *
     * @param list
     */
    public void saveData(MessageRow messageRows, Context context, List<CompanyUpdateImportDto> list) {
        logger.info("list:{}", JSON.toJSONString(list));

        for (CompanyUpdateImportDto e : list) {
            logger.info("e:{}", JSON.toJSONString(e));
            if (!e.getValidatedStatus()) {
                messageRows.fail(e.getRowIndex(), e.getValidatedMessage());
                continue;
            }
            try {
                if (ACTION_COMPANY_NAME.equals(e.getAction())) {
                    boolean result = this.updateCompanyNameByTaxNum(messageRows,   e);
                    if (!result) {
                        continue;
                    }
                } else {
                    //更新税号
                    boolean result = this.updateCompanyTaxNumByTaxNum(messageRows,   e);
                    if (!result) {
                        continue;
                    }
                }
            } catch (Exception ex) {
                messageRows.fail(e.getRowIndex(), e.getAction() + "失败");
                continue;
            }

            messageRows.success(e.getRowIndex());
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 根据原来taxNum
     *
     * @param messageRows 消息
     * @param context     上下文
     * @param dto         CompanyUpdateImportDto
     */
    private boolean updateCompanyTaxNumByTaxNum(MessageRow messageRows,  CompanyUpdateImportDto dto) {
        String taxNum = StringUtils.trimToEmpty(dto.getTaxNum());
        String oldTaxNum = StringUtils.trimToEmpty(dto.getOldTaxNum());

        if (StringUtils.isBlank(taxNum)) {
            messageRows.fail(dto.getRowIndex(), "更新后统一社会信用代码不能为空");
            return false;
        }
        if (taxNum.equals(oldTaxNum)) {
            messageRows.fail(dto.getRowIndex(), "原统一社会信用代码与更新后统一社会信用代码相等，不能更新");
            return false;
        }
        Optional<Company> optional = this.findByTaxNum(oldTaxNum);
        if (!optional.isPresent()) {
            messageRows.fail(dto.getRowIndex(), "公司统一社会信用代码为：【" + oldTaxNum + "】对应公司不存在");
            return false;
        }
        String companyName = StringUtils.trimToEmpty(dto.getCompanyName());
        Company company = optional.get();
        company.setTaxNum(taxNum);
        companyService.saveAndFlush(company);
        return true;
    }

    /**
     * 根据税号更新公司名称
     *
     * @param messageRows            消息
     * @param context                上下文件
     * @param companyUpdateImportDto
     */
    private boolean updateCompanyNameByTaxNum(MessageRow messageRows,   CompanyUpdateImportDto dto) {
        if (StringUtils.isBlank(dto.getOldCompanyName())) {
            messageRows.fail(dto.getRowIndex(), "原公司名称不能为空");
            return false;
        }
        if (StringUtils.isBlank(dto.getCompanyName())) {
            messageRows.fail(dto.getRowIndex(), "更新后公司名称不能为空");
            return false;
        }
        if (dto.getOldCompanyName().equals(dto.getCompanyName())) {
            messageRows.fail(dto.getRowIndex(), "原公司名称与更新后公司名称相等，不能更新");
            return false;
        }
        String taxNum = StringUtils.trimToEmpty(dto.getOldTaxNum());
        Optional<Company> optional = this.findByTaxNum(taxNum);
        if (!optional.isPresent()) {
            messageRows.fail(dto.getRowIndex(), "公司统一社会信用代码为：【" + taxNum + "】对应公司不存在");
            return false;
        }
        String companyName = StringUtils.trimToEmpty(dto.getCompanyName());
        Company company = optional.get();
        company.setCompanyName(companyName);
        companyService.saveAndFlush(company);
        return true;
    }

    /**
     * 根据税号查找公司
     *
     * @param taxNum 税号
     * @return Optional<Company>
     */
    private Optional<Company> findByTaxNum(String taxNum) {
        List<Company> companies = this.companyService.findListByTaxNum(taxNum);
        if (CollectionUtils.isEmpty(companies)) {
            return Optional.empty();
        }
        //公司税号为公司唯一索引
        return Optional.of(companies.get(0));
    }

}
