package com.xforceplus.business.conf.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.Uri;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.PrivacyAgreementModel.Request.*;
import com.xforceplus.business.conf.service.PrivacyAgreementService;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.PrivacyAgreement;
import io.geewit.core.jackson.view.View;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;


/**
 * @author HWT
 * @since  2022/1/6 16:41
 */
@Validated
@RestController
public class PrivacyAgreementController implements Uri {

    private final PrivacyAgreementService privacyAgreementService;

    public PrivacyAgreementController(PrivacyAgreementService privacyAgreementService) {
        this.privacyAgreementService = privacyAgreementService;
    }

    @ApiOperation("分页查询指定隐私协议")
    @GetMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements")
    public ResponseEntity<Page<PrivacyAgreement>> page(@SpringQueryMap Query query, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<PrivacyAgreement> page = privacyAgreementService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @ApiOperation("查询指定隐私协议")
    @GetMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/{id}")
    public ResponseEntity<PrivacyAgreement> getPrivacyAgreement(@PathVariable("id") long id) {
        PrivacyAgreement privacyAgreement = privacyAgreementService.get(id);
        privacyAgreement.sortProtocols();
        return ResponseEntity.ok(privacyAgreement);
    }

    @ApiOperation("新增隐私协议")
    @PostMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements")
    public ResponseEntity<PrivacyAgreement> addPrivacyAgreement(@RequestBody @Validated(ValidationGroup.OnCreate.class) Save model) {
        PrivacyAgreement privacyAgreement = privacyAgreementService.add(model);
        return ResponseEntity.ok(privacyAgreement);
    }

    @ApiOperation("修改隐私协议")
    @PutMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/{id}")
    public ResponseEntity<PrivacyAgreement> updatePrivacyAgreement(@PathVariable("id") long id,
                                                                   @RequestBody @Validated(ValidationGroup.OnUpdate.class) Save model) {
        model.setId(id);
        PrivacyAgreement result = privacyAgreementService.update(model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation("启用隐私协议")
    @PutMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/{id}/enable")
    public ResponseEntity<PrivacyAgreement> enable(@PathVariable("id") long id) {

        PrivacyAgreement result = privacyAgreementService.enable(id);
        return ResponseEntity.ok(result);
    }

    @ApiOperation("禁用隐私协议")
    @PutMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/{id}/disable")
    public ResponseEntity<PrivacyAgreement> disable(@PathVariable("id") long id) {
        PrivacyAgreement result = privacyAgreementService.disable(id);
        return ResponseEntity.ok(result);
    }

    @ApiOperation("刪除隐私协议")
    @DeleteMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/{id}")
    public ResponseEntity<String> deletePrivacyAgreement(@PathVariable("id") long id) {
        privacyAgreementService.delete(id);
        return ResponseEntity.ok();
    }

    @ApiOperation("查询最新版本隐私协议")
    @JsonView(View.class)
    @GetMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/latest")
    public ResponseEntity<PrivacyAgreement> getLatestPrivacyAgreement(@RequestParam( value = "companyType", required = false, defaultValue = "1")  Integer companyType) {
        PrivacyAgreement privacyAgreement = privacyAgreementService.getLatest(companyType);
        return ResponseEntity.ok(privacyAgreement);
    }

    @ApiOperation("提升隐私协议详情序号")
    @PutMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/protocols/{protocolId}/sort-up")
    public ResponseEntity<String> sortUpProtocols(@PathVariable("protocolId") long protocolId) {
        privacyAgreementService.sortProtocols(protocolId, Sort.Direction.ASC);
        return ResponseEntity.ok();
    }


    @ApiOperation("降低隐私协议详情序号")
    @PutMapping(PATH_GLOBAL_PREFIX + "/privacy-agreements/protocols/{protocolId}/sort-down")
    public ResponseEntity<String> sortDownProtocols(@PathVariable("protocolId") long protocolId) {
        privacyAgreementService.sortProtocols(protocolId, Sort.Direction.DESC);
        return ResponseEntity.ok();
    }
}
