package com.xforceplus.business.conf.service;

import com.xforceplus.api.model.PrivacyAgreementModel.Request.ProtocolSave;
import com.xforceplus.api.model.PrivacyAgreementModel.Request.Query;
import com.xforceplus.api.model.PrivacyAgreementModel.Request.Save;
import com.xforceplus.dao.PrivacyAgreementDao;
import com.xforceplus.dao.PrivacyAgreementProtocolDao;
import com.xforceplus.domain.conf.PrivacyAgreementDto;
import com.xforceplus.entity.PrivacyAgreement;
import com.xforceplus.entity.PrivacyAgreementProtocol;
import com.xforceplus.query.PrivacyAgreementQueryHelper;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author HWT
 * @since 2022/1/4 14:37
 */
@Slf4j
@Service
public class PrivacyAgreementService {
    private static final Integer ENABLE = 1;
    private static final Integer DISABLE = 0;

    private final PrivacyAgreementDao privacyAgreementDao;

    private final PrivacyAgreementProtocolDao privacyAgreementProtocolDao;

    public PrivacyAgreementService(PrivacyAgreementDao privacyAgreementDao, PrivacyAgreementProtocolDao privacyAgreementProtocolDao) {
        this.privacyAgreementDao = privacyAgreementDao;
        this.privacyAgreementProtocolDao = privacyAgreementProtocolDao;
    }

    public Page<PrivacyAgreement> page(Query query, Pageable pageable) {
        return privacyAgreementDao.findAll(PrivacyAgreementQueryHelper.querySpecification(query), pageable);
    }

    @Transactional(rollbackFor = Exception.class)
    public PrivacyAgreement add(Save model) {
        Query query = new Query();
        query.setVersion(model.getVersion());
        List<PrivacyAgreement> privacyAgreements = privacyAgreementDao.findAll(PrivacyAgreementQueryHelper.querySpecification(query), EntityGraphs.named(PrivacyAgreement.NAMED_ENTITY_GRAPH_DEFAULT));

        Optional<PrivacyAgreement> existPrivacyAgreementOptional = privacyAgreements.stream().filter(Objects::nonNull).findFirst();
        if (existPrivacyAgreementOptional.isPresent()) {
            throw new IllegalArgumentException("已存在版本为:" + model.getVersion() + "的隐私协议");
        } else {
            PrivacyAgreement privacyAgreement = new PrivacyAgreement();
            privacyAgreement.setContent(model.getContent());
            privacyAgreement.setVersion(model.getVersion());
            privacyAgreement.setCompanyType(ObjectUtils.defaultIfNull(model.getCompanyType(), 1));
            privacyAgreement.setStatus(ObjectUtils.defaultIfNull(model.getStatus(), 1));
            privacyAgreement = privacyAgreementDao.saveAndFlush(privacyAgreement);
            if (model.getProtocols() != null && !model.getProtocols().isEmpty()) {
                final Long privacyAgreementId = privacyAgreement.getId();
                List<PrivacyAgreementProtocol> protocols = model.getProtocols().stream().map(protocolModel -> {
                    PrivacyAgreementProtocol agreementProtocol = new PrivacyAgreementProtocol();
                    agreementProtocol.setAgreementId(privacyAgreementId);
                    agreementProtocol.setTitle(protocolModel.getTitle());
                    agreementProtocol.setUrl(protocolModel.getUrl());
                    return agreementProtocol;
                }).collect(Collectors.toList());

                privacyAgreement.setProtocols(protocols);
                privacyAgreement.arrangeProtocolsOrdinal();
                privacyAgreementProtocolDao.saveAllAndFlush(protocols);
            }

            return privacyAgreement;
        }
    }

    public PrivacyAgreement get(long id) {
        return privacyAgreementDao.findById(id, EntityGraphs.named(PrivacyAgreement.NAMED_ENTITY_GRAPH_DEFAULT))
                .orElseThrow(
                        () -> new IllegalArgumentException("无法找到该信息(" + id + ")"));
    }

    @Transactional(rollbackFor = Exception.class)
    public PrivacyAgreement update(Save model) {
        PrivacyAgreement existEntity =
                privacyAgreementDao
                        .findById(model.getId(), EntityGraphs.named(PrivacyAgreement.NAMED_ENTITY_GRAPH_DEFAULT))
                        .orElseThrow(
                                () -> new IllegalArgumentException("无法找到该信息(" + model.getId() + ")"));
        if (StringUtils.isNotBlank(model.getVersion()) && !model.getVersion().equals(existEntity.getVersion())) {
            Query query = new Query();
            query.setVersion(model.getVersion());
            long count = privacyAgreementDao.count(PrivacyAgreementQueryHelper.querySpecification(query));
            if (count > 0) {
                throw new IllegalArgumentException("该version(" + model.getVersion() + ")的隐私协议已存在");
            }
            existEntity.setVersion(model.getVersion());
        }
        if (StringUtils.isNotBlank(model.getContent())) {
            existEntity.setContent(model.getContent());
        }
        if (model.getCompanyType()!=null && model.getCompanyType()>0) {
            existEntity.setCompanyType(model.getCompanyType());
        }
        if (model.getStatus() != null) {
            existEntity.setStatus(model.getStatus());
        }
        existEntity = privacyAgreementDao.saveAndFlush(existEntity);
        if (model.getProtocols() != null && !model.getProtocols().isEmpty()) {
            privacyAgreementProtocolDao.deleteByAgreementId(existEntity.getId());
            List<PrivacyAgreementProtocol> protocols = new ArrayList<>(model.getProtocols().size());

            List<ProtocolSave> protocolModels = model.getProtocols();
            for (ProtocolSave protocolModel : protocolModels) {
                PrivacyAgreementProtocol agreementProtocol = new PrivacyAgreementProtocol();
                agreementProtocol.setAgreementId(existEntity.getId());
                agreementProtocol.setTitle(protocolModel.getTitle());
                agreementProtocol.setUrl(protocolModel.getUrl());
                protocols.add(agreementProtocol);
            }
            existEntity.setProtocols(protocols);
            existEntity.arrangeProtocolsOrdinal();
            privacyAgreementProtocolDao.saveAllAndFlush(protocols);
        }
        return existEntity;
    }

    @Transactional(rollbackFor = Exception.class)
    public PrivacyAgreement enable(long id) {
        PrivacyAgreement existEntity =
                privacyAgreementDao
                        .findById(id)
                        .orElseThrow(
                                () -> new IllegalArgumentException("无法找到该信息(" + id + ")"));

        existEntity.setStatus(ENABLE);
        return privacyAgreementDao.saveAndFlush(existEntity);
    }

    @Transactional(rollbackFor = Exception.class)
    public PrivacyAgreement disable(Long id) {
        PrivacyAgreement existEntity =
                privacyAgreementDao
                        .findById(id)
                        .orElseThrow(
                                () -> new IllegalArgumentException("无法找到该信息(" + id + ")"));

        existEntity.setStatus(DISABLE);
        return privacyAgreementDao.saveAndFlush(existEntity);
    }

    @Transactional(rollbackFor = Exception.class)
    public void delete(Long id) {
        PrivacyAgreement existEntity =
                privacyAgreementDao
                        .findById(id)
                        .orElseThrow(() -> new IllegalArgumentException("无法找到该信息(" + id + ")"));
        privacyAgreementDao.deleteById(existEntity.getId());
        privacyAgreementProtocolDao.deleteByAgreementId(existEntity.getId());
    }

    public PrivacyAgreement getLatest(Integer companyType) {
        Optional<PrivacyAgreement> privacyAgreementOptional = privacyAgreementDao.findFirstByStatusAndCompanyTypeOrderByCreateTimeDesc(1, companyType);
        if (privacyAgreementOptional.isPresent()) {
            PrivacyAgreement privacyAgreement = privacyAgreementOptional.get();
            List<PrivacyAgreementProtocol> protocols = privacyAgreementProtocolDao.findByAgreementId(privacyAgreement.getId());
            privacyAgreement.setProtocols(protocols);
            privacyAgreement.sortProtocols();
            return privacyAgreement;
        } else {
            throw new IllegalArgumentException("沒有找到有效的信息！");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void sortProtocols(long protocolId, Sort.Direction direction) {
        if (direction == null) {
            return;
        }
        Long agreementId = privacyAgreementProtocolDao.findAgreementIdById(protocolId);
        if (agreementId == null) {
            throw new IllegalArgumentException("隐私协议详情(" + protocolId + ")不存在agreementId的");
        }
        List<PrivacyAgreementProtocol> protocols = privacyAgreementProtocolDao.findByAgreementId(agreementId);
        PrivacyAgreementProtocol protocol = protocols.stream().filter(p -> p.getId().equals(protocolId)).findFirst().orElseThrow(() -> new IllegalArgumentException("不存在id(" + protocolId + ")的隐私协议详情"));
        PrivacyAgreementDto.sortProtocols(protocols);
        PrivacyAgreementDto.arrangeProtocolsOrdinal(protocols);
        int ordinal = ObjectUtils.defaultIfNull(protocol.getOrdinal(), 1);
        if (ordinal >= protocols.size()) {
            privacyAgreementProtocolDao.updateOrdinalById(protocol.getId(), ordinal);
        } else {
            final int anotherOrdinal;
            if (direction.isAscending()) {
                anotherOrdinal = ordinal - 1;
            } else {
                anotherOrdinal = ordinal + 1;
            }

            if (anotherOrdinal > 0 && anotherOrdinal <= protocols.size()) {
                privacyAgreementProtocolDao.updateOrdinalById(protocol.getId(), anotherOrdinal);
            }

            protocols.stream().filter(p -> p.getOrdinal().equals(anotherOrdinal))
                    .findFirst().ifPresent(anotherProtocol -> privacyAgreementProtocolDao.updateOrdinalById(anotherProtocol.getId(), ordinal));
        }
    }
}

