/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: AbstractExcelProcess.java   2020-10-15 13-51-08
 * Author: Evan
 */
package com.xforceplus.business.excel;

import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.dao.ExcelFileStoreDao;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.Optional;

/**
 * <p>
 * Title: 抽像Excel处理类
 * </p>
 * <p>
 * Description:  抽像Excel处理类
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-15 13-51-08
 * @since V1.0
 */
public class AbstractExcelProcess {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(AbstractExcelProcess.class);
    /**
     * 导入文件
     */
    protected ExcelFileStoreDao excelFileStoreDao;

    public AbstractExcelProcess(ExcelFileStoreDao excelFileStoreDao) {
        this.excelFileStoreDao = excelFileStoreDao;
    }

    /**
     * 设置
     *
     * @param context
     */
    protected void beforeSet(Context context) {
        UserInfoHolder.put(context.getAuthorizedUser());
    }

    /**
     * 保存数据
     *
     * @param fileDTO
     */
    protected void beforeUpdate(ExcelFileDTO fileDTO) {
        Optional<ExcelFileStore> importFileOptional = this.excelFileStoreDao.findById(fileDTO.getId());
        if (!importFileOptional.isPresent()) {
            log.warn("没有找到数据:ImportFileDTO-id:{}", fileDTO.getId());
            return;
        }
        ExcelFileStore excelFileStore = importFileOptional.get();
        excelFileStore.setBeginDate(fileDTO.getBeginDate());
        excelFileStore.setStatus(fileDTO.getStatus());
        excelFileStore.setReadTime(fileDTO.getReadTime());
        this.excelFileStoreDao.saveAndFlush(excelFileStore);
    }


    /**
     * 完成后更新数据
     * @param fileDTO ImportFileDTO
     */
    @Transactional(rollbackFor = Exception.class)
    public void afterUpdate(ExcelFileDTO fileDTO) {
        Optional<ExcelFileStore> importFileOptional = this.excelFileStoreDao.findById(fileDTO.getId());
        if (!importFileOptional.isPresent()) {
            log.warn("没有找到数据:ImportFileDTO-id:{}", fileDTO.getId());
            return;
        }
        ExcelFileStore excelFileStore = importFileOptional.get();
        excelFileStore.setStatus(fileDTO.getStatus());
        excelFileStore.setEndDate(fileDTO.getEndDate());
        excelFileStore.setTargetFileId(fileDTO.getTargetFileId());
        excelFileStore.setEndDate(new Date());
        excelFileStore.setCostTime(fileDTO.getCostTime());
        excelFileStore.setTotalSize(fileDTO.getTotalSize());
        excelFileStore.setSuccessSize(fileDTO.getSuccessSize());
        //保存每个Sheet导入成功的数据
        excelFileStore.setResultState(fileDTO.getResultState());
        excelFileStore.setStatus(ExcelFileDTO.Status.FINISHED);
        if (StringUtils.isNotBlank(fileDTO.getTargetFileName())) {
            //文件名
            excelFileStore.setTargetFileName(fileDTO.getTargetFileName());
        }
        //处理多个sheet导出的数据
        excelFileStore.setMessage(fileDTO.getMessage());
        this.excelFileStoreDao.saveAndFlush(excelFileStore);
    }

    /**
     * 计算两个时间运行时间
     *
     * @param beginDate 开始时间
     * @param endDate   结束时间
     * @return Long costTime
     */
    protected Long costTime(Date beginDate, Date endDate) {
        Long costTime = 0L;
        if (beginDate != null && endDate != null) {
            costTime = endDate.getTime() - beginDate.getTime();
        }
        return costTime;
    }

    /**
     * 清除数据
     */
    protected void clear() {
        UserInfoHolder.clearContext();
    }
}
