/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelExportBo.java   2020-09-17 13-33-18
 * Author: Evan
 */
package com.xforceplus.business.excel;

import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * Title: Excel工作薄
 * </p>
 * <p>
 * Description: Excel工作薄 （定义模板文件名，模板名文件路径，输出文件名和文件）
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-17 13-33-18
 * @since V1.0
 */
@Setter
@Getter
@ToString
public class ExcelBook {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(ExcelBook.class);

    /**
     * 业务名称或工作薄名称
     */
    private String name;
    /**
     * 文件扩展名称
     */
    private String fileNameExt;
    /**
     * 模板名称
     */
    private String sourceFileName;
    /**
     * 模板路径
     */
    private String sourceFilePath;

    /**
     * 文件输出名
     */
    private String targetFileName;
    /**
     * 文件输出名
     */
    private String targetFilePath;

    /**
     * 工作表
     */
    private List<ExcelSheet> excelSheets;

    /**
     * 消息内容
     */
    private Map<String, String> messages;
    /**
     * 文件ID S3上传后的文件ID
     */
    private Long targetFileId;
    /**
     * 目标文件，用于下载文件
     */
    private Long sourceFileId;

    /**
     * 异常信息 (异常)
     */
    private Exception exception;
    /**
     * 租户ID
     */
    private Long tenantId;

    /**
     * 用户ID (登录用户)
     */
    private Long userId;

    /**
     * 默认启动Debug等false
     */
    private Boolean debug;
    /**
     * 参数内容
     */
    private Map<String, Object> params;

    /**
     * 默认构建函数
     * @param fileNameExt 文件扩展名
     * @param templateFileName 模板文件
     * @param templatePath 模板路径
     * @param fileName 输出文件
     * @param filePath 输出文件路径
     * @param excelSheets ExcelSheet
     * @param messages Message
     * @param fileId  fileId输出
     * @param tenantId  租户ID
     * @param userId 用户ID
     */
    protected ExcelBook(String fileNameExt,
                        String templateFileName,
                        String templatePath,
                        String fileName,
                        String filePath,
                        List<ExcelSheet> excelSheets,
                        Map<String, String> messages,
                        Long fileId,
                        Long tenantId,
                        Long userId,
                        Boolean debug,
                        String name,
                        Long sourceFileId
    ) {
        this.fileNameExt = fileNameExt;
        this.sourceFileName = templateFileName;
        this.sourceFilePath = templatePath;
        this.targetFileName = fileName;
        this.targetFilePath = filePath;
        this.excelSheets = excelSheets;
        this.messages = messages;
        this.targetFileId = fileId;
        this.userId = userId;
        this.tenantId = tenantId;
        this.debug = debug;
        this.name = name;
        this.params = new HashMap<>(6);
        this.sourceFileId = sourceFileId;
    }

    /**
     * 构建Builder类
     * @return Builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * 添加参数
     *
     * @param key Key
     * @param obj Obj
     */
    public void param(String key, Object obj) {
        this.params.put(key, obj);
    }

    /**
     * 获取参数
     *
     * @param key
     * @param <T>
     * @return
     */
    public <T> T getParam(String key) {
        if (this.params.containsKey(key)) {
            try {
                return (T) this.params.get(key);
            } catch (Exception e) {
                log.warn(e.getMessage(), e);
            }
        }
        return null;
    }

    /**
     * 获取文件模板的File Path
     * @return Path Path
     */
    public Path getSourcePath() {
        //判断是否包含".xlsx"
        if (StringUtils.isNotBlank(this.sourceFileName) && this.sourceFileName.endsWith(ExcelFile.FILE_NAME_EXT)) {
            return Paths.get(this.sourceFilePath, this.sourceFileName);
        }
        if (StringUtils.isBlank(this.fileNameExt)) {
            this.fileNameExt = ExcelFile.FILE_NAME_EXT;
        }
        return Paths.get(this.sourceFilePath, this.sourceFileName + this.fileNameExt);
    }

    /**
     * 获取文件输出全路Path
     * @return Path Path
     */
    public Path getTargetPath() {
        //如果文件为空则获取，java.临时目录
        if (StringUtils.isBlank(this.targetFileName)) {
            return Paths.get(ExcelFile.createExcelFilePath());
        }
        //判断是否包含".xlsx"
        if (StringUtils.isNotBlank(this.targetFileName) && this.targetFileName.endsWith(ExcelFile.FILE_NAME_EXT)) {
            return Paths.get(this.targetFilePath, this.targetFileName);
        }
        if (StringUtils.isBlank(this.fileNameExt)) {
            this.fileNameExt = ExcelFile.FILE_NAME_EXT;
        }
        return Paths.get(this.targetFilePath, this.targetFileName + this.fileNameExt);
    }

    public static class Builder {
        /**
         * 文件扩展名称
         */
        private String fileNameExt;
        /**
         * 模板名称
         */
        private String sourceFileName;
        /**
         * 模板路径
         */
        private String sourceFilePath;

        /**
         * 文件输入名
         */
        private String targetFileName;
        /**
         * 文件输入名
         */
        private String targetFilePath;
        /**
         * 工作表
         */
        private List<ExcelSheet> excelSheets;

        /**
         * 消息内容
         */
        private Map<String, String> messages;
        /**
         * 文件ID
         */
        private Long targetFileId;


        /**
         * 租户ID
         */
        private Long tenantId;

        /**
         * 用户ID (登录用户)
         */
        private Long userId;

        /**
         * 目标文件，用于下载文件
         */
        private Long sourceFileId;

        /**
         * 默认启动Debug等false
         */
        private Boolean debug = Boolean.FALSE;

        /**
         * 参数内容
         */
        private Map<String, Object> params;
        /**
         * 业务名称
         */
        private String name;


        protected Builder() {
            //6条消息
            this.messages = new HashMap<>(6);
            //三个Sheet
            this.excelSheets = new ArrayList<>(3);
            //参数
            this.params = new HashMap<>(6);
        }


        /**
         * 添加参数
         *
         * @param key Key
         * @param obj Obj
         */
        public void param(String key, Object obj) {
            this.params.put(key, obj);
        }

        /**
         * 租户ID
         *
         * @param tenantId 租户ID
         * @return Builder
         */
        public Builder tenantId(Long tenantId) {
            this.tenantId = tenantId;
            return this;
        }


        /**
         * 用户ID
         *
         * @param userId 用户ID
         * @return Builder Builder
         */
        public Builder userId(Long userId) {
            this.userId = userId;
            return this;
        }

        /**
         * 添加ExcelSheet
         *
         * @param excelSheets List<ExcelSheet>
         * @return Builder
         */
        public Builder excelSheets(List<ExcelSheet> excelSheets) {
            this.excelSheets.addAll(excelSheets);
            return this;
        }

        /**
         * 添加ExcelSheet
         *
         * @param excelSheet ExcelSheet
         * @return Builder
         */
        public Builder excelSheet(ExcelSheet excelSheet) {
            this.excelSheets.add(excelSheet);
            return this;
        }

        /**
         * 添加ExcelSheet
         *
         * @param sheetNo   SheetNo
         * @param sheetName SheetName
         * @return ExcelBookDTO ExcelBookDTO
         */
        public Builder excelSheet(Integer sheetNo, String sheetName) {
            ExcelSheet excelSheet = new ExcelSheet(sheetNo, sheetName);
            this.excelSheets.add(excelSheet);
            return this;
        }

        /**
         * 添加ExcelSheet
         *
         * @param sheetNo   SheetNo
         * @param sheetName SheetName
         * @param data      数据
         * @return ExcelBookDTO ExcelBookDTO
         */
        public Builder excelSheet(Integer sheetNo, String sheetName, List<?> data) {
            ExcelSheet excelSheet = new ExcelSheet(sheetNo, sheetName, data);
            this.excelSheets.add(excelSheet);
            return this;
        }

        /**
         * 输入出文件名
         *
         * @param fileName 输出文件名
         * @return Builder
         */
        public Builder fileName(String fileName) {
            this.targetFileName = fileName;
            return this;
        }

        /**
         * 输出文件路径
         *
         * @param filePath 输出文件路径
         * @return Builder
         */
        public Builder filePath(String filePath) {
            this.targetFilePath = filePath;
            return this;
        }

        /**
         * 设置模板路径
         * @param sourceFilePath 模板路径
         * @return Builder
         */
        public Builder sourcePath(String sourceFilePath) {
            this.sourceFilePath = sourceFilePath;
            return this;
        }

        /**
         * 设置模板文件名
         *
         * @param sourceFileName 文件名
         * @return Builder
         */
        public Builder sourceFileName(String sourceFileName) {
            this.sourceFileName = sourceFileName;
            return this;
        }

        /**
         * 设置消息
         *
         * @param key     消息KEY
         * @param content 消息内容
         * @return Builder
         */
        public Builder message(String key, String content) {
            this.messages.put(key, content);
            return this;
        }

        /**
         * 文件上传
         * @param targetFileId 目标文件ID
         * @return Builder
         */
        public Builder targetFileId(Long targetFileId) {
            this.targetFileId = targetFileId;
            return this;
        }

        /**
         * 文件上传
         * @param sourceFileId 源fileId
         * @return Builder
         */
        public Builder sourceFileId(Long sourceFileId) {
            this.sourceFileId = sourceFileId;
            return this;
        }

        /**
         * 设置文件扩展名
         * @param fileNameExt 文件扩展名
         * @return Builder
         */
        public Builder fileNameExt(String fileNameExt) {
            this.fileNameExt = fileNameExt;
            return this;
        }


        /**
         * Debug模式
         *
         * @param debug Debug模式
         * @return Builder
         */
        public Builder debug(Boolean debug) {
            this.debug = debug;
            return this;
        }

        /**
         * 设置业务名称名称
         *
         * @param name 业务名称
         * @return Builder
         */
        public Builder name(String name) {
            this.name = name;
            return this;
        }


        /**
         * 构建ExcelBook
         *
         * @return ExcelBook
         */
        public ExcelBook build() {
            //设置租户默认值
            if (this.tenantId == null) {
                if (null != UserInfoHolder.get()) {
                    this.tenantId = UserInfoHolder.get().getTenantId();
                } else {
                    this.tenantId = 0L;
                }
            }
            //设置用户默认值
            if (this.userId == null) {
                if (null != UserInfoHolder.get()) {
                    this.userId = UserInfoHolder.get().getId();
                } else {
                    this.userId = 0L;
                }
            }
            //设置默认的文件名
            if (StringUtils.isBlank(this.fileNameExt)) {
                this.fileNameExt = ExcelFile.FILE_NAME_EXT;
            }
            //处理TemplatePath为空的情况
            if (StringUtils.isBlank(this.sourceFilePath)) {
                this.sourceFilePath = StringUtils.EMPTY;
            }
            //处理为FilePath空的情况
            if (StringUtils.isBlank(this.targetFilePath)) {
                this.targetFilePath = StringUtils.EMPTY;
            }
            //如果文件名为空，则创建临时目录的文件
            if (StringUtils.isBlank(this.targetFileName)) {
                this.targetFileName = ExcelFile.createExcelFilePath();
            }
            return new ExcelBook(fileNameExt, sourceFileName, sourceFilePath, targetFileName, targetFilePath, excelSheets,
                    messages, targetFileId, tenantId, userId, debug, name, this.sourceFileId);
        }
    }
}
