package com.xforceplus.business.excel;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.*;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.account.service.AccountService;
import com.xforceplus.business.company.service.CompanyServicePackageService;
import com.xforceplus.business.file.service.FileService;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.business.tenant.service.RoleService;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.domain.device.InvoiceType;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.utils.RegExUtil;
import com.xforceplus.utils.excel.ExcelUtils;
import com.xforceplus.utils.excel.exception.ImportException;
import io.geewit.core.exception.ProcessedException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Validator;
import java.util.*;
import java.util.stream.Collectors;

@SuppressWarnings("all")
@Service
public class ExcelService {
    private final static Logger logger = LoggerFactory.getLogger(ExcelService.class);

    private final OrgService orgService;

    private final TenantDao tenantDao;

    private final CompanyServicePackageService companyServicePackageService;

    private final RoleService roleService;

    private final AccountService accountService;

    private final UserService userService;

    private final Validator validator;

    private final FileService fileService;

    public ExcelService(OrgService orgService, TenantDao tenantDao, Validator validator,
                        CompanyServicePackageService companyServicePackageService, RoleService roleService,
                        AccountService accountService, UserService userService, FileService fileService) {
        this.orgService = orgService;
        this.tenantDao = tenantDao;
        this.validator = validator;
        this.companyServicePackageService = companyServicePackageService;
        this.roleService = roleService;
        this.accountService = accountService;
        this.userService = userService;
        this.fileService = fileService;
    }

    public ResponseEntity batchImportCompanies(MultipartFile file) {
        String template = "companyExcelCovertJson.json";

        List<CompanyModel.Request.Import> companies = ExcelUtils.list(file, template, CompanyModel.Request.Import.class);
        List<String> errMsg = new ArrayList<>();
        Set<Long> tenantIds = Sets.newHashSet();
        int success = 0;
        int fail = 0;
        int index = 1;
        for (CompanyModel.Request.Import anImport : companies) {
            List<String> innerError = new ArrayList<>();
            try {
                index++;
                List<String> errorMessages = ExcelUtils.checkField(anImport, validator);
                if (!errorMessages.isEmpty()) {
                    for (String errorMessage : errorMessages) {
                        innerError.add("第" + index + "行，" + errorMessage);
                    }
                } else {
                    Long tenantId = tenantDao.findTenantIdByTenantCode(anImport.getTenantCode());
                    if (tenantId == null) {
                        innerError.add("第" + index + "行，租户code(" + anImport.getTenantName() + ")找不到对应租户");
                        continue;
                    }
                    anImport.setTenantId(tenantId);
                    tenantIds.add(tenantId);
                }
                orgService.saveCompany(anImport.getTenantId(), anImport);
                success++;
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
                fail++;
                innerError.add(e.getMessage());
            }
            if (innerError.size() > 0) {
                errMsg.add(innerError.toString());
            }
        }

        ResponseEntity restResponse = new ResponseEntity();
        if (success > 0) {
            restResponse.setCode("1");
            restResponse.setMessage("导入成功 " + success + " 条");
        } else {
            restResponse.setCode("0");
        }
        if (fail > 0) {
            Long fileId = fileService.uploadMsgExcel(errMsg);
            restResponse.setResult(fileId);
            restResponse.setMessage("导入成功 " + success + " 条, 导入失败 " + fail + " 条");
        }
        return restResponse;
    }


    public ResponseEntity batchImportCompanyPackages(MultipartFile file) {
        String template = "companyServicePackageExcelCovertJson.json";

        List<CompanyModel.Request.CompanyServicePackageExcel> companyServicePackages = ExcelUtils.list(file, template, CompanyModel.Request.CompanyServicePackageExcel.class);
        int success = 0;
        int fail = 0;
        int index = 1;
        List<String> errorMsg = new ArrayList<>();
        for (CompanyModel.Request.CompanyServicePackageExcel companyServicePackageExcel : companyServicePackages) {
            try {
                List<String> errors = ExcelUtils.checkField(companyServicePackageExcel, validator);
                if (CollectionUtils.isEmpty(errors)) {
                    companyServicePackageService.bindCompanyAndPackage(companyServicePackageExcel.getTenantCode(), companyServicePackageExcel.getTaxNum(), companyServicePackageExcel.getServicePackageCode(), companyServicePackageExcel.getStatus(), false);
                } else {
                    int finalIndex = index;
                    errors = errors.stream().map(p -> "第" + finalIndex + "行数据" + p).collect(Collectors.toList());
                    errorMsg.addAll(errors);
                    fail++;
                }
                success++;
            } catch (Exception e) {
                fail++;
                String message = "导入第 " + index + " 行失败," + e.getMessage();
                logger.warn(message, e);
                errorMsg.add(message);
            }
            index++;
        }
        ResponseEntity restResponse = new ResponseEntity();
        if (success > 0) {
            restResponse.setCode("1");
            restResponse.setMessage("导入成功 " + success + " 条");
        } else {
            restResponse.setCode("0");
        }
        if (fail > 0) {
            Long fileId = fileService.uploadMsgExcel(errorMsg);
            restResponse.setResult(fileId);
            restResponse.setMessage("导入成功 " + success + " 条, 导入失败 " + fail + " 条");
        }
        return restResponse;
    }

    public ResponseEntity<Long> batchImportRoles(Long tenantId, MultipartFile file) {
        String template = "roleExcelCovertJson.json";

        List<RoleModel.Request.RoleImportExcel> roleImportExcelList = ExcelUtils.list(file, template, RoleModel.Request.RoleImportExcel.class);
        int success = 0;
        int fail = 0;
        int index = 1;
        List<String> errorMsg = new ArrayList<>();

        for (RoleModel.Request.RoleImportExcel roleImport : roleImportExcelList) {
            try {
                ExcelUtils.validField(roleImport, validator);
                RoleModel.Request.Query query = new RoleModel.Request.Query();
                query.setRoleCode(roleImport.getRoles().replaceAll("/", ";"));
                query.setTenantId(tenantId);
                query.setType(0);
                List<Role> roleList = roleService.list(query, Sort.unsorted());
                if (CollectionUtils.isEmpty(roleList)) {
                    fail++;
                    errorMsg.add("导入第 " + index + " 行失败," + "当前角色不存在");
                    continue;
                }
                Set<Role> roles = new HashSet<>(roleList);

                String accountName = roleImport.getAccount();
                AccountModel.Request.Query accQuery = new AccountModel.Request.Query();
                if (accountName.contains(Separator.AT)) {
                    accQuery.setEmail(accountName);
                } else if (RegExUtil.checkMobile(accountName)) {
                    accQuery.setTelPhone(accountName);
                } else {
                    accQuery.setUsername(accountName);
                }
                List<Account> accounts = accountService.list(accQuery, Sort.unsorted());
                if (CollectionUtils.isEmpty(accounts)) {
                    fail++;
                    errorMsg.add("导入第 " + index + " 行失败," + "当前账号不存在");
                    continue;
                }
                UserModel.Request.Query userQuery = new UserModel.Request.Query();
                userQuery.setAccountId(accounts.get(0).getAccountId());
                userQuery.setTenantId(tenantId);

                List<User> userList = userService.list(userQuery, Sort.unsorted());
                if (CollectionUtils.isEmpty(userList)) {
                    fail++;
                    errorMsg.add("导入第 " + index + " 行失败," + "当前用户不存在");
                    continue;
                }
                Set<User> users = new HashSet<>(userList);
                RoleModel.Request.BindUsers bindUsers = new RoleModel.Request.BindUsers();
                bindUsers.setOverwrite(true);
                bindUsers.setUserIds(users.stream().map(User::getId).collect(Collectors.toList()));
                for (User user : users) {
                    userService.bindRoles(user, null, roles.stream().map(Role::getId).collect(Collectors.toSet()), null, null, null, true, false, true);
                }
                userService.commitBindUserRoles();
                UserService.removeCachedRoleOrgUserRelsThreadLocal();
                success++;
            } catch (Exception e) {
                fail++;
                String message = "导入第 " + index + " 行失败," + e.getMessage();
                logger.warn(message, e);
                errorMsg.add(message);
            }
            index++;
        }

        //上传错误信息oss
        Long fileId = fileService.uploadMsgExcel(errorMsg);
        ResponseEntity restResponse = new ResponseEntity();
        restResponse.setCode(null == fileId ? "1" : "0");
        restResponse.setResult(fileId);
        restResponse.setMessage("导入成功 " + success + " 条, 导入失败 " + fail + " 条");
        return restResponse;
    }

    public ResponseEntity batchImportUsers(Long tenantId, MultipartFile file) {
        String template = "user_batch_import.json";

        List<UserModel.Request.UserImportExcel> userImportExcelList = ExcelUtils.list(file, template, UserModel.Request.UserImportExcel.class);
        int success = 0;
        int fail = 0;
        int index = 1;
        List<String> errorMsg = new ArrayList<>();
        for (UserModel.Request.UserImportExcel userImport : userImportExcelList) {
            try {
                ExcelUtils.validField(userImport, validator);
                this.saveUserFromExcel(userImport, tenantId);
                success++;
            } catch (Exception e) {
                fail++;
                String message = "导入第 " + index + " 行失败," + e.getMessage();
                logger.error(message, e);
                errorMsg.add(message);
            }
            index++;
        }


        //上传错误信息oss
        Long fileId = fileService.uploadMsgExcel(errorMsg);
        ResponseEntity restResponse = new ResponseEntity();
        restResponse.setCode(null == fileId ? "1" : "0");
        restResponse.setResult(fileId);
        restResponse.setMessage("导入成功 " + success + " 条, 导入失败 " + fail + " 条");
        return restResponse;
    }


    private void saveUserFromExcel(UserModel.Request.UserImportExcel userImport, Long tenantId) throws ImportException {
        List<String> errorMsg = new ArrayList<>();
        // 检查设备 不用再检查设备信息
        // 检查发票类型
        if (validInvoiceType(userImport.getInvoiceType())) {
            errorMsg.add("发票类型不正确");
            throw new ImportException(errorMsg);
        }

        // 更换存储方式
        userImport.setTicketOpeningTerminal(userImport.getTicketOpeningTerminal().replaceAll("/", ","));
        userImport.setInvoiceType(userImport.getInvoiceType().replaceAll("/", ","));

        if (!StringUtils.isBlank(userImport.getStatus()) && !checkUserStatus(userImport.getStatus())) {
            errorMsg.add("用户状态不正确");
            throw new ImportException(errorMsg);
        }

        // 检查组织
        List<OrgStruct> orgStructList = Lists.newArrayList();
        if (StringUtils.isBlank(userImport.getUserOrgs())) {
            //errorMsg.add("所属组织不能为空");
            //throw new ImportException(errorMsg);
        } else {
            orgStructList = validOrgsExist(tenantId, userImport.getUserOrgs());
            if (CollectionUtils.isEmpty(orgStructList)) {
                errorMsg.add("不存在的组织code");
                throw new ImportException(errorMsg);
            }
        }

        Optional<Tenant> tenant = tenantDao.findById(tenantId);
        if (!tenant.isPresent()) {
            errorMsg.add("不存在的租户");
            throw new ImportException(errorMsg);
        }

        // 创建账户
        UserModel.Request.Create userModel = new UserModel.Request.Create();
        userModel.setType(AccountType.PHONE_EMAIL);
        AccountModel.Request.Create accountModel = new AccountModel.Request.Create();
        BeanUtils.copyProperties(userImport, userModel);
        BeanUtils.copyProperties(userImport, accountModel);
        accountModel.setTelPhone(userImport.getUserPhone());

        userModel.setStatus(StringUtils.isBlank(userImport.getStatus()) ? null : Integer.parseInt(userImport.getStatus()));
        accountModel.setUsername(userImport.getAccountNum());
        accountModel.setChangePasswordFlag(!StringUtils.isBlank(userImport.getChangePasswordFlag()) && Boolean.parseBoolean(userImport.getChangePasswordFlag()));
        if (userImport.getAccountNum().contains(Separator.AT)) {
            accountModel.setEmail(userImport.getAccountNum());
            userModel.setUserEmailAddr(userImport.getAccountNum());
            accountModel.setTelPhone(null);
        } else if (RegExUtil.checkMobile(userImport.getAccountNum())) {
            accountModel.setTelPhone(userImport.getAccountNum());
            accountModel.setEmail(null);
        } else {
            accountModel.setUsername(userImport.getAccountNum());
            accountModel.setEmail(null);
            accountModel.setTelPhone(null);
            userModel.setType(AccountType.OTHER);
        }
        if (userImport.getUserSex().contains("男")) {
            userModel.setUserSex(0);
        } else {
            userModel.setUserSex(1);
        }

        //关联组织
        Set<String> orgCodes = orgStructList.stream().map(OrgStruct::getOrgCode).collect(Collectors.toSet());
        userModel.setOrgCodes(orgCodes);
        // 保存用户
        userModel.setAccount(accountModel);
        userModel.setTenantId(tenantId);
        userService.save(userModel, true, true, true, true, true, false);
    }

    private List<OrgStruct> validOrgsExist(long tenantId, String orgsStr) {
        return orgService.findByTenantIdAndOrgCode(tenantId, orgsStr.replaceAll("/", ";"));
    }

    private boolean validInvoiceType(String invoiceType) {
        if (StringUtils.isBlank(invoiceType)) {
            return true;
        }
        List<String> invoiceTypeList = InvoiceType.toList();
        String[] invoiceTypes = invoiceType.split("/");
        return Arrays.stream(invoiceTypes).anyMatch(item -> !invoiceTypeList.contains(item));
    }


    private boolean checkUserStatus(String userStatus) {
        Set<String> statusSet = Sets.newHashSet("0", "1", "2", "3");
        return statusSet.contains(userStatus);
    }


    public ResponseEntity processExcel(Long tenantId, MultipartFile file) {
        //存放保存的组织
        Map<String, OrgStruct> currentSaveOrgs = new HashMap<>();
        String template = "orgExcelCovertJson.json";
        int success = 0, fail = 0, i = 0;
        List<String> errorMsg = new ArrayList<>();
        List<OrgModel.Request.ExcelModel> excelModels;
        try {
            excelModels = ExcelUtils.list(file, template, OrgModel.Request.ExcelModel.class);
        } catch (ProcessedException e) {
            throw e;
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
            throw new IllegalArgumentException("导入参数解析异常");
        }
        int size = excelModels.size();
        if (CollectionUtils.isEmpty(excelModels) || size > 1000) {
            throw new IllegalArgumentException("导入数据为空或者大于1000条");
        }
        OrgModel.Request.ExcelModel currentOrg;

        while (i < size && (currentOrg = excelModels.get(i)) != null) {
            i++;
            if (currentSaveOrgs.containsKey(currentOrg.getOrgCode())) {
                success++;
                continue;
            }
            try {
                this.processOrgs(tenantId, currentSaveOrgs, currentOrg, excelModels);
                success++;
            } catch (ImportException e) {
                fail++;
                for (String msg : e.getMsgError()) {
                    errorMsg.add("导入第 " + i + " 行失败," + msg);
                }

            } catch (Exception e) {
                fail++;
                logger.error("导入第 " + i + " 行失败," + e.getMessage(), e);
                logger.error("-----------------------------&&**-----------------------------");
                errorMsg.add("导入第 " + i + " 行失败," + e.getMessage());
            }
        }
        ResponseEntity restResponse = new ResponseEntity();
        if (fail > 0) {
            logger.warn(StringUtils.join(errorMsg), ",");
            Long fileId = fileService.uploadMsgExcel(errorMsg);
            restResponse.setResult(fileId);
            restResponse.setCode("0");
            restResponse.setMessage("导入成功 " + success + " 条, 导入失败 " + fail + " 条");
        } else {
            if (success > 0) {
                restResponse.setMessage("导入成功 " + success + " 条");
            }
            restResponse.setCode("1");
        }
        return restResponse;

    }

    private void processOrgs(Long tenantId, Map<String, OrgStruct> currentSaveOrgs, OrgModel.Request.ExcelModel org, List<OrgModel.Request.ExcelModel> excelModels) throws ImportException {
        if (currentSaveOrgs.containsKey(org.getOrgCode())) {
            return;
        }
        //数据库查询父组织

        OrgModel.Request.ExcelModel curOrg = org;
        Stack<OrgModel.Request.ExcelModel> models = new Stack<>();
        //查询入栈递归
        OrgStruct parentOrg = null;
        while (null == parentOrg) {
            if (StringUtils.isBlank(curOrg.getParentOrgCode()) || StringUtils.isBlank(curOrg.getOrgCode())) {
                throw new ImportException(Lists.newArrayList("当前组织的代码和父组织代码为空!"));
            }
            //当时组织类型=0，转换为：2,即其它类型
            if (OrgType.GROUP.value() == curOrg.getOrgType()) {
                curOrg.setOrgType(OrgType.NORMAL.value());
            }
            if (curOrg.getOrgType() > 2 || curOrg.getOrgType() < 0) {
                throw new ImportException(Lists.newArrayList("组织类型不正确!"));
            }
            models.add(curOrg);
            parentOrg = orgService.findParentOrgType(tenantId, curOrg);
            if (null == parentOrg) {
                curOrg = orgService.findParentOrgInModels(org, excelModels);
                if (null == curOrg) {
                    throw new ImportException(Lists.newArrayList("当前组织的父组织不存在!"));
                }
            }
        }
        //出栈
        do {
            OrgModel.Request.ExcelModel excelOrg = models.pop();
            OrgStruct innerParentOrg = orgService.findParentOrgType(tenantId, curOrg);
            orgService.saveSingle(tenantId, currentSaveOrgs, excelOrg, innerParentOrg);
        } while (!models.isEmpty());


    }
}
