/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: S3ExcelHandler.java   2020-09-17 13-48-06
 * Author: Evan
 */
package com.xforceplus.business.excel;

import com.xforceplus.utils.filetransfer.FileTransferUtilsService;
import io.geewit.web.utils.SpringContextUtil;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;

/**
 * <p>
 * Title: 上传到对象存储（SSO,S3,Minio）
 * </p>
 * <p>
 * Description: 上传到对象存储（SSO,S3,Minio） Last Two Execute
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-17 13-48-06
 * @since V1.0
 */
public class FileExcelWriterHandler implements ExcelHandler {

    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(FileExcelWriterHandler.class);
    /**
     *  文件服务
     */
    private final FileTransferUtilsService fileTransferUtilsService = SpringContextUtil.getBean(FileTransferUtilsService.class);

    /**
     *  处理前
     * @param excelBook Excel上下文
     */
    @Override
    public void before(ExcelBook excelBook) {
        logger.info("上传文件Before");
    }

    /**
     * 处理上传到
     *
     * @param excelBook 上下文
     */
    @Override
    public void after(ExcelBook excelBook) {
        logger.info("开始上传文件...");
        //获取上传文件实例
        File file = excelBook.getTargetPath().toFile();
        String filePath = file.getPath();
        //文件检查是否为空
        try {
            logger.info("上传文件路径是：{}", filePath);
            this.checkSize(file);
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
            excelBook.setException(e);
            FileUtils.deleteQuietly(file);
            return;
        }
        try {
            Long fileId = fileTransferUtilsService.upload(filePath, excelBook.getUserId(), excelBook.getTenantId());
            logger.info("上传成功文件ID:{}", fileId);
            excelBook.setTargetFileId(fileId);
        } catch (FileNotFoundException e) {
            logger.error("导出信息上传文件服务失败,{}", filePath, e);
            excelBook.setException(e);
        } finally {
            //删除文件
            FileUtils.deleteQuietly(file);
        }
        logger.info("结束上传文件...");
    }

    /**
     * 检查文件在小
     *
     * @param file File
     */
    protected void checkSize(File file) {
        long size = FileUtils.sizeOf(file);
        logger.info("file size:{}", size);
        if (size == 0) {
            throw new IllegalStateException("文件导出入失败");
        }
    }
}