/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: MessageExcelHandler.java   2020-09-17 13-48-15
 * Author: Evan
 */
package com.xforceplus.business.excel;

import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.notice.dto.ExcelNoticeConfig;
import com.xforceplus.business.notice.dto.NoticeMessage;
import com.xforceplus.business.notice.service.NoticeService;
import com.xforceplus.utils.StrUtils;
import io.geewit.web.utils.SpringContextUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Title: 消息通知
 * </p>
 * <p>
 * Description: 消息通知(根据模板消息通知)  Last First Handler
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-17 13-48-15
 * @since V1.0
 */
public class MessageExcelWriterHandler implements ExcelHandler {
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(MessageExcelWriterHandler.class);

    /**
     *  消息服务
     */
    private final NoticeService noticeService = SpringContextUtil.getBean(NoticeService.class);

    /**
     *  处理前
     * @param excelBook Excel上下文
     */
    @Override
    public void before(ExcelBook excelBook) {
        logger.info("发送消息通知-Before");
        if (excelBook.getException() != null) {
            logger.error(excelBook.getException().getMessage(), excelBook.getException().getCause());
            String noticeKey = ExcelNoticeConfig.EXP_EXCEPTION;
            String message = excelBook.getException().getMessage();
            if (StringUtils.isNotEmpty(message)) {
                message = StrUtils.matcherZh(message, Separator.SPACE);
            }
            if (StringUtils.isBlank(message)) {
                message = "系统发生错误，请稍后重试或联系运营人员";
            }
            //处理导入失败情况
            Object[] params = new Object[]{excelBook.getName(), message};
            //发送消息
            this.sendNoticeMessage(excelBook, noticeKey, params);
        }
    }

    /**
     * 处理后
     *
     * @param excelBook Excel上下文
     */
    @Override
    public void after(ExcelBook excelBook) {
        String noticeKey = ExcelNoticeConfig.EXP_SUCCESS;
        Object successSize = this.successSize(excelBook);
        String downloadUrl = ExcelConfig.getDownloadUrl();
        logger.info("发送消息通知-after,downloadUrl:{}", downloadUrl);
        Object[] params = new Object[]{excelBook.getName(), successSize, downloadUrl + excelBook.getTargetFileId(), String.valueOf(excelBook.getTargetFileId())};
        if (excelBook.getException() != null) {
            //处理异常消息
            noticeKey = ExcelNoticeConfig.EXP_EXCEPTION;
            params = new Object[]{excelBook.getName(), excelBook.getException().getMessage()};
        }
        //发送消息
        this.sendNoticeMessage(excelBook, noticeKey, params);
    }

    /**
     * 发送消息
     *
     * @param excelBook 工作薄
     * @param noticeKey 消息模板KEY
     * @param params    参数
     */
    protected void sendNoticeMessage(ExcelBook excelBook, String noticeKey, Object[] params) {
        try {
            NoticeMessage noticeMessage = NoticeMessage.builder()
                    .userId(excelBook.getUserId())
                    .tenantId(excelBook.getTenantId())
                    .key(noticeKey)
                    .params(params)
                    .build();
            //发送消息通知
            this.sendNotice(noticeMessage);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    /**
     * 获取成功记录数
     * @param excelBook ExcelBook
     * @return String String
     */
    private String successSize(ExcelBook excelBook) {
        if (excelBook.getParams().containsKey(ExcelFile.SUCCESS_SIZE)) {
            return String.valueOf(excelBook.getParams().get(ExcelFile.SUCCESS_SIZE));
        }
        return String.valueOf(ExcelFile.ZERO_SIZE);
    }

    /**
     * 发送消息
     * @param noticeMessage  NoticeMessage
     */
    protected void sendNotice(NoticeMessage noticeMessage) {
        this.noticeService.sendExcelProcessNotice(noticeMessage.getExcelProcessNotice());
    }
}
