/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelWriter.java   2020-09-17 13-42-57
 * Author: Evan
 */
package com.xforceplus.business.excel;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.*;

/**
 * <p>
 * Title: ExcelWriter写入服务
 * </p>
 * <p>
 * Description: ExcelWriter写入服务
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-17 13-42-57
 * @since V1.0
 */
public class SimpleExcelWriter {
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(SimpleExcelWriter.class);
    /**
     * 处理级
     */
    private Map<String, ExcelHandler> handlers;
    /**
     * 工作薄对象
     */
    private ExcelBook excelBook;
    /**
     * Excel写入对象
     */
    private ExcelWriter excelWriter;
    /**
     * 填写Sheet表
     */
    private Map<String, WriteSheet> writeSheets;


    /**
     * 初始化SimpleExcelWriter类
     *
     * @param handlers  拦截器
     * @param excelBook Excel下上文
     */
    private SimpleExcelWriter(Map<String, ExcelHandler> handlers, ExcelBook excelBook) {
        this.handlers = handlers;
        this.excelBook = excelBook;
        this.writeSheets = new HashMap<>(2);
    }

    /**
     * 构建SimpleExcelWriter
     * @return Builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * 开始写入
     *
     * @return SimpleExcelWriter
     */
    protected SimpleExcelWriter start() {
        logger.info("开始导出文件...");
        //获取ExcelBook
        logger.info("filePath:{}", this.excelBook.getTargetPath().toFile());

        logger.info("TemplateFilePath:{}", this.excelBook.getSourcePath().toFile());
        try {
            //模板路径
            InputStream inputStream = this.loadTemplateFile(this.excelBook.getSourcePath());
            //导出文件
            //创建excelWriter
            this.excelWriter = EasyExcel
                    .write(this.excelBook.getTargetPath().toFile())
                    .withTemplate(inputStream)
                    .build();
        } catch (RuntimeException e) {
            excelBook.setException(e);
        }
        //执行拦截器 before
        this.beforeHandler();
        try {
            //创建WriteSheet
            this.createWriteSheet(this.excelBook.getExcelSheets());
        } catch (Exception e) {
            excelBook.setException(e);
        }
        return this;
    }

    /**
     * 获取ExcelBook
     *
     * @return
     */
    public ExcelBook getExcelBook() {
        return excelBook;
    }

    /**
     * 加载文件列表
     * @param path 路径
     * @return String
     */
    private InputStream loadTemplateFile(Path path) {
        InputStream inputStream;
        try {
            logger.info("path:{}", path.toString());
            inputStream = new ClassPathResource(path.toString()).getInputStream();
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
            //将异常保存到下一个环节处理
            throw new IllegalArgumentException(e);
        }
        return inputStream;
    }

    /**
     * 创建 WriteSheet
     *
     * @param sheets ExcelSheet对象
     */
    protected void createWriteSheet(List<ExcelSheet> sheets) {
        if (CollectionUtils.isEmpty(sheets)) {
            logger.warn("无Sheet文件");
            return;
        }
        //创建WriteSheet
        for (ExcelSheet sheet : sheets) {
            logger.info("sheetNo:{},SheetName:{}", sheet.getSheetNo(), sheet.getSheetName());
            this.writeSheets.put(sheet.getSheetName(), EasyExcel.writerSheet(sheet.getSheetNo(), sheet.getSheetName()).build());
        }
    }

    /**
     * 执行拦截器before方法
     */
    protected void beforeHandler() {
        if (this.isHandlerEmpty()) {
            return;
        }
        for (Map.Entry<String, ExcelHandler> entry : this.handlers.entrySet()) {
            logger.info("Execute before Handler:{}", entry.getKey());
            entry.getValue().before(this.excelBook);
        }
    }

    /**
     * 按模板小批量数据请执行 finish方法关闭执行
     * @return SimpleExcelWriter   简单Excel写入对象
     */
    public SimpleExcelWriter fill() {
        List<ExcelSheet> excelSheets = this.excelBook.getExcelSheets();
        if (CollectionUtils.isEmpty(excelSheets)) {
            logger.warn("ExcelContext hasn't ExcelSheet data");
            return this;
        }
        //写入数据
        for (ExcelSheet sheet : excelSheets) {
            //过滤无数据写入情况
            if (CollectionUtils.isEmpty(sheet.getData())) {
                continue;
            }
            this.write(sheet.getSheetName(), sheet.getData());
        }
        return this;
    }

    /**
     * 添加参数
     *
     * @param key Key
     * @param obj Obj
     */
    public void param(String key, Object obj) {
        this.excelBook.getParams().put(key, obj);
    }

    /**
     * 设置异常
     * @param e 异常
     */
    public void setException(Exception e) {
        this.excelBook.setException(e);
    }

    /**
     * 按模板分批填充写，执行完成，请执行 finish方法关闭执行
     *
     * @param sheetName SheetName
     * @param data      分而对象
     * @return SimpleExcelWriter
     */
    public SimpleExcelWriter fill(String sheetName, List<?> data) {
        if (CollectionUtils.isEmpty(data)) {
            logger.warn("SheetName:{}写入数据为空", sheetName);
            return this;
        }
        this.write(sheetName, data);
        return this;
    }

    protected void write(String sheetName, List<?> data) {
        if (!this.writeSheets.containsKey(sheetName)) {
            logger.info("没有找到Sheet名称为：{}的Sheet", sheetName);
            return;
        }
        WriteSheet writeSheet = this.writeSheets.get(sheetName);
        this.excelWriter.fill(data, writeSheet);
    }

    /**
     * 执行关闭
     * @return SimpleExcelWriter 简单Excel写入对象
     */
    public SimpleExcelWriter finish() {
        //防止无数据时，模板还是显示模板参数
        this.beforeFinish();
        //关闭文件流
        if (this.excelWriter != null) {
            this.excelWriter.finish();
        }
        //执行拦截器 after
        this.afterHandler();
        logger.info("结束导出文件...");
        return this;
    }

    /**
     * 解决无数据时，Excel模板的参数还显示出来的情况；
     */
    protected void beforeFinish() {
        List<ExcelSheet> excelSheets = this.excelBook.getExcelSheets();
        if (CollectionUtils.isEmpty(excelSheets)) {
            logger.warn("ExcelContext hasn't ExcelSheet data");
            return;
        }
        //写入数据
        for (ExcelSheet sheet : excelSheets) {
            this.write(sheet.getSheetName(), Collections.emptyList());
        }
    }

    /**
     * 判断处理为空情况
     * @return boolean Boolean
     */
    protected boolean isHandlerEmpty() {
        if (CollectionUtils.isEmpty(this.handlers)) {
            logger.warn("SimpleExcelWriter hasn't Excel Handler");
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    /**
     * 执行拦截器
     */
    protected void afterHandler() {
        if (this.isHandlerEmpty()) {
            return;
        }
        for (Map.Entry<String, ExcelHandler> entry : this.handlers.entrySet()) {
            logger.info("Execute After Handler:{}", entry.getKey());
            entry.getValue().after(this.excelBook);
        }
    }

    /**
     * 创建SimpleExcelWriter对象
     */
    public static class Builder {
        /**
         * 默认消息通知 {@value}
         */
        public static final String DEFAULT_MESSAGE_HANDLER = "defaultMessageHandler";
        /**
         * 默认的文件上传
         */
        public static final String DEFAULT_FILE_EXCEL_HANDLER = "DEFAULT_FILE_EXCEL_HANDLER";
        /**
         * Handler集合
         */
        private Map<String, ExcelHandler> handlers;

        /**
         * ExcelBook
         */
        private ExcelBook excelBook;

        Builder() {
            handlers = new LinkedHashMap<>(5);
        }

        /**
         * 初始化类
         */
        protected void init() {

            //文件上传S3
            this.handlers.put(DEFAULT_FILE_EXCEL_HANDLER, new FileExcelWriterHandler());
            //默认Handler处理器
            this.handlers.put(DEFAULT_MESSAGE_HANDLER, new MessageExcelWriterHandler());
        }

        /**
         * 添加拦截器
         * @param name HandlerName
         * @param handler ExcelHandler
         * @return Builder Builder
         */
        public Builder handler(String name, ExcelHandler handler) {
            this.handlers.put(name, handler);
            return this;
        }

        /**
         * 设置Excel上下文
         * @param excelBook ExcelBook
         * @return Builder Builder
         */
        public Builder excelBook(ExcelBook excelBook) {
            this.excelBook = excelBook;
            return this;
        }

        /**
         * 创建SimpleExcelBuilder实例 并调用Start方法
         *
         * @return SimpleExcelWriter SimpleExcelWriter
         */
        public SimpleExcelWriter build() {
            this.init();
            return new SimpleExcelWriter(handlers, excelBook).start();
        }
    }
}
