/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyExcelDTO.java   2020-09-22 10-23-50
 * Author: Evan
 */
package com.xforceplus.business.excel.company;

import com.alibaba.excel.annotation.ExcelIgnore;
import com.alibaba.excel.annotation.ExcelProperty;
import com.alibaba.excel.annotation.format.DateTimeFormat;
import com.xforceplus.business.excel.DataRow;
import com.xforceplus.constants.RegPattern;
import com.xforceplus.validation.constraints.StringInclude;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.Range;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.Pattern;
import javax.validation.constraints.Size;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Date;

/**
 * <p>
 * Title: 公司批量导出Excel类
 * </p>
 * <p>
 * Description: 公司批量导出Excel类
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-22 10-23-50
 * @since V1.0
 */
@Setter
@Getter
@ToString
public class CompanyExcelImportDTO extends DataRow implements Serializable {

    /**
     * 租户ID
     */
    @ExcelIgnore
    protected Long tenantId;
    /**
     * 操作
     */

    @NotBlank(message = "操作 不能为空")
    @StringInclude(message = "操作 应该为:{\"新建\",\"修改\"}", range = {"新建", "修改"})
    @ExcelProperty(value = "操作")
    protected String action;
    /**
     * 租户代码
     */
    @ExcelProperty("租户代码")
    @NotBlank(message = "租户代码不能为空")
    protected String tenantCode;
    /**
     * 公司注册省份
     */
    @ExcelProperty("公司注册省份")
    @Length(max = 50, message = "公司注册省份字符长度<=50个字符")
    protected String registLocationArea;
    /**
     * 公司注册城市
     */
    @ExcelProperty("公司注册城市")
    @Length(max = 50, message = "公司注册城市字符长度<=50个字符")
    protected String registLocationCity;
    /**
     * 公司注册详细地址
     */
    @ExcelProperty("公司注册详细地址")
    @Length(max = 500, message = "公司注册详细地址字符长度<=500个字符")
    protected String registLocationAddr;
    /**
     * 公司经营所在省份
     */
    @ExcelProperty("公司经营所在省份")
    @Length(max = 50, message = "公司经营所在省份字符长度<=50个字符")
    protected String locationArea;
    /**
     * 公司经营所在市区
     */
    @ExcelProperty("公司经营所在市区")
    @Length(max = 50, message = "公司经营所在市区字符长度<=50个字符")
    protected String locationCity;
    /**
     * 公司经营详细地址
     */
    @ExcelProperty("公司经营详细地址")
    @Length(max = 500, message = "公司经营详细地址字符长度<=500个字符")
    protected String locationAddr;
    /**
     * 公司联系电话
     */
    @ExcelProperty("公司联系电话")
    @Length(max = 63, message = "公司联系电话在市区字符长度<=63个字符")
    protected String companyPhone;
    /**
     * 营业期限开始时间
     */
    @ExcelProperty("营业期限开始时间")
    @DateTimeFormat("yyyy-MM-dd")
    protected Date businessStartTime;
    /**
     * 营业期限结束时间
     */
    @ExcelProperty("营业期限结束时间")
    @DateTimeFormat("yyyy-MM-dd")
    protected Date businessEndTime;
    /**
     * 营业时间是否长期
     */
    @ExcelProperty("营业时间长期")
    @Range(max = 1, min = 0, message = "营业时间是否长期：0-不是期或1-是长期")
    protected Integer businessTimeLong;
    /**
     * 企业经营范围
     */
    @ExcelProperty("企业经营范围")
    @Length(max = 2000, message = "法人归属地长度<=2000个字符")
    protected String businessScope;
    /**
     * 平台管理人身份,0:法定代表人, 1:代理人
     */
    @ExcelProperty("平台管理人身份")
    @Range(max = 1, min = 0, message = "平台管理人身份：0-代理人或1-法人")
    protected Integer platManagerStatus;
    /**
     * 法人归属地
     */
    @ExcelProperty("法人归属地")
    @Length(max = 200, message = "法人归属地长度<=200个字符")
    protected String managerLocation;
    /**
     * 法人姓名
     */
    @ExcelProperty("法人姓名")
    @Length(max = 50, message = "法人姓名长度<=500个字符")
    protected String managerName;
    /**
     * 法人证件开始时间
     */
    @ExcelProperty("法人证件开始时间")
    @DateTimeFormat("yyyy-MM-dd")
    protected Date managerIdCardStartTime;
    /**
     * 法人证件结束时间
     */
    @ExcelProperty("法人证件结束时间")
    @DateTimeFormat("yyyy-MM-dd")
    protected Date managerIdCardEndTime;
    /**
     * 法人身份证长期
     */
    @ExcelProperty("法人身份证长期")
    @Range(max = 1, min = 0, message = "法人身份证长期类型必须是：1-长期或0-非长期")
    protected Integer managerIdCardTimeLong;
    /**
     * 法人证件类型  1-身份证
     */
    @ExcelProperty("法人证件类型")
    protected String managerCardType;
    /**
     * 法人身份证
     */
    @ExcelProperty("法人身份证")
    @Length(max = 50, message = "法人身份证长度<=50个字符")
    protected String managerIdCard;
    /**
     * 法人联系方式
     */
    @ExcelProperty("法人联系方式")
    @Length(max = 50, message = "法人联系方式长度<=50个字符")
    protected String managerPhone;
    /**
     * 代理人姓名
     */
    @ExcelProperty("代理人姓名")
    @Length(max = 50, message = "代理人姓名长度<=50个字符")
    protected String proxyManagerName;
    /**
     * 代理人证件开始时间
     */
    @ExcelProperty("代理人证件开始时间")
    @DateTimeFormat("yyyy-MM-dd")
    protected Date proxyManagerIdCardStartTime;
    /**
     * 代理人证件结束时间
     */
    @ExcelProperty("代理人证件结束时间")
    @DateTimeFormat("yyyy-MM-dd")
    protected Date proxyManagerIdCardEndTime;
    /**
     * 代理人身份证长期
     */
    @ExcelProperty("代理人身份证长期")
    @Range(max = 1, min = 0, message = "代理人身份证长期类型必须是：1-长期或0-非长期")
    protected Integer proxyManagerIdCardTimeLong;
    /**
     * 代理人证据类型  1-身份证
     */
    @ExcelProperty("代理人证据类型")
    protected String proxyManagerCardType;
    /**
     * 代理人身份证
     */
    @ExcelProperty("代理人身份证")
    @Length(max = 50, message = "代理人身份证方式长度<=50个字符")
    protected String proxyManagerIdCard;
    /**
     * 代理人联系方式
     */
    @ExcelProperty("代理人联系方式")
    @Length(max = 50, message = "代理人联系方式长度<=50个字符")
    protected String proxyManagerPhone;
    /**
     * 开户银行名称
     */
    @ExcelProperty("开户银行名称")
    @Length(max = 50, message = "开户银行名称长度<=50个字符")
    protected String bankName;
    /**
     * 开户银行支行名称
     */
    @ExcelProperty("开户银行支行名称")
    @Length(max = 50, message = "开户银行支行名称长度<=50个字符")
    protected String bankBranchName;
    /**
     * 对公银行账户
     */
    @ExcelProperty("对公银行账户")
    @Length(max = 100, message = "对公银行账户长度<=100个字符")
    protected String bankNo;
    /**
     * 纳税人资质类型 1:一般增值税纳税人 2:小规模纳税人
     */
    @ExcelProperty("纳税人资质类型")
    @Range(max = 2, min = 1, message = "纳税人资质类型：1:一般增值税纳税人 或2:小规模纳税人")
    protected Integer taxpayerQualificationType;
    /**
     * 增值税普通发票限额
     */
    @ExcelProperty("增值税普通发票限额")
    protected BigDecimal cquota;
    /**
     * 增值税专用发票限额
     */
    @ExcelProperty("增值税专用发票限额")
    protected BigDecimal squota;
    /**
     * 增值税电子普通发票限额
     */
    @ExcelProperty("增值税电子普通发票限额")
    protected BigDecimal ceQuota;
    /**
     * 增值税普通发票-卷票限额
     */
    @ExcelProperty("增值税普通发票-卷票限额")
    protected BigDecimal juQuota;
    /**
     * 增值税电子专票限额
     */
    @ExcelProperty("增值税电子专用发票限额")
    protected BigDecimal seQuota;

    /**
     * 机动车发票限额
     */
    @ExcelProperty("机动车发票限额")
    protected BigDecimal vehicleLimit;
    /**
     * 税号
     */
    @NotEmpty(message = "统一社会信用代码不能为空")
    @Pattern(regexp = RegPattern.TAX_NUM, message = "统一社会信用代码长度为15、16、17、18或者20位数字字母组合")
    @ExcelProperty("统一社会信用代码")
    private String taxNum;
    /**
     * 企业名称
     */
    @NotEmpty(message = "公司名称不为空")
    @ExcelProperty("公司名称")
    @Size(min = 1, max = 60, message = "公司名称长度介于1~60个字符之间")
    private String companyName;

    public void setAction(String action) {
        this.action = StringUtils.trim(action);
    }

    public void setTenantCode(String tenantCode) {
        this.tenantCode = StringUtils.trim(tenantCode);
    }

    public void setRegistLocationArea(String registLocationArea) {
        this.registLocationArea = StringUtils.trim(registLocationArea);
    }

    public void setRegistLocationCity(String registLocationCity) {
        this.registLocationCity = StringUtils.trim(registLocationCity);
    }

    public void setRegistLocationAddr(String registLocationAddr) {
        this.registLocationAddr = StringUtils.trim(registLocationAddr);
    }

    public void setLocationArea(String locationArea) {
        this.locationArea = StringUtils.trim(locationArea);
    }

    public void setLocationCity(String locationCity) {
        this.locationCity = StringUtils.trim(locationCity);
    }

    public void setLocationAddr(String locationAddr) {
        this.locationAddr = StringUtils.trim(locationAddr);
    }

    public void setCompanyPhone(String companyPhone) {
        this.companyPhone = StringUtils.trim(companyPhone);
    }

    public void setBusinessScope(String businessScope) {
        this.businessScope = StringUtils.trim(businessScope);
    }

    public void setManagerLocation(String managerLocation) {
        this.managerLocation = StringUtils.trim(managerLocation);
    }

    public void setManagerName(String managerName) {
        this.managerName = StringUtils.trim(managerName);
    }

    public void setManagerCardType(String managerCardType) {
        this.managerCardType = StringUtils.trim(managerCardType);
    }

    public void setManagerIdCard(String managerIdCard) {
        this.managerIdCard = StringUtils.trim(managerIdCard);
    }

    public void setManagerPhone(String managerPhone) {
        this.managerPhone = StringUtils.trim(managerPhone);
    }

    public void setProxyManagerName(String proxyManagerName) {
        this.proxyManagerName = StringUtils.trim(proxyManagerName);
    }

    public void setProxyManagerCardType(String proxyManagerCardType) {
        this.proxyManagerCardType = StringUtils.trim(proxyManagerCardType);
    }

    public void setProxyManagerIdCard(String proxyManagerIdCard) {
        this.proxyManagerIdCard = StringUtils.trim(proxyManagerIdCard);
    }

    public void setProxyManagerPhone(String proxyManagerPhone) {
        this.proxyManagerPhone = StringUtils.trim(proxyManagerPhone);
    }

    public void setBankName(String bankName) {
        this.bankName = StringUtils.trim(bankName);
    }

    public void setBankBranchName(String bankBranchName) {
        this.bankBranchName = StringUtils.trim(bankBranchName);
    }

    public void setBankNo(String bankNo) {
        this.bankNo = StringUtils.trim(bankNo);
    }

    public void setTaxNum(String taxNum) {
        this.taxNum = StringUtils.trim(taxNum);
    }

    public void setCompanyName(String companyName) {
        this.companyName = StringUtils.trim(companyName);
    }
}
