/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: Context.java   2020-09-22 14-13-45
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import com.alibaba.excel.metadata.CellData;
import com.xforceplus.business.excel.*;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.web.utils.JsonUtils;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <p>
 * Title: 导入上下文
 * </p>
 * <p>
 * Description: 导入上下文 （包括业务）
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-22 14-13-45
 * @since V1.0
 */
@Setter
@Getter
@ToString
public class Context {
    private static final Logger logger = LoggerFactory.getLogger(Context.class);

    /**
     * 导入对象
     */
    private ExcelFileDTO fileDTO;
    /**
     * 导入处理类型
     */
    private BusinessType businessType;
    /**
     * 工作薄
     */
    private ExcelBook excelBook;
    /**
     * 实例数据
     */
    private Class<?> clazz;
    /**
     * 指定FilePath;
     */
    private String sourceFilePath;
    /**
     * 上传文件的InputStream
     */
    private InputStream is;

    /**
     * 目标文件路径
     */
    private String targetFilePath;

    /**
     * 结果数据，对应sheetName,及sheet的数据
     */
    private Map<String, MessageRow> messageRows;

    /**
     * 多表对应文件头信息
     */
    private Map<String, Map<Integer, CellData>> headMaps;

    /**
     * Excel读取数据
     */
    private SimpleExcelReader simpleExcelReader;

    /**
     * Excel读取数据
     */
    private SimpleExcelWriter simpleExcelWriter;
    /**
     * 参数
     */
    private Map<String, Object> params;

    /**
     * 导出最大行数
     */
    private Integer maxSize;

    /**
     * 用户上下文
     */
    private IAuthorizedUser authorizedUser;


    private Map<String, Integer> sheetHeaderMap;


    /**
     * 构造函数
     *
     * @param fileDTO
     * @param businessType
     * @param excelBook
     * @param clazz
     * @param sourceFilePath
     * @param targetFilePath
     * @param authorizedUser IAuthorizedUser
     */
    private Context(ExcelFileDTO fileDTO, BusinessType businessType, ExcelBook excelBook, Class<?> clazz, String sourceFilePath,
                    String targetFilePath, IAuthorizedUser authorizedUser, Map<String, Object> params, Integer maxSize, Map<String, Integer> sheetHeaderMap) {
        this.fileDTO = fileDTO;
        this.businessType = businessType;
        this.excelBook = excelBook;
        this.clazz = clazz;
        this.sourceFilePath = sourceFilePath;
        this.messageRows = new ConcurrentHashMap<>(6);
        this.headMaps = new ConcurrentHashMap<>(6);
        this.targetFilePath = targetFilePath;
        this.authorizedUser = authorizedUser;
        this.params = params;
        this.maxSize = maxSize;
        this.sheetHeaderMap = sheetHeaderMap;
    }

    /**
     * 创建对象
     *
     * @return Builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * 获取SimpleExcelReader
     *
     * @return
     */
    public SimpleExcelReader getSimpleExcelReader() {
        return simpleExcelReader;
    }

    /**
     * 设置SimpleExcelReader
     *
     * @param simpleExcelReader
     */
    public void setSimpleExcelReader(SimpleExcelReader simpleExcelReader) {
        this.simpleExcelReader = simpleExcelReader;
    }

    /**
     * 获取单
     *
     * @param sheetName
     * @return Map<Integer, CellData>
     */
    public Map<Integer, CellData> getHeadMap(String sheetName) {
        return headMaps.get(sheetName);
    }

    /**
     * SheetName
     *
     * @param sheetName
     * @param headMap
     */
    public void setHeadMap(String sheetName, Map<Integer, CellData> headMap) {
        this.headMaps.put(sheetName, headMap);
    }

    public List<ExcelSheet> excelSheets() {
        return excelBook.getExcelSheets();
    }

    /**
     * 最大页数
     *
     * @return Long 最大页数
     */
    public int getMaxPageSize() {
        int pageSize = this.maxSize / ExcelFile.PAGE_SIZE;
        if (pageSize > 0) {
            pageSize = pageSize - 1;
        }
        return pageSize;
    }

    /**
     * 获取属性值
     *
     * @param key 参数Key
     * @return Value
     */
    public <T> T getParam(String key, Class<T> clazz) {
        if (this.fileDTO.getRerun()) {
            String jsonStr = JsonUtils.toJson(this.params.get(key));
            return JsonUtils.fromJson(jsonStr, clazz);
        }
        return (T) this.params.get(key);
    }

    /**
     * 添加消息
     *
     * @param sheetName
     */
    public void messageRow(String sheetName, MessageRow messageRow) {
        if (this.messageRows.containsKey(sheetName)) {
            MessageRow messageRowTemp = this.messageRows.get(sheetName);
            messageRowTemp.putAll(messageRow.getRows());
            this.messageRows.put(sheetName, messageRowTemp);
        } else {
            this.messageRows.put(sheetName, messageRow);
        }
    }

    /**
     * 添加消息
     *
     * @param sheetName
     */
    public MessageRow messageRow(String sheetName, Integer index, String message) {
        MessageRow existMessageRow;
        if (this.messageRows.containsKey(sheetName)) {
            existMessageRow = this.messageRows.get(sheetName);
        } else {
            existMessageRow = new MessageRow(sheetName);
        }
        existMessageRow.fail(index, message);
        this.messageRows.put(sheetName, existMessageRow);
        return existMessageRow;
    }

    public void success(String sheetName, Integer rowIndex) {
        MessageRow existMessageRow = this.messageRows.get(sheetName);
        if (existMessageRow == null) {
            String message = "sheet(" + sheetName + "不存在";
            logger.warn(message);
        } else {
            existMessageRow.success(rowIndex);
        }
    }


    public static final class Builder {
        /**
         * 参数
         */
        private final Map<String, Object> params;
        /**
         * 导入对象
         */
        private ExcelFileDTO fileDTO;
        /**
         * 导入处理类型
         */
        private BusinessType businessType;
        /**
         * 工作薄
         */
        private ExcelBook excelBook;
        /**
         * 实例数据
         */
        private Class<?> clazz;
        /**
         * 指定FilePath;
         */
        private String sourceFilePath;
        /**
         * 目标文件路径
         */
        private String targetFilePath;
        /**
         * 用户上下文
         */
        private IAuthorizedUser authorizedUser;
        /**
         * 导出最大行数
         */
        private Integer maxSize;

        private Map<String, Integer> sheetHeaderMap;

        private Builder() {
            params = new HashMap<>(2);
        }

        public Builder sheetHeaderMap(Map<String, Integer> sheetHeaderMap) {
            this.sheetHeaderMap = sheetHeaderMap;
            return this;
        }

        public Builder fileDTO(ExcelFileDTO fileDTO) {
            this.fileDTO = fileDTO;
            return this;
        }

        public Builder businessType(BusinessType businessType) {
            this.businessType = businessType;
            return this;
        }

        public Builder excelBook(ExcelBook excelBook) {
            this.excelBook = excelBook;
            return this;
        }

        public Builder clazz(Class<?> clazz) {
            this.clazz = clazz;
            return this;
        }

        public Builder sourceFilePath(String sourceFilePath) {
            this.sourceFilePath = sourceFilePath;
            return this;
        }

        public Builder targetFilePath(String targetFilePath) {
            this.targetFilePath = targetFilePath;
            return this;
        }


        public Builder authorizedUser(IAuthorizedUser authorizedUser) {
            this.authorizedUser = authorizedUser;
            return this;
        }

        public Builder params(Map<String, Object> params) {
            if (null != params) {
                this.params.putAll(params);
            }

            return this;
        }

        public Builder params(String key, Object value) {
            this.params.put(key, value);
            return this;
        }

        /**
         * 导出最大限制行数，建议使用默认配置
         *
         * @param maxSize 最大限制行数
         * @return Builder
         */
        public Builder maxSize(Integer maxSize) {
            if (maxSize > 0) {
                throw new IllegalArgumentException("导出限制最大行数应该大于0");
            }
            this.maxSize = maxSize;
            return this;
        }


        public Context build() {
            //创建目录输入文件
            if (StringUtils.isBlank(this.targetFilePath)) {
                this.targetFilePath = ExcelFile.createExcelFilePath();
            }
            //默认使用Excel配置
            if (this.maxSize == null) {
                this.maxSize = ExcelConfig.getLimitMaxSize();
            }
            return new Context(fileDTO, businessType, excelBook, clazz, sourceFilePath, targetFilePath, authorizedUser, this.params, this.maxSize, this.sheetHeaderMap);
        }
    }
}
