/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelImportProcessImpl.java   2020-09-22 15-09-56
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import com.google.common.base.Joiner;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.excel.AbstractExcelProcess;
import com.xforceplus.business.excel.ExcelBook;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelProcess;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.reader.MessageRow.Row;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.dao.ExcelFileStoreDao;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.text.MessageFormat;
import java.util.*;
import java.util.Map.Entry;

import static com.xforceplus.business.excel.ExcelFile.FILE_NAME_EXT;
import static com.xforceplus.business.excel.ExcelFile.IMPORT_RESULT_TEXT;
import static com.xforceplus.business.excel.file.ExcelFileDTO.ResultState.*;


/**
 * <p>
 * Title: ExcelImport读取实现处理类
 * </p>
 * <p>
 * Description: ExcelImport读取实现处理类
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-22 15-09-56
 * @since V1.0
 */
@Slf4j
@Component("excelImportProcess")
public class ExcelImportProcessImpl extends AbstractExcelProcess implements ExcelProcess {
    /**
     * 处理前数据
     */
    private static final Logger logger = LoggerFactory.getLogger(ExcelImportProcessImpl.class);


    public ExcelImportProcessImpl(ExcelFileStoreDao excelFileStoreDao) {
        super(excelFileStoreDao);
    }

    /**
     * 文件处理前
     *
     * @param context 处理上下文
     */
    @Override
    public void before(Context context) {
        logger.info("Excel Import process before...");
        //设置线程用户上下文
        this.beforeSet(context);
        ExcelFileDTO excelFileDTO = context.getFileDTO();
        excelFileDTO.setBeginDate(new Date());
        SimpleExcelReader simpleExcelReader = null;
        try {
            //构建Book
            ExcelBook excelBook = ExcelBook
                    .builder()
                    .excelSheets(context.getBusinessType().sheets())
                    .sourceFileId(context.getFileDTO().getSourceFileId())
                    .sourceFileName(context.getSourceFilePath())
                    .userId(context.getFileDTO().getUserId())
                    .tenantId(context.getFileDTO().getTenantId())
                    .name(context.getBusinessType().getBusinessName())
                    .build();
            //设置ExcelBook
            context.setExcelBook(excelBook);
            //本地文件
            excelFileDTO.setSourceFilePath(context.getSourceFilePath());
            //处理中心
            excelFileDTO.setStatus(ExcelFileDTO.Status.PROCESSING);
            //读取次数加一次
            excelFileDTO.setReadTime(excelFileDTO.getReadTime() + 1);
            //读取数据
            simpleExcelReader = SimpleExcelReader
                    .builder()
                    .excelBook(excelBook)
                    .build();
            //写入数据
            context.setSimpleExcelReader(simpleExcelReader);
        } catch (Exception e) {
            if (simpleExcelReader != null) {
                simpleExcelReader.setException(new IllegalArgumentException("Excel文件解析错误，请稍后重试"));
            }
        }
        try {
            this.beforeUpdate(excelFileDTO);
        } catch (Exception e) {
            if (simpleExcelReader != null) {
                simpleExcelReader.setException(new IllegalArgumentException("Excel文件解析错误，请稍后重试"));
            }
        }
    }

    /**
     * 读取数据
     *
     * @param context 上下文
     */
    @Override
    public void process(Context context) {
        //处理之前信息
        this.before(context);
        //读取数据
        this.doRead(context);
        //读取数据处理错误信息
        this.after(context);
    }

    /**
     * 实现doRead
     *
     * @param context Context
     */
    protected void doRead(Context context) {
        //获取处理器
        ExcelReaderService excelReaderService = ExcelReaderUtils.getExcelReader(context.getBusinessType());
        try {
            if (excelReaderService != null) {
                excelReaderService.importExcel(context);
            }
        } catch (RuntimeException e) {
            log.warn(e.getMessage(), e.getCause());
            context.getExcelBook().setException(e);
        }
    }


    /**
     * 文件处理前
     *
     * @param context 处理上下文
     */
    @Override
    public void after(Context context) {

        ExcelBook excelBook = context.getExcelBook();
        logger.info("保存信息");
        context.getMessageRows().forEach((key, value) ->
                logger.info("key:{},value:{}", key, value)
        );
        if (StringUtils.isEmpty(context.getSourceFilePath())) {
            context.setSourceFilePath(excelBook.getSourcePath().toString());
        }
        //回写数据，同是返回targetFilePath
        String targetFilePath = ExcelMessageUtils.write(context);
        logger.info("targetFilePath:{}", targetFilePath);
        excelBook.setTargetFileName(targetFilePath);
        //分为单个sheet或多sheet统计
        Map<String, Summary> bookSummary = this.calcBookSummary(context.getMessageRows());
        //计算总表
        Summary totalSummary = this.collectTotalSummary(bookSummary);

        //处理后续流程
        SimpleExcelReader simpleExcelReader = context.getSimpleExcelReader();
        simpleExcelReader.setExcelBook(excelBook);
        simpleExcelReader.param(ExcelFile.TOTAL_SIZE, totalSummary.getTotal());
        simpleExcelReader.param(ExcelFile.SUCCESS_SIZE, totalSummary.getSuccess());
        simpleExcelReader.param(ExcelFile.FAIL_SIZE, totalSummary.getFail());
        simpleExcelReader.param(ExcelFile.SHEET_MESSAGE_KEY, totalSummary.getMessage());
        simpleExcelReader.param(ExcelFile.SHEET_SIZE, bookSummary.size());
        simpleExcelReader.param(ExcelFile.RESULT_STATE, this.isPartSuccess(totalSummary));
        simpleExcelReader.finish();
        ExcelFileDTO excelFileDTO = context.getFileDTO();
        //保存数据
        excelFileDTO.setTargetFileId(simpleExcelReader.getExcelBook().getTargetFileId());
        //导出文件
        excelFileDTO.setTargetFileName(context.getBusinessType().getBusinessName() + IMPORT_RESULT_TEXT + FILE_NAME_EXT);
        excelFileDTO.setEndDate(new Date());
        excelFileDTO.setCostTime(this.costTime(excelFileDTO.getBeginDate(), excelFileDTO.getEndDate()));
        excelFileDTO.setTotalSize(totalSummary.getTotal());
        excelFileDTO.setSuccessSize(totalSummary.getSuccess());
        //保存每个Sheet导入成功的数据
        if (bookSummary.size() > 0) {
            excelFileDTO.setMessage(totalSummary.getMessage());
        }
        excelFileDTO.setResultState(this.isPartSuccess(totalSummary));
        excelFileDTO.setStatus(ExcelFileDTO.Status.FINISHED);
        //数据统计
        this.afterUpdate(excelFileDTO);
        //记录是否导入成功；
        this.clear();
    }

    /**
     * 判断是否为部份成功
     *
     * @param summary Summary
     * @return (全部成功 ， 部分成功 ， 失败)
     */
    protected Integer isPartSuccess(Summary summary) {
        if (summary.getSuccess().equals(summary.getTotal())) {
            return SUCCEED;
        } else if (summary.getSuccess() > 0 && summary.getSuccess() < summary.getTotal()) {
            return PART_SUCCEED;
        }
        return FAILED;
    }

    /**
     * 构建Book汇总数据
     *
     * @param summaryMap summaryMap
     * @return Summary 统计结果
     */
    protected Summary collectTotalSummary(Map<String, Summary> summaryMap) {
        Integer total = 0;
        Integer success = 0;
        Integer fail = 0;
        List<String> sheetMessages = new ArrayList<>(summaryMap.size());
        for (Entry<String, Summary> entry : summaryMap.entrySet()) {
            Summary summary = entry.getValue();
            total += summary.total;
            success += summary.success;
            fail += summary.fail;
            sheetMessages.add(summary.getMessage());
        }
        String message = Joiner.on(Separator.SEMICOLON).join(sheetMessages);
        return new Summary(StringUtils.EMPTY, total, success, fail, message);
    }

    /**
     * 计算Book 各Sheet导入数量
     *
     * @param messageRowMap Map<String, MessageRow>
     * @return Map<String, Summary>
     */
    protected Map<String, Summary> calcBookSummary(Map<String, MessageRow> messageRowMap) {
        Map<String, Summary> summaryMap = new HashMap<>(messageRowMap.size());
        for (Entry<String, MessageRow> rows : messageRowMap.entrySet()) {
            summaryMap.put(rows.getKey(), this.calcSheetSummary(rows.getKey(), rows.getValue().getRows()));
        }
        return summaryMap;
    }

    /**
     * 计算Sheet
     *
     * @param sheetName SheetName
     * @param rows      Map<Integer, Row>
     * @return Summary
     */
    protected Summary calcSheetSummary(String sheetName, Map<Integer, Row> rows) {
        int total = 0;
        int success = 0;
        logger.info("sheetName:{},result-row size:{}", rows.size(), sheetName);
        for (Entry<Integer, Row> row : rows.entrySet()) {
            total++;
            if (MessageRow.SUCCESS.equals(row.getValue().getStatus())) {
                success++;
            }
        }
        int fail = total - success;
        Summary summary = new Summary(sheetName, total, success, fail);
        summary.setMessage(this.sheetMessage(summary));
        return summary;
    }

    /**
     * 计算SheetName消息
     *
     * @param summary Summary
     * @return String
     */
    protected String sheetMessage(Summary summary) {
        try {
            return MessageFormat.format(ExcelFile.SHEET_MESSAGE, summary.getSheetName(),
                    String.valueOf(summary.getTotal()),
                    String.valueOf(summary.getSuccess()),
                    String.valueOf(summary.getFail()));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return StringUtils.EMPTY;
    }


    /**
     * 内部统计类
     */
    @Setter
    @Getter
    @ToString
    protected static class Summary {
        /**
         * 表格名称
         */
        private String sheetName;
        /**
         * 失败记录
         */
        private Integer total;

        /**
         * 成功的记录
         */
        private Integer success;
        /**
         * 失败记录数
         */
        private Integer fail;
        /**
         * 消息
         */
        private String message;

        /**
         * 构建函数
         *
         * @param sheetName SheetName
         * @param total     总记录数
         * @param success   成功数
         * @param fail      失败数
         */
        public Summary(String sheetName, Integer total, Integer success, Integer fail) {
            this.sheetName = sheetName;
            this.total = total;
            this.success = success;
            this.fail = fail;
        }

        /**
         * 有消息构建函数
         *
         * @param sheetName SheetName
         * @param total     Total
         * @param success   成功数
         * @param fail      失败数
         * @param message   消息
         */
        public Summary(String sheetName, Integer total, Integer success, Integer fail, String message) {
            this(sheetName, total, success, fail);
            this.message = message;
        }
    }
}
