/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelMessageUtils.java   2020-09-23 15-42-26
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.xssf.usermodel.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Map;

/**
 * <p>
 * Title: 数据处理结果回写
 * </p>
 * <p>
 * Description: 数据处理结果回写
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-23 15-42-26
 * @since V1.0
 */
public final class ExcelMessageUtils {

    private ExcelMessageUtils(){

    }

    /**
     * 导入数据状态名称 ：{@value}
     */
    public static final String IMPORT_STATUS_COL_NAME = "导入数据状态";
    /**
     * 信息提示:{@value}
     */
    public static final String IMPORT_MESSAGE_COL_NAME = "错误提示";
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(ExcelMessageUtils.class);

    /**
     * 将上下文中错误消息回写到Excel中
     * @param context 下下文
     * @return String 文件路径
     */
    public static String write(Context context) {

        try (FileInputStream fileInputStream = new FileInputStream(context.getSourceFilePath());
             XSSFWorkbook workbook = new XSSFWorkbook(fileInputStream)) {
            for (Map.Entry<String, MessageRow> entry : context.getMessageRows().entrySet()) {
                XSSFSheet sheet = workbook.getSheet(entry.getKey());
                if (sheet == null) {
                    continue;
                }
                //处理异常时候，异常的位置
                Map<String, Integer> sheetHeaderMap = context.getSheetHeaderMap();
                Integer sheetHeaderNum = 0;
                if (sheetHeaderMap != null) {
                    sheetHeaderNum = sheetHeaderMap.get(sheet.getSheetName());
                    //编号从0开始，要减去一
                    if (sheetHeaderNum != null && sheetHeaderNum > 0) {
                        sheetHeaderNum = sheetHeaderNum - 1;
                    } else {
                        sheetHeaderNum = 0;
                    }
                }
                //设置样式
                XSSFCellStyle cellStyle = getCellStyle(workbook);
                //将错误消息写入到各Sheet中

                writeSheet(sheet, entry, context.getHeadMap(entry.getKey()).size(), cellStyle, sheetHeaderNum);
            }
            //写入到指定文件
            try (FileOutputStream out = new FileOutputStream(context.getTargetFilePath())) {
                out.flush();
                workbook.write(out);
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return context.getTargetFilePath();
    }

    /**
     * 设置
     *
     * @param workbook
     * @return
     */
    private static XSSFCellStyle getCellStyle(XSSFWorkbook workbook) {
        XSSFCellStyle cellStyle = workbook.createCellStyle();
        XSSFFont font = workbook.createFont();
        font.setColor(IndexedColors.RED.getIndex());
        cellStyle.setFillForegroundColor(IndexedColors.YELLOW.getIndex());
        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        cellStyle.setFont(font);
        cellStyle.setWrapText(true);
        return cellStyle;
    }

    /**
     * @param sheet    XSSFSheet
     * @param entry    写入数据
     * @param headSize 指定列
     */
    private static void writeSheet(XSSFSheet sheet, Map.Entry<String, MessageRow> entry, Integer headSize, XSSFCellStyle cellStyle, Integer headerNum) {
        //获取Header的行号
        XSSFRow hssfRow = sheet.getRow(headerNum);

        hssfRow.createCell(headSize).setCellValue(IMPORT_STATUS_COL_NAME);
        hssfRow.createCell(headSize + 1).setCellValue(IMPORT_MESSAGE_COL_NAME);
        for (Map.Entry<Integer, MessageRow.Row> rowEntry : entry.getValue().getRows().entrySet()) {
            //按Row 错误消息写入
            hssfRow = sheet.getRow(rowEntry.getKey());
            // 空行跳过
            if (null == hssfRow) {
                continue;
            }
            //错误信息设置为红色的
            if (MessageRow.FAIL.equals(rowEntry.getValue().getStatus())) {
                XSSFCell cell = hssfRow.createCell(headSize);
                cell.setCellStyle(cellStyle);
                cell.setCellValue(rowEntry.getValue().getStatus());
                XSSFCell cellMessage = hssfRow.createCell(headSize + 1);
                cellMessage.setCellStyle(cellStyle);
                cellMessage.setCellValue(rowEntry.getValue().getMessage());
            } else {
                //按Row 错误消息写入
                hssfRow.createCell(headSize).setCellValue(rowEntry.getValue().getStatus());
                hssfRow.createCell(headSize + 1).setCellValue(rowEntry.getValue().getMessage());
            }
        }
    }
}