/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: FileExcelReaderHandler.java   2020-09-22 16-19-40
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import com.xforceplus.business.excel.ExcelBook;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelHandler;
import com.xforceplus.utils.filetransfer.FileTransferUtilsService;
import io.geewit.web.utils.SpringContextUtil;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-22 16-19-40
 * @since V1.0
 */
public class FileExcelReaderHandler implements ExcelHandler {
    /**
     * 日志处理
     */
    private static final Logger logger = LoggerFactory.getLogger(FileExcelReaderHandler.class);

    /**
     *  处理前
     * @param excelBook Excel上下文
     */
    @Override
    public void before(ExcelBook excelBook) {

        if (StringUtils.isNotBlank(excelBook.getSourceFileName())) {
            logger.info("文件已在本地存在");
            return;
        }
        //根据FileId
        FileTransferUtilsService fileTransferUtilsService = SpringContextUtil.getBean(FileTransferUtilsService.class);
        String sourceFilePath = ExcelFile.createExcelFilePath();
        excelBook.setSourceFileName(sourceFilePath);
        try {
            logger.info("sourceFilePath:{}", sourceFilePath);
            fileTransferUtilsService.download(sourceFilePath, excelBook.getSourceFileId(), excelBook.getUserId(), excelBook.getTenantId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            excelBook.setException(new IllegalArgumentException("上传文件保存失败，请稍后重试"));
        }
        //检查文件大小
        this.checkSize(new File(sourceFilePath));
        //下载文件到本地
        logger.info("从OSS下载文件");
    }


    /**
     * 处理后
     *
     * @param excelBook Excel上下文
     */
    @Override
    public void after(ExcelBook excelBook) {
        logger.info("开始上传文件...");
        //获取上传文件实例
        FileTransferUtilsService fileTransferUtilsService = SpringContextUtil.getBean(FileTransferUtilsService.class);

        File file = excelBook.getTargetPath().toFile();
        String filePath = file.getPath();
        //文件检查是否为空
        try {
            this.checkSize(file);
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
            excelBook.setException(e);
            return;
        }
        try {
            logger.info("上传文件路径是：{}", filePath);
            Long fileId = fileTransferUtilsService.upload(filePath, excelBook.getUserId(), excelBook.getTenantId());
            logger.info("上传成功文件ID:{}", fileId);
            excelBook.setTargetFileId(fileId);
        } catch (FileNotFoundException e) {
            logger.error("错误信息上传文件服务失败,{}", filePath, e);
            excelBook.setException(new IllegalArgumentException("Excel解析错误提示文件，保存失败"));
        } finally {
            //删除文件
            FileUtils.deleteQuietly(file);
            //删除原文件
            FileUtils.deleteQuietly(excelBook.getSourcePath().toFile());
        }
        logger.info("结束上传文件...");
    }


    /**
     * 检查文件大小
     *
     * @param file 文件路径
     */
    protected void checkSize(File file) {
        long size = FileUtils.sizeOf(file);
        logger.info("filePath:{},fileSize:{}", file.getPath(), size);
        if (size == 0) {
            throw new IllegalStateException("文件为空");
        }
    }
}
