/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: MessageExcelReaderHandler.java   2020-09-22 16-15-56
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import com.alibaba.excel.exception.ExcelAnalysisException;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.excel.*;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.notice.dto.ExcelNoticeConfig;
import com.xforceplus.business.notice.dto.NoticeMessage;
import com.xforceplus.business.notice.service.NoticeService;
import com.xforceplus.utils.StrUtils;
import io.geewit.web.utils.SpringContextUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Title:   ExcelReader 消息处理类
 * </p>
 * <p>
 * Description: ExcelReader 消息处理类
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-22 16-15-56
 * @since V1.0
 */
public class MessageExcelReaderHandler implements ExcelHandler {
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(MessageExcelWriterHandler.class);
    /**
     * 消息服务
     */
    private NoticeService noticeService = SpringContextUtil.getBean(NoticeService.class);

    /**
     * 处理前
     *
     * @param excelBook Excel上下文
     */
    @Override
    public void before(ExcelBook excelBook) {
        logger.info("{}发送消息通知", excelBook.getName());
        String noticeKey = ExcelNoticeConfig.IMP_START;
        Object[] params = new Object[]{excelBook.getName()};
        if (excelBook.getException() != null) {
            logger.error(excelBook.getException().getMessage(), excelBook.getException().getCause());
            noticeKey = ExcelNoticeConfig.IMP_EXCEPTION;
            //处理导入失败情况
            params = new Object[]{excelBook.getName(), excelBook.getException().getMessage()};
        }
        //发送消息
        this.sendNoticeMessage(excelBook, noticeKey, params);
    }

    /**
     * 处理后
     *
     * @param excelBook Excel上下文
     */
    @Override
    public void after(ExcelBook excelBook) {
        logger.info("{}发送消息通知", excelBook.getName());
        String noticeKey = ExcelNoticeConfig.IMP_SUCCESS;
        Object successSize = this.dataSize(excelBook, ExcelFile.SUCCESS_SIZE);
        Object failSize = this.dataSize(excelBook, ExcelFile.FAIL_SIZE);
        Object totalSize = this.dataSize(excelBook, ExcelFile.TOTAL_SIZE);
        String downloadUrl = ExcelConfig.getDownloadUrl();
        logger.info("发送消息通知-after,downloadUrl:{}", downloadUrl);
        Object[] params = new Object[]{excelBook.getName(), totalSize, successSize, downloadUrl + excelBook.getTargetFileId(), String.valueOf(excelBook.getTargetFileId())};
        Integer resultState = this.getResultState(excelBook, ExcelFile.RESULT_STATE);
        //部分成功的
        Exception e = excelBook.getException();
        if (e != null) {
            String message = e.getMessage();
            logger.error(message, e.getCause());
            noticeKey = ExcelNoticeConfig.IMP_EXCEPTION;
            if (e instanceof ExcelAnalysisException) {
                String messageZh = "Excel解析 数据转换失败为:\"" + StrUtils.matcherZh(e.getMessage(), Separator.SPACE) + "\"";
                //处理导入失败情况
                params = new Object[]{excelBook.getName(), messageZh};
            } else {
                if (StringUtils.isNotEmpty(message)) {
                    message = StrUtils.matcherZh(message, Separator.SPACE);
                }
                if (StringUtils.isBlank(message)) {
                    message = "系统发生错误，请稍后重试或联系运营人员";
                }
                //处理导入失败情况
                params = new Object[]{excelBook.getName(), message};
            }

        } else if (ExcelFileDTO.ResultState.PART_SUCCEED == resultState) {
            //部分成功的消息
            noticeKey = ExcelNoticeConfig.IMP_PART_SUCCESS;
            params = new Object[]{excelBook.getName(), totalSize, successSize, failSize, downloadUrl + excelBook.getTargetFileId(), String.valueOf(excelBook.getTargetFileId())};
        } else if (ExcelFileDTO.ResultState.FAILED == resultState) {
            //完成失败的情况
            noticeKey = ExcelNoticeConfig.IMP_FAILED;
            params = new Object[]{excelBook.getName(), totalSize, failSize, downloadUrl + excelBook.getTargetFileId(), String.valueOf(excelBook.getTargetFileId())};
        }
        //发送消息
        this.sendNoticeMessage(excelBook, noticeKey, params);
    }


    /**
     * 发送消息
     *
     * @param excelBook 工作薄
     * @param noticeKey NoticeKey
     * @param params    参数
     */
    protected void sendNoticeMessage(ExcelBook excelBook, String noticeKey, Object[] params) {
        try {
            NoticeMessage noticeMessage = NoticeMessage.builder()
                    .userId(excelBook.getUserId())
                    .tenantId(excelBook.getTenantId())
                    .key(noticeKey)
                    .params(params)
                    .build();
            //发送消息通知
            this.sendNotice(noticeMessage);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    /**
     * 获取处理完成状态
     *
     * @param excelBook ExcelBook
     * @param key       参数KEY
     * @return Integer
     */
    public Integer getResultState(ExcelBook excelBook, String key) {
        Integer state = -1;
        try {
            state = Integer.parseInt(excelBook.getParams().get(key).toString());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return state;
    }


    /**
     * 获取成功记录数
     *
     * @param excelBook ExcelBook
     * @param key       参数KEY
     * @return String Key
     */
    private String dataSize(ExcelBook excelBook, String key) {
        if (excelBook.getParams().containsKey(key)) {
            return String.valueOf(excelBook.getParams().get(key));
        }
        return String.valueOf(ExcelFile.ZERO_SIZE);
    }

    /**
     * 发送消息
     *
     * @param noticeMessage NoticeMessage
     */
    protected void sendNotice(NoticeMessage noticeMessage) {
        this.noticeService.sendExcelProcessNotice(noticeMessage.getExcelProcessNotice());

    }
}
