/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: SimpleDataListener.java   2020-09-23 01-38-00
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.metadata.Cell;
import com.alibaba.excel.metadata.CellData;
import com.xforceplus.business.excel.DataRow;
import com.xforceplus.business.excel.ExcelValidator;
import com.xforceplus.utils.BatchUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * <p>
 * Title: SimpleDataReadListener
 * </p>
 * <p>
 * Description: SimpleDataReadListener
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-23 01-38-00
 * @since V1.0
 */
@Slf4j
public class SimpleDataReadListener {
    private SimpleDataReadListener(){

    }
    /**
     * 处理BatchSize
     */
    public static final int BATCH_SIZE = 3000;

    /**
     * 构建S
     *
     * @param context  系统上下文
     * @param consumer 消息处理
     * @return AnalysisEventListener
     */
    public static <T> AnalysisEventListener<T> listener(Context context, Consumer<List<T>> consumer) {
        return new AnalysisEventListener<T>() {
            /**
             * 数据
             */
            private final List<T> list = new ArrayList<>();
            /**
             * 默认为3000
             */
            private final int batchSize = context.getFileDTO() != null ? context.getFileDTO().getBatchSize() : BATCH_SIZE;

            /**
             * When analysis one row trigger invoke function.
             *
             * @param data    one row value. Is is same as {@link AnalysisContext#readRowHolder()}
             * @param analysisContext  AnalysisContext
             */
            @SneakyThrows
            @Override
            public void invoke(T data, AnalysisContext analysisContext) {

                if (data instanceof DataRow && !isEmptyRow(analysisContext)) {
                    Integer rowIndex = analysisContext.readRowHolder().getRowIndex();
                    ((DataRow) data).setRowIndex(rowIndex);
                    list.add(data);
                    //全局校验
                    if (list.size() >= batchSize) {
                        this.process(list);
                    }
                }
            }

            /**
             * 获取文件头信息
             * @param headMap Map<Integer, CellData>
             * @param analysisContext AnalysisContext
             */
            @Override
            public void invokeHead(Map<Integer, CellData> headMap, AnalysisContext analysisContext) {
                context.setHeadMap(analysisContext.readSheetHolder().getSheetName(), headMap);
                super.invokeHead(headMap, analysisContext);
            }

            /**
             * 读取异常处理
             * @param exception Exception
             * @param context AnalysisContext
             * @throws Exception Exception
             */
            @Override
            public void onException(Exception exception, AnalysisContext context) throws Exception {
                if (list.size() > 0) {
                    this.process(list);
                }
                super.onException(exception, context);
            }

            /**
             * Excel读取完成后操作
             * if have something to do after all analysis
             * @param context AnalysisContext
             */
            @SneakyThrows
            @Override
            public void doAfterAllAnalysed(AnalysisContext context) {
                if (list.size() > 0) {
                    this.process(list);
                }
            }


            /**
             * 优化判断数据行为空方法，不使用反射
             * @param analysisContext
             * @return
             */
            private boolean isEmptyRow(AnalysisContext analysisContext) {
                Map<Integer, Cell> celMap = analysisContext.readRowHolder().getCellMap();
                Set<Map.Entry<Integer, Cell>> entrySet = celMap.entrySet();
                for (Map.Entry<Integer, Cell> entry : entrySet) {
                    if (!StringUtils.isEmpty(String.valueOf(entry.getValue()))) {
                        return false;
                    }
                }
                return true;
            }

            /**
             * 提高并行度，不等所有的数据都校验完再去处理，一批结束直接走后续流程
             * @param dataList
             */
            @SneakyThrows
            private void process(List<T> dataList) {
                List<T> dataToProcess = new ArrayList<>(dataList);
                CompletableFuture<Void> future = BatchUtils.doBatchAsync(dataToProcess, this::batchValidate);
                future.get();
                consumer.accept(dataToProcess);
                list.clear();
            }

            private void batchValidate(List<T> dataList) {
                dataList.forEach(ExcelValidator::validate);
            }
        };
    }
}
