/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: SimpleExcelReader.java   2020-09-22 15-59-17
 * Author: Evan
 */
package com.xforceplus.business.excel.reader;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.xforceplus.business.excel.ExcelBook;
import com.xforceplus.business.excel.ExcelHandler;
import com.xforceplus.business.excel.SimpleExcelWriter;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * <p>
 * Title: 简单读写操作
 * </p>
 * <p>
 * Description: 简单读写操作
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-22 15-59-17
 * @since V1.0
 */
@Getter
@Setter
@ToString
public class SimpleExcelReader {
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(SimpleExcelReader.class);
    /**
     * 处理级
     */
    private Map<String, ExcelHandler> handlers;
    /**
     * 工作薄对象
     */
    private ExcelBook excelBook;


    /**
     * 填写Sheet表
     */
    private Map<String, WriteSheet> writeSheets;

    /**
     *
     * @param handlers
     * @param excelBook
     */
    public SimpleExcelReader(Map<String, ExcelHandler> handlers, ExcelBook excelBook) {
        this.handlers = handlers;
        this.excelBook = excelBook;
    }

    /**
     * 构建SimpleExcelReader
     * @return Builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * ß
     * @return
     */
    protected SimpleExcelReader start() {
        //开始构建数据
        this.beforeHandler();
        return this;
    }

    /**
     * 设置异常
     * @param e 异常
     */
    public void setException(Exception e) {
        this.excelBook.setException(e);
    }

    /**
     * 添加参数
     *
     * @param key Key
     * @param obj Obj
     */
    public void param(String key, Object obj) {
        this.excelBook.getParams().put(key, obj);
    }

    protected void beforeHandler() {
        if (this.isHandlerEmpty()) {
            return;
        }
        for (Map.Entry<String, ExcelHandler> entry : this.handlers.entrySet()) {
            logger.info("Execute before Handler:{}", entry.getKey());
            entry.getValue().before(this.excelBook);
        }
    }

    /**
     * 读取数据
     *
     * @param clazz 读取Clazz
     */
    public <T> void read(Class<T> clazz, AnalysisEventListener<?> listener) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet().doRead();
    }



    /**
     * 读取数据
     *
     * @param clazz 读取Clazz
     */
    public <T> void read(Class<T> clazz, AnalysisEventListener<?> listener,int headerRowNum) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet().headRowNumber(headerRowNum).doRead();
    }
    /**
     * 同步读取数据
     *
     * @param clazz 读取Clazz
     */
    public <T> void readSync(Class<T> clazz, AnalysisEventListener<?> listener) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet().doReadSync();
    }

    /**
     *  读取数据
     * @param clazz
     * @param sheetName SheetName
     */
    public void read(Class<?> clazz, AnalysisEventListener<?> listener, String sheetName) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet(sheetName).doRead();
    }



    /**
     *  读取数据
     * @param clazz
     * @param sheetName SheetName
     */
    public void read(Class<?> clazz, AnalysisEventListener<?> listener, String sheetName,int headerRowNum) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet(sheetName).headRowNumber(headerRowNum).doRead();
    }
    /**
     *  读取数据
     * @param clazz 实体类
     * @param sheetNo  SheetName
     * @param sheetName SheetName
     */
    public <T> void read(Class<T> clazz, Integer sheetNo, AnalysisEventListener<?> listener, String sheetName) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet(sheetNo, sheetName).doRead();
    }

    /**
     *  读取数据
     * @param clazz 实体类
     * @param sheetNo  SheetName
     * @param sheetName SheetName
     */
    public <T> void read(Class<T> clazz, Integer sheetNo, AnalysisEventListener<?> listener, String sheetName,int headerRowNum) {
        EasyExcel.read(this.getExcelBook().getSourcePath().toFile(), clazz, listener).sheet(sheetNo, sheetName).headRowNumber(headerRowNum).doRead();
    }

    /**
     * 判断处理为空情况
     * @return
     */
    protected boolean isHandlerEmpty() {
        if (CollectionUtils.isEmpty(this.handlers)) {
            logger.warn("SimpleExcelWriter hasn't Excel Handler");
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    /**
     * 执行 afterHandler方法
     */
    protected void afterHandler() {
        if (this.isHandlerEmpty()) {
            return;
        }
        for (Map.Entry<String, ExcelHandler> entry : this.handlers.entrySet()) {
            logger.info("Execute After Handler:{}", entry.getKey());
            entry.getValue().after(this.excelBook);
        }
    }

    /**
     * 结束导入
     */
    public void finish() {
        this.afterHandler();
    }

    /**
     * 创建SimpleExcelReader对象
     */
    public static class Builder {
        /**
         * 默认消息通知 {@value}
         */
        public static final String DEFAULT_MESSAGE_HANDLER = "defaultMessageReaderHandler";
        /**
         * 默认的文件上传
         */
        public static final String DEFAULT_FILE_HANDLER = "defaultFileReaderHandler";
        /**
         * Handler集合
         */
        private Map<String, ExcelHandler> handlers = new LinkedHashMap<>(5);

        /**
         * ExcelBook
         */
        private ExcelBook excelBook;

        Builder() {
        }

        /**
         * 初始化类
         */
        protected void init() {
            //文件上传S3及下载S3文件
            this.handlers.put(DEFAULT_FILE_HANDLER, new FileExcelReaderHandler());
            //默认Handler处理器
            this.handlers.put(DEFAULT_MESSAGE_HANDLER, new MessageExcelReaderHandler());
        }

        /**
         * 添加拦截器
         * @param name HandlerName
         * @param handler ExcelHandler
         * @return Builder Builder
         */
        public Builder handler(String name, ExcelHandler handler) {
            this.handlers.put(name, handler);
            return this;
        }

        /**
         * 设置Excel上下文
         * @param excelBook
         * @return Builder
         */
        public Builder excelBook(ExcelBook excelBook) {
            this.excelBook = excelBook;
            return this;
        }

        /**
         * 创建SimpleExcelBuilder实例 并调用Start方法
         *
         * @return SimpleExcelWriter SimpleExcelWriter
         */
        public SimpleExcelReader build() {
            this.init();
            return new SimpleExcelReader(handlers, excelBook).start();
        }
    }
}
