/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelExcelProcessImpl.java   2020-10-15 10-48-06
 * Author: Evan
 */
package com.xforceplus.business.excel.writer;

import com.xforceplus.business.excel.AbstractExcelProcess;
import com.xforceplus.business.excel.ExcelBook;
import com.xforceplus.business.excel.ExcelProcess;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.dao.ExcelFileStoreDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;

import static com.xforceplus.business.excel.ExcelFile.EXPORT_TEXT;
import static com.xforceplus.business.excel.ExcelFile.FILE_NAME_EXT;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-15 10-48-06
 * @since V1.0
 */
@Service("excelExportProcess")
public class ExcelExportProcessImpl extends AbstractExcelProcess implements ExcelProcess {
    /**
     *
     */
    private static final Logger log = LoggerFactory.getLogger(ExcelExportProcessImpl.class);


    public ExcelExportProcessImpl(ExcelFileStoreDao excelFileStoreDao) {
        super(excelFileStoreDao);
    }

    /**
     * 文件处理前
     * @param context 处理上下文
     */
    @Override
    public void before(Context context) {
        log.info("Excel process before...");
        //设置线程用户上下文
        this.beforeSet(context);
        ExcelFileDTO excelFileDTO = context.getFileDTO();
        excelFileDTO.setBeginDate(new Date());
        SimpleExcelWriter simpleExcelWriter = null;
        try {
            //构建Book
            ExcelBook excelBook = ExcelBook.builder()
                    .name(context.getBusinessType().getBusinessName())
                    .tenantId(excelFileDTO.getTenantId())
                    .userId(excelFileDTO.getUserId())
                    .excelSheets(context.getBusinessType().sheets())
                    .sourceFileName(context.getBusinessType().getExportTemplateFile())
                    .build();
            simpleExcelWriter = SimpleExcelWriter.builder().excelBook(excelBook).build();
            //设置ExcelBook
            context.setExcelBook(excelBook);
            //本地文件
            excelFileDTO.setSourceFilePath(context.getSourceFilePath());
            //处理中心
            excelFileDTO.setStatus(ExcelFileDTO.Status.PROCESSING);
            //读取次数加一次
            excelFileDTO.setReadTime(excelFileDTO.getReadTime() + 1);
            //原始文件
            context.setSourceFilePath(excelBook.getSourcePath().toString());
            //写入数据
            context.setSimpleExcelWriter(simpleExcelWriter);
        } catch (Exception e) {
            if (simpleExcelWriter != null) {
                simpleExcelWriter.setException(new IllegalArgumentException("Excel文件解析错误，请稍后重试"));
            }
        }
        try {
            this.beforeUpdate(excelFileDTO);
        } catch (Exception e) {
            if (simpleExcelWriter != null) {
                simpleExcelWriter.setException(new IllegalArgumentException("Excel文件解析错误，请稍后重试"));
            }
        }
    }


    /**
     * 读取数据
     *
     * @param context 上下文
     */
    @Override
    public void process(Context context) {
        this.before(context);
        //处理Excel导出
        try {
            this.doProcess(context);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            context.getExcelBook().setException(e);
        }
        this.after(context);
    }

    /**
     * 处理导出
     *
     * @param context Context
     */
    private void doProcess(Context context) {
        //获取处理器
        ExcelWriteService excelWriteService = ExcelWriteUtils.getExcelWriter(context.getBusinessType());
        if (excelWriteService != null) {
            excelWriteService.write(context);
        }
    }

    /**
     * 文件处理前
     * @param context 处理上下文
     */
    @Override
    public void after(Context context) {
        ExcelFileDTO excelFileDTO = context.getFileDTO();
        //处理后续流程
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        simpleExcelWriter.finish();

        ExcelBook excelBook = context.getExcelBook();
        log.info("targetFileId:{},targetFilePath:{}", simpleExcelWriter.getExcelBook().getTargetFileId(), excelBook.getTargetFileName());
        //保存数据
        excelFileDTO.setTargetFileId(simpleExcelWriter.getExcelBook().getTargetFileId());
        excelFileDTO.setEndDate(new Date());
        excelFileDTO.setCostTime(this.costTime(excelFileDTO.getBeginDate(), excelFileDTO.getEndDate()));
        excelFileDTO.setStatus(ExcelFileDTO.Status.FINISHED);
        //导出文件
        excelFileDTO.setTargetFileName(context.getBusinessType().getBusinessName() + EXPORT_TEXT + FILE_NAME_EXT);
        //成功
        excelFileDTO.setResultState(ExcelFileDTO.ResultState.SUCCEED);
        //成功记录数
        excelFileDTO.setSuccessSize(0);
        //数据统计
        this.afterUpdate(excelFileDTO);
        //记录是否导入成功；
        this.clear();
    }


}
