/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelFileStoreController.java   2020-10-16 15-57-27
 * Author: Evan
 */
package com.xforceplus.business.file.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelConfig;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ExcelBusinessTypeRespVo;
import com.xforceplus.business.file.controller.vo.ExcelFileStorePageReqVo;
import com.xforceplus.business.file.controller.vo.ExcelFileStorePageRespVo;
import com.xforceplus.business.file.controller.vo.ExcelFileStoreRerunRespVo;
import com.xforceplus.business.file.service.ExcelFileStoreService;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import static com.xforceplus.api.common.Uri.PATH_GLOBAL_PREFIX;
import static com.xforceplus.business.excel.ExcelFile.EXPORT_TEXT;
import static com.xforceplus.business.excel.ExcelFile.IMPORT_TEXT;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-16 15-57-27
 * @since V1.0
 */
@RestController
@RequestMapping(PATH_GLOBAL_PREFIX + "/excelFileStories")
@Api(value = "ExcelFileStore", tags = "ExcelFileStore")
@Slf4j
public class ExcelFileStoreController {

    /**
     * ExcelFileStoreService
     */
    @Resource
    private ExcelFileStoreService excelFileStoreService;


    @Resource
    private ExportFileService exportFileService;


    @Resource
    private ImportFileService importFileService;


    /**
     * 导入与导入文件列表
     * @return ResponseEntity<List < ExcelBusinessTypeRespVo>>
     */
    @ApiOperation(value = "业务类型列表", notes = "获取业务列表")
    @GetMapping(name = "业务类型列表", value = "/businessType")
    public ResponseEntity<List<ExcelBusinessTypeRespVo>> getBusinessType() {
        List<ExcelBusinessTypeRespVo> respVos = new ArrayList<>();
        //导出
        for (ExcelConfigBusinessType type : ExcelConfigBusinessType.values()) {
            respVos.add(new ExcelBusinessTypeRespVo(type.getBusinessName(), type.getName()));
        }
        return ResponseEntity.ok(respVos);
    }

    @ApiOperation(value = "分页列表查询", notes = "根据查询条件分页查询ExcelFile列表")
    @GetMapping(name = "分页列表查询")
    public ResponseEntity<Page<ExcelFileStorePageRespVo>> page(@ApiParam(value = "request") ExcelFileStorePageReqVo query,
                                                               @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        ExcelFileDTO fileDTO = new ExcelFileDTO();
        BeanUtils.copyProperties(query, fileDTO);
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        //按租户查询
        if (authorizedUser != null) {
            fileDTO.setTenantId(authorizedUser.getTenantId());
            fileDTO.setUserId(authorizedUser.getId());
        }
        Page<ExcelFileStore> excelFileStorePage = excelFileStoreService.page(fileDTO, currentPageable);
        Page<ExcelFileStorePageRespVo> pageRespVos = excelFileStorePage.map(e -> {
            ExcelFileStorePageRespVo reqVo = new ExcelFileStorePageRespVo();
            BeanUtils.copyProperties(e, reqVo);
            Optional<BusinessType> excelBusinessType = this.businessTypeValueOf(e.getBusinessType());
            if (excelBusinessType.isPresent()) {
                //导入和导出处理
                if (ExcelFileDTO.ExcelFileType.EXPORT == e.getExcelFileType()) {
                    reqVo.setBusinessType(ExcelConfigBusinessType.valueOf(e.getBusinessType()).getBusinessName() + EXPORT_TEXT);
                } else {
                    reqVo.setBusinessType(excelBusinessType.get().getBusinessName() + IMPORT_TEXT);
                }
            }
            //计算是否有效期 默认为三天有效
            reqVo.setExpired(this.excelFileExpired(e.getCreateTime()));
            return reqVo;
        });
        return ResponseEntity.ok(pageRespVos);
    }

    /**
     * 转换
     *
     * @param businessTypeName
     * @return BusinessType
     */
    private Optional<BusinessType> businessTypeValueOf(String businessTypeName) {
        Optional<BusinessType> businessType = Optional.empty();
        try {
            businessType = Optional.of(ExcelConfigBusinessType.valueOf(businessTypeName));
        } catch (Exception ex) {
            log.warn(ex.getMessage(), ex);
        }
        return businessType;
    }

    /**
     * 根据创建时间判断是否有效，默认为3天有效期
     *
     * @param createTime 文件创建时间
     * @return true 代码有效的, false 代表无效的
     */
    private Boolean excelFileExpired(Date createTime) {
        DateTime dateTime = new DateTime(createTime);
        dateTime = dateTime.plusDays(ExcelConfig.getExpiredDay());
        return !dateTime.isAfterNow();
    }

    /**
     * 重新run
     *
     * @param id
     * @return ResponseEntity<ExcelFileStoreRerunRespVo>
     */
    @ApiOperation(value = "重新执行", notes = "根据ID重新执行导入导出任务")
    @GetMapping(value = "/rerun/{id}", name = "根据ID重新执行导入或导出任务")
    public ResponseEntity<ExcelFileStoreRerunRespVo> rerun(@PathVariable("id") Long id) {
        //首先校验是否有数据
        Optional<ExcelFileStore> optionalExcelFileStore = this.excelFileStoreService.findById(id);
        if (!optionalExcelFileStore.isPresent()) {
            return ResponseEntity.fail("0", "数据不存在");
        }
        ExcelFileStore excelFileStore = optionalExcelFileStore.get();
        //判断是否重试已经超过三次,不处理，从Apollo读取配置，默认为3次
        if (excelFileStore.getReadTime() >= ExcelConfig.getMaxRerunReadTime()) {
            return ResponseEntity.fail("0", "已超过重试3次数");
        }
        //正在执行中
        if (ExcelFileDTO.Status.PROCESSING == excelFileStore.getStatus()) {
            return ResponseEntity.fail("0", "正在执行中...");
        }
        //修改为可以执行状态 处理中
        excelFileStore.setStatus(ExcelFileDTO.Status.PROCESSING);
        excelFileStore.setSuccessSize(0);
        excelFileStore.setTotalSize(0);
        excelFileStore.setCostTime(0L);
        //更新数据
        this.excelFileStoreService.update(excelFileStore);
        ExcelFileDTO excelFileDTO = new ExcelFileDTO();
        BeanUtils.copyProperties(excelFileStore, excelFileDTO);
        //判断是Export&Import然后调用类，执行导入出
        if (ExcelFileDTO.ExcelFileType.EXPORT == excelFileStore.getExcelFileType()) {
            this.exportFileService.rerun(excelFileDTO);
        } else if (ExcelFileDTO.ExcelFileType.IMPORT == excelFileStore.getExcelFileType()) {
            this.importFileService.rerun(excelFileDTO);
        } else {
            return ResponseEntity.fail("0", "没有找相关类型 的数据");
        }
        //再执行
        return ResponseEntity.ok(new ExcelFileStoreRerunRespVo());
    }
}
