/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExcelFileStoreService.java   2020-10-16 15-55-37
 * Author: Evan
 */
package com.xforceplus.business.file.service;

import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.dao.ExcelFileStoreDao;
import com.xforceplus.entity.ExcelFileStore;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.persistence.criteria.Predicate;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <p>
 * Title:  ExcelFileStoreService 存储服务
 * </p>
 * <p>
 * Description: ExcelFileStoreService 存储服务
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-16 15-55-37
 * @since V1.0
 */
@Service
public class ExcelFileStoreService {


    @Resource
    private ExcelFileStoreDao excelFileStoreDao;

    /**
     * 根据ID查询
     *
     * @param id Id
     * @return Optional<ExcelFileStore>
     */
    public Optional<ExcelFileStore> findById(Long id) {
        return excelFileStoreDao.findById(id);
    }

    /**
     * 分页查询
     *
     * @param excelFileDTO
     * @param pageable
     * @return Page<ExcelFileStore>
     */
    public Page<ExcelFileStore> page(ExcelFileDTO excelFileDTO, Pageable pageable) {
        return this.excelFileStoreDao.findAll(QueryHelper.querySpecification(excelFileDTO), pageable);
    }

    /**
     * 更新数据
     *
     * @param excelFileStore
     */
    @Transactional(rollbackFor = Exception.class)
    public void update(ExcelFileStore excelFileStore) {
        this.excelFileStoreDao.saveAndFlush(excelFileStore);
    }

    /**
     * 查询
     */
    public static final class QueryHelper {
        private QueryHelper(){

        }
        public static Specification<ExcelFileStore> querySpecification(ExcelFileDTO excelFileDTO) {
            return (Specification<ExcelFileStore>) (root, criteriaQuery, builder) -> {
                List<Predicate> predicates = new ArrayList<>();
                //租户ID
                predicates.add(builder.equal(root.<Long>get("tenantId"), excelFileDTO.getTenantId()));
                //用户Id
                predicates.add(builder.equal(root.<Long>get("userId"), excelFileDTO.getUserId()));
                //BusinessType
                if (StringUtils.isNotBlank(excelFileDTO.getBusinessType())) {
                    predicates.add(builder.equal(root.<String>get("businessType"), excelFileDTO.getBusinessType()));
                }
                //ExcelFileType类型
                if (excelFileDTO.getExcelFileType() != null) {
                    predicates.add(builder.equal(root.<Integer>get("excelFileType"), excelFileDTO.getExcelFileType()));
                }
                //status
                if (excelFileDTO.getStatus() != null) {
                    predicates.add(builder.equal(root.<Integer>get("status"), excelFileDTO.getStatus()));
                }
                //resultState
                if (excelFileDTO.getResultState() != null) {
                    predicates.add(builder.equal(root.<Integer>get("resultState"), excelFileDTO.getResultState()));
                }
                if (!predicates.isEmpty()) {
                    criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
                }
                return criteriaQuery.getRestriction();
            };
        }
    }
}
