/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ExportFileService.java   2020-10-15 10-39-43
 * Author: Evan
 */
package com.xforceplus.business.file.service;

import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.eventbus.SubscriberExceptionContext;
import com.google.common.eventbus.SubscriberExceptionHandler;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelProcess;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.FileController;
import com.xforceplus.config.ImportExportThreadPool;
import com.xforceplus.dao.ExcelFileStoreDao;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.Map;

import static com.xforceplus.business.excel.ExcelErrorCode.EXCEL_BUSINESS_TYPE;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-15 10-39-43
 * @since V1.0
 */

@Service
public class ExportFileService implements InitializingBean, DisposableBean {

    /**
     * EventBus Name:{@value}
     */
    public static final String ASYNC_EXCEL_EXPORT_PROCESS_EVENT_BUS = "AsyncExcelExportProcessEventBus";
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(ExportFileService.class);

    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;

    private final ExcelProcess excelProcess;

    private final ExcelFileStoreDao excelFileStoreDao;

    public ExportFileService(@Qualifier("excelExportProcess") ExcelProcess excelProcess, ExcelFileStoreDao excelFileStoreDao) {
        this.excelProcess = excelProcess;
        this.excelFileStoreDao = excelFileStoreDao;
    }

    @Override
    public void destroy() throws Exception {
        //关闭线程
    }


    /**
     * 初始化Guava EventBus
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        SubscriberExceptionHandler subscriberExceptionHandler = new ExportSubscriberExceptionHandlerImpl();
        //构建异步线程池
        asyncEventBus = new AsyncEventBus(ImportExportThreadPool.get(), subscriberExceptionHandler);
        //异步注册
        asyncEventBus.register(new AsyncExcelProcessListener());
    }

    /**
     * 异步按当前用户导出
     *
     * @param params       Map<String,Object> 查询参数
     * @param businessType 导出业务类型
     * @return ExcelFileDTO
     */
    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore asyncExcelExport(Map<String, Object> params, BusinessType businessType) {
        if (CollectionUtils.isEmpty(params)) {
            throw new IllegalArgumentException("导出查询条件不能为空，请重新输入！");
        }
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        return this.asyncExcelExport(params, businessType, authorizedUser.getId(), authorizedUser.getTenantId());
    }

    /**
     * 异步按当前用户导出数据
     *
     * @param params   查询参数
     * @param userId   用户ID
     * @param tenantId 租户ID
     * @return ExcelFileDTO
     */
    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore asyncExcelExport(Map<String, Object> params, BusinessType businessType, Long userId, Long tenantId) {
        ExcelFileDTO excelFileDTO = new ExcelFileDTO();
        excelFileDTO.setBusinessType(businessType.getName());
        excelFileDTO.setTenantId(tenantId);
        excelFileDTO.setUserId(userId);
        excelFileDTO.setStatus(ExcelFileDTO.Status.PENDING);
        excelFileDTO.setResultState(ExcelFileDTO.ResultState.PENDING);
        excelFileDTO.setReadTime(0);
        excelFileDTO.setSuccessSize(0);
        excelFileDTO.setTotalSize(0);
        excelFileDTO.setExcelFileType(ExcelFileDTO.ExcelFileType.EXPORT);
        excelFileDTO.setParams(JsonUtils.toJson(params));
        excelFileDTO.setCreateTime(new Date());
        //保存数据
        ExcelFileStore excelFileStore = this.create(excelFileDTO);
        //异步处理
        this.asyncExcelProcess(params, excelFileDTO, businessType);
        return excelFileStore;
    }

    /**
     * 创建上传文件数据
     * @param excelFileDTO ImportFileDTO
     * @return ImportFile ImportFile
     */
    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore create(ExcelFileDTO excelFileDTO) {
        ExcelFileStore excelFileStore = new ExcelFileStore();
        BeanUtils.copyProperties(excelFileDTO, excelFileStore, new String[]{ExcelFileDTO.ID});
        //更新处理中
        this.excelFileStoreDao.saveAndFlush(excelFileStore);
        BeanUtils.copyProperties(excelFileStore, excelFileDTO);
        return excelFileStore;
    }

    /**
     * 重跑业务
     *
     * @param excelFileDTO ExcelFileDTO
     * @return ExcelFileDTO
     */
    @Transactional(rollbackFor = Exception.class)
    public ExcelFileDTO rerun(ExcelFileDTO excelFileDTO) {
        if (StringUtils.isBlank(excelFileDTO.getParams())) {
            throw new IllegalArgumentException("查询参数不能为空!");
        }
        excelFileDTO.setRerun(true);
        Map<String, Object> params = JsonUtils.fromJson(excelFileDTO.getParams(), new TypeReference<Map<String, Object>>() {
        });
        BusinessType businessType;
        try {
            businessType = ExcelConfigBusinessType.valueOf(excelFileDTO.getBusinessType());
        } catch (Exception e) {
            throw new IllegalArgumentException(EXCEL_BUSINESS_TYPE);
        }

        log.info("params:{},businessType:{}", params, businessType);
        //执行异步
        this.asyncExcelProcess(params, excelFileDTO, businessType);
        return excelFileDTO;
    }

    /**
     * 异步处理
     * @param excelFileDTO 文件导入中间类
     * @param importBusinessType importBusinessType
     */
    private void asyncExcelProcess(Map<String, Object> params, ExcelFileDTO excelFileDTO, BusinessType importBusinessType) {
        log.info("importBusinessType:{},importFileDTO:{}", JSON.toJSONString(importBusinessType), excelFileDTO);
        //发送
        Context context = Context.builder()
                .businessType(importBusinessType)
                .fileDTO(excelFileDTO)
                .sourceFilePath(excelFileDTO.getSourceFilePath())
                .params(params)
                .authorizedUser(UserInfoHolder.get())
                .build();
        try {
            //异步处理
            this.asyncEventBus.post(context);
        } catch (Exception e) {
            log.error("asyncExcelProcess:" + e.getMessage(), e);
            throw new IllegalArgumentException("导入数据失败，当前服务器繁忙，请稍后重试");
        }
    }


    protected class ExportSubscriberExceptionHandlerImpl implements SubscriberExceptionHandler {
        /**
         * Handles exceptions thrown by subscribers.
         *
         * @param exception
         * @param context
         */
        @Override
        public void handleException(Throwable exception, SubscriberExceptionContext context) {
            log.error(exception.getMessage(), exception);
            log.error("context:{}", JSON.toJSONString(context));
        }
    }

    /**
     * 事件监控处理器
     */
    protected class AsyncExcelProcessListener {
        @Subscribe
        public void doProcess(Context context) {
            excelProcess.process(context);
        }
    }

}
