/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ImportFileService.java   2020-09-23 17-55-36
 * Author: Evan
 */
package com.xforceplus.business.file.service;

import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.eventbus.SubscriberExceptionContext;
import com.google.common.eventbus.SubscriberExceptionHandler;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelProcess;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.file.ExcelFileDTO.ExcelFileType;
import com.xforceplus.business.excel.file.ExcelFileDTO.ResultState;
import com.xforceplus.business.excel.file.ExcelFileDTO.Status;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.ExcelReaderUtils;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.dao.ExcelFileStoreDao;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.utils.FileUtils;
import com.xforceplus.config.ThreadPoolConfig;
import com.xforceplus.utils.filetransfer.FileTransferUtilsService;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.poifs.filesystem.FileMagic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

import static com.xforceplus.business.excel.ExcelErrorCode.*;
import static com.xforceplus.business.excel.ExcelFile.EXCEL_IMPORT_POOL_NAME;
import static com.xforceplus.business.excel.file.ExcelFileDTO.Status.PROCESSING;

/**
 * <p>
 * Title:  ImportFileService
 * </p>
 * <p>
 * Description: ImportFileService
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-23 17-55-36
 * @since V1.0
 */

@Service
public class ImportFileService implements InitializingBean, DisposableBean {
    /**
     * ID 字段
     */
    public static final String ID = "id";

    /**
     * EventBus Name:{@value} 同步
     */
    public static final String EXCEL_IMPORT_PROCESS_EVENT_BUS = "ExcelImportProcessEventBus";
    /**
     * EventBus Name:{@value}
     */
    public static final String ASYNC_EXCEL_IMPORT_PROCESS_EVENT_BUS = "ExcelImportProcessEventBus";
    /**
     * 文件大小
     */
    public static final String FILE_SIZE_M = "M";
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(ImportFileService.class);
    /**
     * ImportFileDao
     */
    private final ExcelFileStoreDao excelFileStoreDao;
    /**
     * 注入ExcelImportProcess
     */
    private final ExcelProcess excelImportProcess;
    /**
     * 文件上传服务
     */
    private final FileTransferUtilsService fileTransferUtilsService;
    /**
     * 文件最大小限制
     */
    @Value("${excel.import.file.size:20}")
    private Integer excelFileSize;
    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;
    /**
     * 创建线程池
     */
    private ThreadPoolExecutor threadPoolExecutor;

    public static final int MAX_NAME_LENGTH = 512;


    public ImportFileService(ExcelFileStoreDao excelFileStoreDao, @Qualifier("excelImportProcess") ExcelProcess excelImportProcess, FileTransferUtilsService fileTransferUtilsService) {
        this.excelFileStoreDao = excelFileStoreDao;
        this.excelImportProcess = excelImportProcess;
        this.fileTransferUtilsService = fileTransferUtilsService;
    }

    /**
     * Invoked by the containing {@code BeanFactory} on destruction of a bean.
     *
     * @throws Exception in case of shutdown errors. Exceptions will get logged
     *                   but not rethrown to allow other beans to release their resources as well.
     */
    @Override
    public void destroy() throws Exception {
        //关闭线程
        this.threadPoolExecutor.shutdown();
    }

    /**
     * 初始化Guava EventBus
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        threadPoolExecutor = ThreadPoolConfig.config().name(EXCEL_IMPORT_POOL_NAME).build();
        SubscriberExceptionHandler subscriberExceptionHandler = new SubscriberExceptionHandlerImpl();
        //构建异步线程池
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, subscriberExceptionHandler);
        //异步注册
        asyncEventBus.register(new AsyncExcelProcessListener());
    }

    /**
     *  判断类型是否正确
     * @return ExcelBusinessType
     */
    protected BusinessType isBusinessType(String businessType) {
        //去空格，处理
        businessType = StringUtils.trimToEmpty(businessType).toUpperCase();
        if (StringUtils.isEmpty(businessType)) {
            throw new IllegalArgumentException(EXCEL_FILE_SAVE_EXCEPTION);
        }
        BusinessType importBusinessType;
        try {
            importBusinessType = ExcelConfigBusinessType.valueOf(businessType);
        } catch (Exception e) {
            throw new IllegalArgumentException(EXCEL_BUSINESS_TYPE);
        }
        return importBusinessType;
    }

    /**
     * 上传到文件服务器
     * @param file
     * @param sourceFilePath
     * @param tenantId  租户Id
     * @param userId 用户Id
     * @return fileId Long
     */
    protected Long uploadToFileServer(MultipartFile file, String sourceFilePath, Long tenantId, Long userId) {
        try (InputStream inputStream = file.getInputStream(); FileOutputStream fileOutputStream = new FileOutputStream(sourceFilePath)) {
            log.info("sourceFilePath:{}", sourceFilePath);
            IOUtils.copy(inputStream, fileOutputStream);
            //保存到本地服务器
            return fileTransferUtilsService.upload(sourceFilePath, userId, tenantId);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            throw new IllegalArgumentException(EXCEL_FILE_SAVE_EXCEPTION);
        }
    }

    /**
     * 检查是否为Excel
     * @param file
     * @return 检查是否为Excel
     */
    private Boolean checkExcelFile(MultipartFile file) {
        if (null == file) {
            log.error("上传文件不存在");
            throw new IllegalArgumentException(EXCEL_FILE_NOT_EXISTED);
        }
        //获取上传文件名,包含后缀
        String filenName = file.getOriginalFilename();
        //文件名长度小于等于512字符
        if (StringUtils.length(filenName) > MAX_NAME_LENGTH) {
            throw new IllegalArgumentException(EXCEL_FILE_NAME_LENGTH);
        }

        if (!StringUtils.endsWith(filenName, ExcelTypeEnum.XLS.getValue()) && !StringUtils.endsWith(filenName, ExcelTypeEnum.XLSX.getValue())) {
            throw new IllegalArgumentException(EXCEL_ERROR_XLSX);
        }
        //检查检查
        boolean sizeFlag = FileUtils.checkFileSize(file.getSize(), excelFileSize, FILE_SIZE_M);
        if (!sizeFlag) {
            log.warn("用户上Excel文件在于20M");
            throw new IllegalArgumentException(EXCEL_FILE_SIZE_LIMIT);
        }
        //判断是否为空Excel
        if (!this.isExcelFile(file)) {
            log.warn("用户上传的非2007版的Excel文件");
            throw new IllegalArgumentException(EXCEL_ERROR_XLSX);
        }
        return Boolean.TRUE;
    }

    /**
     * 判断是否为ExcelFile
     *
     * @param file
     * @return Boolean
     */
    private Boolean isExcelFile(MultipartFile file) {
        Boolean result = Boolean.FALSE;
        try (InputStream is = file.getInputStream()) {
            FileMagic fileMagic = FileMagic.valueOf(is);
            if (Objects.equals(fileMagic, FileMagic.OOXML)) {
                result = Boolean.TRUE;
            }
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            throw new IllegalArgumentException(EXCEL_ERROR_XLSX);
        }
        return result;
    }

    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore createAndSaveFile(MultipartFile file, String businessType, Long userId, Long tenantId) {
        return this.createAndSaveFile(file, businessType, userId, tenantId, null);
    }

    /**
     *  上传文件及保存记录
     * @param file
     * @param businessType
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore createAndSaveFile(MultipartFile file, String businessType, Long userId, Long tenantId, Object params) {
        businessType = StringUtils.trimToEmpty(businessType).toUpperCase();
        log.info("businessType:{}", businessType);
        //判断导入类型是否正确
        BusinessType importBusinessType = this.isBusinessType(businessType);
        //
        return this.create(file, importBusinessType, userId, tenantId, params);
    }


    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore create(MultipartFile file, BusinessType businessType, Long userId, Long tenantId) {
        return this.create(file, businessType, userId, tenantId, null);
    }

    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore create(MultipartFile file, BusinessType businessType, Long userId, Long tenantId, Object params) {
        //检查文件类型是否正确
        this.checkExcelFile(file);
        String sourceFilePath = ExcelFile.createExcelFilePath();
        //上传文件到文件服务器，并返回 FileId
        Long sourceFileId = this.uploadToFileServer(file, sourceFilePath, tenantId, userId);
        log.info("fileName:{},sourceFileId:{}", file.getOriginalFilename(), sourceFileId);

        ExcelFileDTO excelFileDTO = new ExcelFileDTO();
        excelFileDTO.setBusinessType(businessType.getName());
        excelFileDTO.setTenantId(tenantId);
        excelFileDTO.setUserId(userId);
        excelFileDTO.setSourceFileId(sourceFileId);
        excelFileDTO.setSourceFilePath(sourceFilePath);
        //文件名长度为：512
        excelFileDTO.setSourceFileName(file.getOriginalFilename());
        excelFileDTO.setStatus(Status.PENDING);
        excelFileDTO.setResultState(ResultState.PENDING);
        //读取时间
        excelFileDTO.setReadTime(0);
        excelFileDTO.setExcelFileType(ExcelFileType.IMPORT);
        excelFileDTO.setCreateTime(new Date());

        //设置参数json
        if (null != params) {
            excelFileDTO.setParams(JsonUtils.toJson(params));
        }

        //每批处理记录数
        excelFileDTO.setBatchSize(businessType.batchSize());
        //保存数据，并转为异步处理
        return this.create(excelFileDTO, businessType);
    }

    /**
     * 创建上传文件数据
     * @param excelFileDTO ImportFileDTO
     * @param importBusinessType ImportBusinessType
     * @return ImportFile ImportFile
     */
    @Transactional(rollbackFor = Exception.class)
    public ExcelFileStore create(ExcelFileDTO excelFileDTO, BusinessType importBusinessType) {
        ExcelFileStore excelFileStore = new ExcelFileStore();
        BeanUtils.copyProperties(excelFileDTO, excelFileStore, new String[]{ID});
        //更新处理中
        excelFileStore.setStatus(PROCESSING);
        this.excelFileStoreDao.saveAndFlush(excelFileStore);
        BeanUtils.copyProperties(excelFileStore, excelFileDTO);
        //异步处理
        this.asyncExcelProcess(excelFileDTO, importBusinessType);
        //上传文件
        return excelFileStore;
    }

    /**
     * 异步处理
     * @param excelFileDTO 文件导入中间类
     * @param importBusinessType importBusinessType
     */
    private void asyncExcelProcess(ExcelFileDTO excelFileDTO, BusinessType businessType) {
        if (log.isDebugEnabled()) {
            log.debug("importBusinessType:{},importFileDTO:{}", JSON.toJSONString(businessType), JSON.toJSONString(excelFileDTO));
        }
        Map<String, Object> params = Collections.emptyMap();
        if (StringUtils.isNotBlank(excelFileDTO.getParams())) {
            params = JsonUtils.fromJson(excelFileDTO.getParams(), new TypeReference<Map<String, Object>>() {
            });
        }

        Map<String,Integer> headerMap = ExcelReaderUtils.getHeaderNum(businessType);

        //发送
        Context context = Context.builder()
                .businessType(businessType)
                .fileDTO(excelFileDTO)
                .sourceFilePath(excelFileDTO.getSourceFilePath())
                .authorizedUser(UserInfoHolder.get())
                .params(params)
                .sheetHeaderMap(headerMap)
                .build();

        try {
            //异步处理
            this.asyncEventBus.post(context);
        } catch (Exception e) {
            log.error("asyncExcelProcess:" + e.getMessage(), e);
            throw new IllegalArgumentException("导入数据失败，当前服务器繁忙，请稍后重试");
        }
    }

    /**
     * 重新执行上传任务
     *
     * @param excelFileDTO
     */
    public void rerun(ExcelFileDTO excelFileDTO) {
        BusinessType businessType = null;
        try {
            businessType = ExcelConfigBusinessType.valueOf(excelFileDTO.getBusinessType());
        } catch (Exception ex) {
            log.warn(ex.getMessage(), ex);
        }
        this.asyncExcelProcess(excelFileDTO, businessType);
    }

    protected class SubscriberExceptionHandlerImpl implements SubscriberExceptionHandler {

        /**
         * Handles exceptions thrown by subscribers.
         *
         * @param exception
         * @param context
         */
        @Override
        public void handleException(Throwable exception, SubscriberExceptionContext context) {
            log.error(exception.getMessage(), exception);
            log.error("context:{}", JSON.toJSONString(context));
        }
    }

    /**
     * 事件监控处理器
     */
    protected class AsyncExcelProcessListener {
        @Subscribe
        public void doProcess(Context context) {
            excelImportProcess.process(context);
        }
    }

    /**
     * 事件监控处理器
     */
    protected class ExcelProcessListener {
        @Subscribe
        public void doProcess(Context context) {
            excelImportProcess.process(context);
        }
    }
}
