package com.xforceplus.business.log.service;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.xforceplus.dao.SystemLogDao;
import com.xforceplus.domain.log.SystemLogDTO;
import com.xforceplus.entity.SystemLog;
import com.xforceplus.utils.EntityHelp;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.persistence.Id;
import javax.persistence.Table;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;

/**
 * 版权：    上海云砺信息科技有限公司
 *
 * @author duanhy
 * 创建时间: 2020/12/3 14:58
 * 功能描述:
 * 修改历史:
 */
@Service
public class LogService {
    private static Logger logger = LoggerFactory.getLogger(LogService.class);

    @Autowired
    private SystemLogDao systemLogDao;
    /**
     * 日志线程池
     */
    private ExecutorService logThreadPool = createThreadPool();

    /**
     * 日志实体生成
     *
     * @param doEntity 操作实体
     * @param <T>
     * @return
     * @throws IllegalAccessException
     */
    private static <T> SystemLog getLogEntity(T doEntity)
            throws IllegalAccessException {
        SystemLog log = new SystemLog();
        log.setCreateTime(new Date());

        Class<?> doClass = doEntity.getClass();
        Table table = doClass.getAnnotation(Table.class);
        if (null == table) {
            throw new IllegalArgumentException(doClass.getSimpleName() + "未有@Table注解");
        }

        log.setTableName(table.name());

        try {
            for (Method method : doClass.getMethods()) {
                if (null != method.getAnnotation(Id.class) && method.getName().startsWith("get")) {
                    method.setAccessible(true);
                    log.setIdentityId((Long) method.invoke(doEntity));
                    break;
                }
            }
        } catch (InvocationTargetException e) {
            logger.error("build SystemLog error", e);
        }

        return log;
    }

    /**
     * 操作日志写入
     *
     * @param logEntity
     */
    private void doInsertLog(SystemLog logEntity) {
        systemLogDao.saveAndFlush(logEntity);
    }

    private ExecutorService createThreadPool() {
        int corePoolSize = 10;
        int maximumPoolSize = 200;
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder()
                .setNameFormat("log-pool-%d").build();
        ExecutorService threadPool = new ThreadPoolExecutor(corePoolSize, maximumPoolSize,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(10),
                namedThreadFactory,
                new ThreadPoolExecutor.AbortPolicy());
        return threadPool;
    }

    /**
     * 批量修改日志
     *
     * @param doEntities
     */
    public void insertLogBatch(List<Pair> doEntities, SystemLogDTO dto) {
        logThreadPool.execute(() -> {
            for (int i = 0; i < doEntities.size(); i++) {
                this.doLog(doEntities.get(i).getLeft(), doEntities.get(i).getRight(), dto);
            }
        });
    }

    /**
     * 修改日志
     *
     * @param doEntity
     * @param oldEntity
     */
    public <T> void insertLog(T oldEntity, T doEntity, SystemLogDTO dto) {
        logThreadPool.execute(() -> {
            this.doLog(oldEntity, doEntity, dto);
        });
    }

    /**
     * 修改日志
     *
     * @param doEntity
     * @param oldEntity
     */
    public <T> void doLog(T oldEntity, T doEntity, SystemLogDTO dto) {
        if (null == doEntity || null == oldEntity) {
            return;
        }

        try {
            SystemLog saveObj = getLogEntity(doEntity);
            saveObj.setBatchId(dto.getBatchId());
            saveObj.setBusinessType(dto.getBusinessType());
            saveObj.setActionCode(dto.getActionCode());
            saveObj.setRemark(dto.getRemark());
            Map contentMap = new HashMap();
            switch (dto.getActionCode()) {
                case D:
                    BeanUtils.populate(oldEntity, contentMap);
                    break;
                case I:
                    BeanUtils.populate(doEntity, contentMap);
                    break;
                case U:
                    contentMap = EntityHelp.compareObject(oldEntity, doEntity);
                    break;
                default:
                    break;
            }
            if (!contentMap.isEmpty()) {
                String json = JsonUtils.toJson(contentMap);
                saveObj.setContent(json);
            }
            doInsertLog(saveObj);
        } catch (Exception ex) {
            logger.error("DB操作日志修改写入失败", ex);
        }
    }


}
