/*
 * Copyright (c)  2015~2021, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: PubSubAsyncService.java   2021-01-13 15-33-21
 * Author: Evan
 */
package com.xforceplus.business.messagebus.bus;

import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.eventbus.SubscriberExceptionHandler;
import com.xforceplus.business.pub.dto.PubContext;
import com.xforceplus.business.pub.service.SubscriberExceptionHandlerImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @author duanhy
 */
@Slf4j
@Service
public class MessageBusAsyncService implements InitializingBean, DisposableBean {

    /**
     * EventBus Name:{@value}
     */
    public static final String ASYNC_MESSAGE_EVENT_BUS = "ASYNC_MESSAGE_EVENT_BUS";
    @SuppressWarnings("all")
    @Autowired(required = false)
    private MessageBusService messageBusService;
    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;
    /**
     * 创建线程池
     */
    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    /**
     * Invoked by the containing {@code BeanFactory} on destruction of a bean.
     *
     * @throws Exception in case of shutdown errors. Exceptions will get logged
     *                   but not rethrown to allow other beans to release their resources as well.
     */
    @Override
    public void destroy() throws Exception {
//        if (this.threadPoolExecutor != null) {
//            this.threadPoolExecutor.shutdown();
//        }
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        //使用全局线程池配置
        SubscriberExceptionHandler subscriberExceptionHandler = new SubscriberExceptionHandlerImpl();
        //构建异步线程池
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, subscriberExceptionHandler);
        //异步注册
        asyncEventBus.register(new AsyncPubSubProcessListener());
    }

    /**
     * 发送消息
     *
     * @param pubContext 上下文
     */
    @Async("threadPoolExecutor")
    public void send(PubContext pubContext) {
        if (this.messageBusService == null) {
            log.warn("messageBusService enable:{}", false);
        }
        this.asyncEventBus.post(pubContext);
    }


    /**
     * 事件监控处理器
     */
    protected class AsyncPubSubProcessListener {
        @Subscribe
        public void doProcess(PubContext context) {
            try {
                String messagePubResponse = messageBusService.sendMessage(context.getCode(),
                        context.getContent(),
                        context.getParam());
                log.info("code:{},messagePubResponse:{}", context.getCode(), messagePubResponse);
            } catch (Exception ignored) {
            }
        }
    }

    public void sendMessage(String code, Map<String, String> tagkvs, String content) {
        PubContext pubContext = PubContext.builder()
                .code(code)
                .param(tagkvs)
                .content(content)
                .build();
        try {
            this.send(pubContext);
        } catch (Exception e) {
            log.error("xforce.message.bus.error,requestName={}", code, e);
        }
    }

}
