package com.xforceplus.business.messagebus.bus;

import com.alibaba.fastjson.JSON;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.xforceplus.business.company.dto.CompanyTaxwareDto;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.janus.message.sdk.MBClient;
import com.xforceplus.janus.message.sdk.ResponseMessage;
import com.xforceplus.janus.message.sdk.request.AckRequest;
import com.xforceplus.janus.message.sdk.response.AckResponse;
import com.xforceplus.janus.message.sdk.response.SubResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import javax.annotation.PostConstruct;
import java.util.Collections;
import java.util.concurrent.ThreadPoolExecutor;

import static com.alibaba.fastjson.JSON.toJSONString;

/**
 * <p>从税件同步纳税人性质
 * 对接文档:https://wiki.xforceplus.com/pages/viewpage.action?pageId=123339756
 * 用单线程拉数据，然后异步处理获取的数据
 * </p>
 *
 * @author chenpengpeng@xforceplus.com
 * @date 2021/7/9
 **/
@Slf4j
@Component
@ConditionalOnExpression("${xforce.taxpayer.sync.enabled:false}")
public class TaxPayerSyncService implements InitializingBean, DisposableBean {
    @Autowired
    private CompanyService companyService;

    @Value("${remote.message.bus.server.url:}")
    private String url;

    @Value("${remote.message.bus.server.token:}")
    private String token;

    @Value("${xforce.taxpayer.topic.name:taxware_device_qualifications}")
    private String requestName;

    private MBClient client;

    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    private AsyncEventBus asyncEventBus;

    @Scheduled(fixedRate = 60 * 1000)
    public void pullTaxPayerMessage() {
        log.info("polling taxPayer message...");
        SubResponse subResponse = client.sub();
        if (subResponse.getSuccess()) {
            subResponse.getResponseMessages().forEach(message -> {
                //过滤并处理消息
                if (requestName.equalsIgnoreCase(message.getPubCode())) {
                    log.info("get message id:{},content:{}", message.getId(), message.getContent());
                    asyncEventBus.post(message);
                }
            });
        } else {
            log.error("pull taxware message error:{}", subResponse.getError());
        }

    }

    /**
     * Invoked by the containing {@code BeanFactory} on destruction of a bean.
     *
     * @throws Exception in case of shutdown errors. Exceptions will get logged
     *                   but not rethrown to allow other beans to release their resources as well.
     */
    @Override
    public void destroy() throws Exception {
        threadPoolExecutor.shutdown();
    }

    /**
     * Invoked by the containing {@code BeanFactory} after it has set all bean properties
     * and satisfied {@link org.springframework.beans.factory.BeanFactoryAware}, {@code ApplicationContextAware} etc.
     * <p>This method allows the bean instance to perform validation of its overall
     * configuration and final initialization when all bean properties have been set.
     *
     * @throws Exception in the event of misconfiguration (such as failure to set an
     *                   essential property) or if initialization fails for any other reason
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, ((exception, context) -> {
            log.error(exception.getMessage(), exception);
        }));
        asyncEventBus.register(new AsyncEventProcessListener());
    }

    @PostConstruct
    public void initPubsubConf() {
        try {
            log.info("starting init MBClient...");
            client = MBClient.getInstance(url, token);
        } catch (Exception e) {
            log.error("error while initializing pubsub config:{}", e.getMessage());
        }
    }

    /**
     * 处理拉取的消息
     *
     * @param message 消息体
     * @return true:处理成功, false:处理失败
     */
    public boolean handleMessage(ResponseMessage message) {
        if (message == null) {
            log.error("cannot handle null message");
            return true;
        }
        String content = message.getContent();
        if (ObjectUtils.isEmpty(content)) {
            log.error("cannot handle null content, messageId {}", message.getId());
            return true;
        }
        CompanyTaxwareDto companyTaxwareDto = JSON.parseObject(content, CompanyTaxwareDto.class);
        return companyService.handleTaxwareMessage(companyTaxwareDto);
    }


    private class AsyncEventProcessListener {
        @Subscribe
        public void process(ResponseMessage responseMessage) {
            try {
                boolean result = handleMessage(responseMessage);
                if (result) {
                    //每条消息处理完后单独ACK，避免部分消息处理失败导致批量ACK失败，对于处理失败的消息不做ACK，5分钟后会重新拉取处理
                    AckRequest ackRequest = new AckRequest(Collections.singletonList(responseMessage.getReceiptHandle()));
                    AckResponse ackResponse = client.ack(ackRequest);
                    if (!ackResponse.getSuccess()) {
                        log.error("ack message error:[{}]", responseMessage.getId());
                    }
                } else {
                    log.info("failed to handle message:{}", toJSONString(responseMessage));
                }
            } catch (Exception e) {
                log.error("error while handling message with messageId:{}, exception: {}", responseMessage.getId(), e.getMessage());
            }
        }
    }

}
