package com.xforceplus.business.messagebus.bus;

import com.alibaba.fastjson.JSON;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.messagebus.model.ZeusCompany;
import com.xforceplus.business.messagebus.model.ZeusTenant;
import com.xforceplus.business.messagebus.model.ZeusUser;
import com.xforceplus.business.messagebus.model.ZeusUserRelCompany;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.business.tenant.service.TenantService;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.dao.AccountDao;
import com.xforceplus.dao.OrgUserRelDao;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.dao.UserDao;
import com.xforceplus.entity.*;
import com.xforceplus.janus.message.sdk.MBClient;
import com.xforceplus.janus.message.sdk.ResponseMessage;
import com.xforceplus.janus.message.sdk.request.AckRequest;
import com.xforceplus.janus.message.sdk.response.AckResponse;
import com.xforceplus.janus.message.sdk.response.SubResponse;
import com.xforceplus.tenant.security.core.domain.OrgType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.PostConstruct;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;

/**
 * @author dhy
 * 拉取zues数据
 */
@Slf4j
@Component
@ConditionalOnExpression("${xforce.zeus.sync.enabled:false}")
public class ZeusPayerSyncService implements InitializingBean, DisposableBean {
    @Autowired
    private CompanyService companyService;
    @Autowired
    private TenantService tenantService;
    @Autowired
    private OrgService orgService;
    @Autowired
    private UserDao userDao;
    @Autowired
    private AccountDao accountDao;
    @Autowired
    private TenantDao tenantDao;

    @Autowired
    private UserService userService;

    @Autowired
    private OrgUserRelDao orgUserRelDao;

//    code style检查共享sdf实例有线程安全隐患
//    static final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");

    @Value("${remote.message.bus.server.url:}")
    private String url;

    @Value("${remote.message.bus.server.token:}")
    private String token;

    @Value("${zeus.bus.topic.company:zeus.pub.tenant-company-rel}")
    private String companyTopic;
    @Value("${zeus.bus.topic.user:zeus.pub.user}")
    private String userTopic;
    @Value("${zeus.bus.topic.tenant:zeus.pub.tenant}")
    private String tenantTopic;
    @Value("${zeus.bus.topic.user-rel-company:zeus.pub.user-rel-company}")
    private String userRelCompanyTopic;

    private MBClient client;

    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    private AsyncEventBus asyncEventBus;

    @Scheduled(fixedRate = 1000)
    public void pullZeusMessage() {
        log.info("polling zeusPayer message...");
        SubResponse subResponse = client.sub();
        if (subResponse.getSuccess()) {
            subResponse.getResponseMessages().forEach(message -> {
                //过滤并处理消息/
                if (companyTopic.equalsIgnoreCase(message.getPubCode())
                        || userTopic.equalsIgnoreCase(message.getPubCode())
                        || tenantTopic.equalsIgnoreCase(message.getPubCode())
                        || userRelCompanyTopic.equalsIgnoreCase(message.getPubCode())) {
                    log.info("get message id:{},content:{}", message.getId(), message.getContent());
                    asyncEventBus.post(message);
                }
            });
        } else {
            log.error("pull zeus message error:{}", subResponse.getError());
        }

    }

    /**
     * Invoked by the containing {@code BeanFactory} on destruction of a bean.
     *
     * @throws Exception in case of shutdown errors. Exceptions will get logged
     *                   but not rethrown to allow other beans to release their resources as well.
     */
    @Override
    public void destroy() throws Exception {
        threadPoolExecutor.shutdown();
    }

    /**
     * Invoked by the containing {@code BeanFactory} after it has set all bean properties
     * and satisfied {@link org.springframework.beans.factory.BeanFactoryAware}, {@code ApplicationContextAware} etc.
     * <p>This method allows the bean instance to perform validation of its overall
     * configuration and final initialization when all bean properties have been set.
     *
     * @throws Exception in the event of misconfiguration (such as failure to set an
     *                   essential property) or if initialization fails for any other reason
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, ((exception, context) -> {
            log.error(exception.getMessage(), exception);
        }));
        asyncEventBus.register(new AsyncEventProcessListener());
    }

    @PostConstruct
    public void initPubsubConf() {
        try {
            log.info("starting init MBClient...");
            client = MBClient.getInstance(url, token);
        } catch (Exception e) {
            log.error("error while initializing pubsub config:{}", e.getMessage());
        }
    }

    /**
     * 处理拉取的消息
     *
     * @param message 消息体
     * @return true:处理成功, false:处理失败
     */
    public boolean handleMessage(ResponseMessage message) {
        if (message == null) {
            log.error("cannot handle null message");
            return true;
        }
        String content = message.getContent();
        if (ObjectUtils.isEmpty(content)) {
            log.error("cannot handle null content, messageId {}", message.getId());
            return true;
        }
        if (companyTopic.equalsIgnoreCase(message.getPubCode())) {
            try {
                this.doCompany(content);
            } catch (Exception e) {
                log.error("消费zeus公司异常", e);
            }
        } else if (userTopic.equalsIgnoreCase(message.getPubCode())) {
            try {
                this.doUser(content);
            } catch (Exception e) {
                log.error("消费zeus用户异常", e);
            }
        } else if (tenantTopic.equalsIgnoreCase(message.getPubCode())) {
            try {
                this.doTenant(content);
            } catch (Exception e) {
                log.error("消费zeus租户异常", e);
            }
        } else if (userRelCompanyTopic.equalsIgnoreCase(message.getPubCode())) {
            try {
                this.doUserRelCompany(content);
            } catch (Exception e) {
                log.error("消费zeus人员-rel-公司异常", e);
            }
        }
        return true;
    }


    private class AsyncEventProcessListener {
        @Subscribe
        public void process(ResponseMessage responseMessage) {
            try {
                boolean result = handleMessage(responseMessage);
                if (result) {
                    //每条消息处理完后单独ACK，避免部分消息处理失败导致批量ACK失败，对于处理失败的消息不做ACK，5分钟后会重新拉取处理
                    AckRequest ackRequest = new AckRequest(Collections.singletonList(responseMessage.getReceiptHandle()));
                    AckResponse ackResponse = client.ack(ackRequest);
                    if (!ackResponse.getSuccess()) {
                        log.error("ack message error:[{}]", responseMessage.getId());
                    }
                } else {
                    log.error("failed to handle message:{}", toJSONString(responseMessage));
                }
            } catch (Exception e) {
                log.error("error while handling message with messageId:{}, exception: {}", responseMessage.getId(), e.getMessage());
            }
        }
    }


    private void doCompany(String content) {
        //code style，全局simpleDateFormat存在线程安全问题，改造为其它方式改动比较大，暂时使用局部变量，如果有性能问题再使用其它方式来处理
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
        ZeusCompany.Model.CompanyRelModel companyRelModel = JSON.parseObject(content, ZeusCompany.Model.CompanyRelModel.class);
        ZeusCompany.Model.CompanyModel companyModel = companyRelModel.getCompany();
        ZeusCompany.Model.RelModel relModel = companyRelModel.getRel();

        if (null == companyModel) {
            throw new IllegalArgumentException("zeus.company.companyModel is null");
        }
        if (null == relModel) {
            throw new IllegalArgumentException("zeus.company.relModel is null");
        }
        if (null == companyModel.getExternalId()) {
            throw new IllegalArgumentException("zeus.company.externalId is null");
        }

        Long tenantId = relModel.getTenantId();

        if (null == tenantId) {
            throw new IllegalArgumentException("zeus.company.tenantId is null");
        }
        Company saveObj = companyService.getByTaxNum(companyModel.getTaxNum());

        CompanyModel.Request.Query query = new CompanyModel.Request.Query();
        query.setCompanyId(companyModel.getExternalId());
        Company extCompany = companyService.findOne(query).orElse(null);

        if (null != saveObj && null != extCompany && !saveObj.getCompanyId().equals(extCompany.getCompanyId())) {
            throw new IllegalArgumentException("已存在的税号id:" + saveObj.getCompanyId() + "与zeus的税号id:" + extCompany.getCompanyId() + "不一致");
        }


        List<OrgStruct> orgStructs = new ArrayList<>();
        Company save = new Company();
        save.setCompanyId(companyModel.getExternalId());
        save.setTaxNum(companyModel.getTaxNum());
        save.setStatus(1);
        save.setRegistLocationAddr(companyModel.getChineseAddress());
        save.setTaxpayerQualificationType(companyModel.getTaxpayerType());
        save.setCompanyName(companyModel.getCompanyName());
        save.setCompanyCode(companyModel.getCompanyCode());
        save.setHostTenantId(tenantId);
        save.setUpdaterId(companyModel.getUpdateUserId());
        save.setUpdaterName(companyModel.getUpdateUserId());
        try {
            save.setUpdateTime(sdf.parse(companyModel.getUpdateTime()));
        } catch (ParseException e) {
        }
        if (null == saveObj && null == extCompany) {
            //新增
            save.setCreaterId(companyModel.getCreateUserId());
            save.setHostTenantId(relModel.getTenantId());
            save.setCreaterId(companyModel.getCreateUserId());
            save.setCreaterName(companyModel.getCreateUserId());
            try {
                save.setCreateTime(sdf.parse(companyModel.getCreateTime()));
            } catch (ParseException ig) {
            }
            companyService.saveAndFlush(save);
        } else if (null != extCompany) {
            if (sdf.format(extCompany.getUpdateTime()).equals(companyModel.getUpdateTime())) {
                log.info("company.updateTime一致，不做操作id:{}", extCompany.getCompanyId());
                return;
            }
            save.setCompanyId(extCompany.getCompanyId());
            companyService.saveAndFlush(save);
            orgStructs = orgService.findByTenantIdAndCompanyId(tenantId, save.getCompanyId());
        } else {
            if (sdf.format(saveObj.getUpdateTime()).equals(companyModel.getUpdateTime())) {
                log.info("company.updateTime一致，不做操作id:{}", saveObj.getCompanyId());
                return;
            }
            save.setCompanyId(saveObj.getCompanyId());
            companyService.saveAndFlush(save);
            orgStructs = orgService.findByTenantIdAndCompanyId(tenantId, save.getCompanyId());
        }

        if (CollectionUtils.isEmpty(orgStructs)) {
            OrgStruct org = new OrgStruct();
            org.setTenantId(tenantId);
            org.setCompanyId(save.getCompanyId());
            org.setOrgName(save.getCompanyName());
            org.setOrgCode(save.getCompanyCode());
            org.setOrgType(OrgType.COMPANY);
            org.setStatus(1);
            org.setOrgDesc("zeus同步");
            orgService.saveOrgStructEntity(org);
        } else {
            for (OrgStruct org : orgStructs) {
                org.setOrgName(save.getCompanyName());
                org.setOrgCode(save.getCompanyCode());
                org.setStatus(1);
                org.setOrgDesc("zeus同步");
                orgService.saveOrgStructEntity(org);
            }
        }


    }

    private void doUser(String content) {
        //code style，全局simpleDateFormat存在线程安全问题，改造为其它方式改动比较大，暂时使用局部变量，如果有性能问题再使用其它方式来处理
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
        ZeusUser.Model.UserAccountModel userAccountModel = JSON.parseObject(content, ZeusUser.Model.UserAccountModel.class);
        Long tenantId = userAccountModel.getTenantId();
        ZeusUser.Model.AccountModel accountModel = userAccountModel.getAccount();

        ZeusUser.Model.UserModel userModel = userAccountModel.getUser();
        if (null == tenantId) {
            throw new IllegalArgumentException("zeus.account.tenantId is null");
        }
        if (null == userModel) {
            throw new IllegalArgumentException("zeus.account.userModel is null");
        }
        if (null == accountModel) {
            throw new IllegalArgumentException("zeus.account.accountModel is null");
        }
        if (null == accountModel.getExternalId()) {
            throw new IllegalArgumentException("zeus.account.externalId is null");
        }
        accountModel.setPassword(accountModel.getPassword().toUpperCase());


        Map<Long, Account> accountMap = new HashMap<>();
        if (StringUtils.isNotBlank(accountModel.getEmail())) {
            Account account = accountDao.findByEmail(accountModel.getEmail());
            if (null != account) {
                accountMap.put(account.getAccountId(), account);
            }
        }
        if (StringUtils.isNotBlank(accountModel.getMobile())) {
            Account account = accountDao.findByTelPhone(accountModel.getMobile());
            if (null != account) {
                accountMap.put(account.getAccountId(), account);
            }
        }
        if (StringUtils.isNotBlank(accountModel.getAccountName())) {
            Account account = accountDao.findByUsername(accountModel.getAccountName());
            if (null != account) {
                accountMap.put(account.getAccountId(), account);
            }
        }

        //报错了 zeus一个账号对应用户中心多个
        if (accountMap.size() > 1) {
            log.error("zeus一个账号对应用户中心多个账号，email：{},mobile：{}，account：{}", accountModel.getEmail(), accountModel.getMobile(), accountModel.getAccountName());
            throw new IllegalArgumentException("zeus一个账号对应用户中心多个账号");
        }

        if (accountMap.size() == 0) {
            Account account = new Account();
            account.setAccountId(accountModel.getExternalId());
            account.setRawPassword(accountModel.getPassword());
            account.setPassword(accountModel.getPassword());
            account.setUsername(accountModel.getAccountName());
            account.setTelPhone(accountModel.getMobile());
            account.setEmail(accountModel.getEmail());
            account.setChangePasswordFlag(false);

            account.setCreaterId(accountModel.getCreateUserId());
            account.setCreaterName(accountModel.getCreateUserId());
            try {
                account.setCreateTime(sdf.parse(accountModel.getCreateTime()));
            } catch (ParseException e) {
            }
            account.setUpdaterId(accountModel.getUpdateUserId());
            account.setUpdaterName(accountModel.getUpdateUserId());
            try {
                account.setUpdateTime(sdf.parse(accountModel.getUpdateTime()));
            } catch (ParseException e) {
            }
            account = accountDao.saveAndFlush(account);

            User user = new User();
            ZeusUser.Model.Build.buildUserEntity(user, userModel);
            user.setTenantId(tenantId);
            user.setAccountId(account.getAccountId());

            user.setCreaterId(userModel.getCreateUserId());
            user.setCreaterName(userModel.getCreateUserId());
            try {
                user.setCreateTime(sdf.parse(userModel.getCreateTime()));
            } catch (ParseException e) {
            }
            user.setUpdaterId(userModel.getUpdateUserId());
            user.setUpdaterName(userModel.getUpdateUserId());
            try {
                user.setUpdateTime(sdf.parse(userModel.getUpdateTime()));
            } catch (ParseException e) {
            }
            userDao.saveAndFlush(user);
        }

        if (accountMap.size() == 1) {
            Account account = accountMap.entrySet().stream().findFirst().get().getValue();
            if (!account.getAccountId().equals(accountModel.getExternalId())) {
                String message = "zeus对应的accountId：{" + accountModel.getExternalId() + "}与用户中心accountId：{" + account.getAccountId() + "}不一致";
                throw new IllegalArgumentException(message);
            }


            if (sdf.format(account.getUpdateTime()).equals(accountModel.getUpdateTime())) {
                log.info("accout.updateTime一致，不做操作id:{}", account.getAccountId());
                return;
            }
            account.setRawPassword(accountModel.getPassword());
            account.setPassword(accountModel.getPassword());
            try {
                account.setUpdateTime(sdf.parse(accountModel.getUpdateTime()));
            } catch (ParseException e) {
            }
            account.setUpdaterId(accountModel.getUpdateUserId());
            account.setUpdaterName(accountModel.getUpdateUserId());
            accountDao.saveAndFlush(account);


            Optional<User> optionalUser = userDao.findById(userModel.getExternalId());
            if (!optionalUser.isPresent()) {
                throw new IllegalArgumentException("不存在的zeus.userId:" + userModel.getExternalId());
            }

            User user = optionalUser.get();
            if (sdf.format(user.getUpdateTime()).equals(userModel.getUpdateTime())) {
                log.info("user.updateTime一致，不做操作id:{}", user.getId());
                return;
            }

            ZeusUser.Model.Build.buildUserEntity(user, userModel);
            user.setTenantId(tenantId);
            user.setAccountId(account.getAccountId());

            try {
                user.setUpdateTime(sdf.parse(userModel.getUpdateTime()));
            } catch (ParseException e) {
            }
            user.setUpdaterId(userModel.getUserId());
            user.setUpdaterName(userModel.getUserName());
            userDao.saveAndFlush(user);

        }
    }

    private void doTenant(String content) {
        //code style，全局simpleDateFormat存在线程安全问题，改造为其它方式改动比较大，暂时使用局部变量，如果有性能问题再使用其它方式来处理
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
        ZeusTenant.Model.TenantModel tenantModel = JSON.parseObject(content, ZeusTenant.Model.TenantModel.class);
        Long externalId = tenantModel.getExternalId();
        if (null == externalId) {
            return;
        }
        Tenant tenant = tenantService.findById(externalId);

        if (sdf.format(tenant.getUpdateTime()).equals(tenantModel.getUpdateTime())) {
            log.info("tenant.updateTime一致，不做操作id:{}", tenant.getTenantId());
            return;
        }

        tenant.setTenantCode(tenantModel.getTenantCode());
        tenant.setTenantName(tenantModel.getTenantName());
        tenant.setStatus(Integer.parseInt(tenantModel.getStatus()));
        tenant.setUpdaterId(tenantModel.getUpdateUserId());
        tenant.setUpdaterName(tenantModel.getUpdateUserId());
        try {
            tenant.setUpdateTime(sdf.parse(tenantModel.getUpdateTime()));
        } catch (ParseException e) {
        }
        tenantDao.saveAndFlush(tenant);


    }


    private void doUserRelCompany(String content) {
        //code style，全局simpleDateFormat存在线程安全问题，改造为其它方式改动比较大，暂时使用局部变量，如果有性能问题再使用其它方式来处理
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
        ZeusUserRelCompany.Model.ZeusUserRelCompanyModel relModel = JSON.parseObject(content, ZeusUserRelCompany.Model.ZeusUserRelCompanyModel.class);
        Long userId = relModel.getUserId();
        if (null == userId) {
            return;
        }
        Optional<User> optionalUser = userDao.findById(userId);
        if (optionalUser.isPresent()) {
            throw new IllegalArgumentException("不存在的用户id:" + userId);
        }

        if (CollectionUtils.isEmpty(relModel.getList())) {
            orgUserRelDao.deleteByUserId(userId);
            return;
        }
        User user = optionalUser.get();
        Set<Long> orgIds = new HashSet<>();
        for (Long companyId : relModel.getList()) {
            Set<Long> set = orgService.findByTenantIdAndCompanyId(user.getTenantId(), companyId).stream().map(OrgStruct::getOrgId).collect(Collectors.toSet());
            if (!CollectionUtils.isEmpty(set)) {
                orgIds.addAll(set);
            }
        }
        if (!CollectionUtils.isEmpty(orgIds)) {
            userService.bindOrgs(user.getTenantId(), userId, orgIds, null, true, false, false);
        }
    }

}
