package com.xforceplus.business.messagebus.impl;

import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.company.dto.TenantCompanyRelPubDTO;
import com.xforceplus.business.company.service.CompanyExtensionService;
import com.xforceplus.business.messagebus.CompanyPubService;
import com.xforceplus.business.messagebus.bus.MessageBusAsyncService;
import com.xforceplus.business.pub.service.PubSubAsyncService;
import com.xforceplus.business.tenant.service.OrgUserService;
import com.xforceplus.dao.CompanyDao;
import com.xforceplus.dao.OrgCompanynoDao;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.dao.TenantCompanyRelDao;
import com.xforceplus.domain.company.CompanyExtensionDto;
import com.xforceplus.entity.*;
import com.xforceplus.json.ExtensionView;
import io.geewit.web.utils.JsonUtils;
import lombok.Builder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

/**
 * <p>
 * Title: CompanyPubService
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 *
 * @author dhy
 * <b>Creation Time:</b>  2020/11/25 22:00
 * @since V1.0
 */
@Service
public class CompanyPubServiceImpl implements InitializingBean, DisposableBean, CompanyPubService {

    private final static Logger logger = LoggerFactory.getLogger(OrgUserService.class);
    /**
     * EventBus Name:{@value}
     */
    private static final String EVENT_BUS_COMPANY_PUB = "EVENT_BUS_COMPANY_PUB";
    @Autowired
    private CompanyDao companyDao;
    @Autowired
    private OrgStructDao orgStructDao;
    @Autowired
    private TenantCompanyRelDao tenantCompanyRelDao;
    @Autowired
    private OrgCompanynoDao orgCompanynoDao;
    @Autowired
    private CompanyExtensionService companyExtensionService;
    @Value("${xforce.pub.company:xforce.tenant.company}")
    private String companyPubCode;

    @Value("${xforce.pub.tenantcompanyrel:tenant_company_rel}")
    private String tenantCompanyRel;

    @Autowired(required = false)
    private MessageBusAsyncService messageBusAsyncService;

    @Autowired(required = false)
    private PubSubAsyncService pubSubAsyncService;
    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;

    @Value("${company.pub.interval:1000}")
    private Long companyPubInterval;

    /**
     * 创建线程池
     */
    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    @Override
    public void destroy() {
        threadPoolExecutor.shutdown();
    }

    @Override
    public void afterPropertiesSet() {
        //使用全局线程池配置
        //扩大线程池数量，这里有个耗时的动作。
//        threadPoolExecutor = ThreadPoolConfig
//                .config()
//                .corePoolSize(20)
//                .maximumPoolSize(50)
//                .queueSize(1500)
//                .name(EVENT_BUS_COMPANY_PUB)
//                .build();
        //构建异步线程池
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, (exception, context) -> {
            logger.error(exception.getMessage(), exception);
        });

        //注册
        asyncEventBus.register(new AsyncProcessListener());

    }

    @Override
    public void pub(Long companyId) {
        logger.info("异步company pub,companyId={}", companyId);
        //region 构造上下文
        Context context = Context.builder()
                .companyId(companyId)
                .build();

        //region 异步处理
        try {
            this.asyncEventBus.post(context);
        } catch (Exception e) {
            String message = "公司信息pub失败，当前服务器繁忙，请稍后重试";
            logger.error(message, e);
            throw new IllegalArgumentException(message);
        }
    }

    public void pubMsg(Long companyId) {

        try {
            //为避免线程池全部进入sleep状态，将休眠时间放在apollo上配置，默认给1s
            Thread.sleep(companyPubInterval);
        } catch (InterruptedException e) {
            logger.error("error", e);
        }
        if (null == pubSubAsyncService) {
            logger.info("pubsubService == null, pubsub.enable=false");
            return;
        }
        if (null == messageBusAsyncService) {
            logger.info("messageBusAsyncService == null, pubsub.enable=false");
            return;
        }

        Optional<Company> optionalCompany = companyDao.findById(companyId);
        if (!optionalCompany.isPresent()) {
            logger.info("company pub error，company is null companyId={}", companyId);
            return;
        }
        Company company = optionalCompany.get();

        List<Tenant> tenants = tenantCompanyRelDao.findTenantsByCompanyId(companyId);
        if (CollectionUtils.isEmpty(tenants)) {
            logger.info("company pub error，tenant is null companyId={}", companyId);
            return;
        }
        Tenant tenant = tenants.get(0);
        logger.info("company pub companyId={}", companyId);

        List<OrgStruct> orgList = orgStructDao.findByCompanyId(companyId);
        OrgStruct org = null;
        if (!CollectionUtils.isEmpty(orgList)) {
            org = orgList.get(0);
        }

        List<OrgCompanyRel> companyNos = null;
        if (null != org) {
            companyNos = orgCompanynoDao.findByOrgId(org.getOrgId());
            company.setCompanyNos(companyNos.stream().map(OrgCompanyRel::getCompanyNo).collect(Collectors.toList()));
        }
        //扩展信息
        List<CompanyExtension> extensions = companyExtensionService.findByComapnyId(companyId);
        if (!CollectionUtils.isEmpty(extensions)) {
            List<CompanyExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(toList());
            company.setExtensions(extensionDtos);
        }

        Map<String, String> tagkvs = new HashMap<>();
        tagkvs.put("requestName", companyPubCode);
        tagkvs.put("tenantId", tenant.getTenantId().toString());
        tagkvs.put("businessNo", companyId + "");

        String content = JsonUtils.toJson(company, ExtensionView.Extension.class);
        logger.info("company pub saveObj = {}", content);

        //pubv2
        messageBusAsyncService.sendMessage(companyPubCode, tagkvs, content);

        //pubv1
        pubSubAsyncService.sendMessage(companyPubCode, tagkvs, content);
    }

    @Override
    public void sendTenantCompanyRelMsg(String operation, Company company, Tenant tenant) {
        TenantCompanyRelPubDTO tenantCompanyRelPubDTO = new TenantCompanyRelPubDTO();
        tenantCompanyRelPubDTO.setOperation(operation);
        tenantCompanyRelPubDTO.setTenantId(tenant.getTenantId());
        tenantCompanyRelPubDTO.setTenantCode(tenant.getTenantCode());
        tenantCompanyRelPubDTO.setTenantName(tenant.getTenantName());
        tenantCompanyRelPubDTO.setCompanyId(company.getCompanyId());
        tenantCompanyRelPubDTO.setCompanyCode(company.getCompanyCode());
        tenantCompanyRelPubDTO.setCompanyName(company.getCompanyName());
        tenantCompanyRelPubDTO.setTaxNum(company.getTaxNum());

        Map<String, String> tagkvs = new HashMap<>();
        tagkvs.put("requestName", tenantCompanyRel);
        tagkvs.put("businessNo", tenantCompanyRelPubDTO.getCompanyId() + Separator.UNDERLINE + tenantCompanyRelPubDTO.getTenantId());
        tagkvs.put("tenantId", String.valueOf(tenant.getTenantId()));


        Map<String, Object> map = new HashMap<>();
        map.put("tenantId", tenant.getTenantId());
        map.put("tenantCompanyRel", tenantCompanyRelPubDTO);
        String content = JsonUtils.toJson(map);

        messageBusAsyncService.sendMessage(tenantCompanyRel, tagkvs, content);
    }


    @Builder
    static class Context {
        private Long companyId;
    }

    /**
     * 事件监控处理器
     */
    protected class AsyncProcessListener {
        /**
         * 执行异步操作
         *
         * @param context 上下文信息
         */
        @Subscribe
        public void doProcess(Context context) {
            logger.info("AsyncProcessListener.doProcess.company.pub,context.companyId = {}", context.companyId);
            CompanyPubServiceImpl.this.pubMsg(context.companyId);

        }
    }
}
