/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: NoticeMessage.java   2020-09-27 09-58-29
 * Author: Evan
 */
package com.xforceplus.business.notice.dto;

import com.xforceplus.business.notice.entity.ExcelProcessNotice;
import lombok.Getter;
import lombok.ToString;

import java.text.MessageFormat;

/**
 * <p>
 * Title: 消息通知构建消息通知内容
 * </p>
 * <p>
 * Description: 消息通知构建消息通知内容
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-09-27 09-58-29
 * @since V1.0
 */
@Getter
@ToString
public class NoticeMessage {
    /**
     * 消息KEY
     */
    private final String key;

    /**
     * 用户ID
     */
    private final Long userId;
    /**
     * 租户ID
     */
    private final Long tenantId;
    /**
     * 文件ID
     */
    private final Long fileId;

    /**
     * 消息内容参数
     */
    private final Object[] params;

    /**
     * 消息内容
     */
    private final ExcelProcessNotice excelProcessNotice;


    /**
     * 构建消息
     *
     * @param key                Key
     * @param userId             userid
     * @param tenantId           tenantId
     * @param fileId             fileId
     * @param params             params
     * @param excelProcessNotice ExcelProcessNotice
     */
    public NoticeMessage(String key, Long userId, Long tenantId, Long fileId, Object[] params, ExcelProcessNotice excelProcessNotice) {
        this.key = key;
        this.userId = userId;
        this.tenantId = tenantId;
        this.fileId = fileId;
        this.params = params;
        this.excelProcessNotice = excelProcessNotice;
    }

    /**
     * Builder构建
     * @return Builder Builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder 对象
     */
    public static final class Builder {
        /**
         * Key
         */
        private String key;
        /**
         * userId
         */
        private Long userId;
        /**
         * 租户ID
         */
        private Long tenantId;
        /**
         * fileId
         */
        private Long fileId;
        /**
         * params
         */
        private Object[] params;


        private Builder() {
        }


        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder userId(Long userId) {
            this.userId = userId;
            return this;
        }

        public Builder tenantId(Long tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder fileId(Long fileId) {
            this.fileId = fileId;
            return this;
        }

        public Builder params(Object[] params) {
            this.params = params;
            return this;
        }

        /**
         * 消息模板填充值
         *
         * @param message 消息
         * @param params  模板
         * @return String
         */
        private String format(String message, Object[] params) {
            return MessageFormat.format(message, params);
        }

        public NoticeMessage build() {
            //获取模板消息
            NoticeTemplate noticeTemplate = ExcelNoticeConfig.getTemplate(key);
            //填充内容模板
            String content = this.format(noticeTemplate.getContent(), this.params);
            //填充标题模板
            String title = this.format(noticeTemplate.getTitle(), this.params);

            //构建消息送对象
            ExcelProcessNotice excelProcessNotice = ExcelProcessNotice.builder()
                    .content(content)
                    .title(title)
                    .tenantId(tenantId)
                    .senderId(userId)
                    .build();
            return new NoticeMessage(key, userId, tenantId, fileId, params, excelProcessNotice);
        }

    }
}
