package com.xforceplus.business.org.virtual.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.current.org.virtual.OrgVirtualApi;
import com.xforceplus.api.model.OrgVirtualModel;
import com.xforceplus.api.model.TreeModel;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.org.virtual.service.OrgVirtualNodeService;
import com.xforceplus.business.org.virtual.service.OrgVirtualService;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.OrgVirtual;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.xforceplus.api.current.org.virtual.OrgVirtualApi.Path.BATCH_EXPORT;
import static com.xforceplus.api.current.org.virtual.OrgVirtualApi.Path.IMPORT;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.ORG_VIRTUAL_CONFIG_EXCEL_FILE;

/**
 * 版权：    上海云砺信息科技有限公司
 * 创建时间: 2020/12/22 16:17
 * 功能描述:
 * 修改历史:
 * @author duanhy
 */
@Api(value = "虚拟组织")
@Controller
@Slf4j
public class OrgVirtualController extends AbstractController implements OrgVirtualApi, com.xforceplus.api.tenant.org.virtual.OrgVirtualApi {


    private final OrgVirtualService orgVirtualService;
    private final OrgVirtualNodeService orgVirtualNodeService;
    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;

    private final ImportFileService importFileService;

    @Value("${params.max-size.orgIds:100}")
    private int maxSizeOrgIds;

    public OrgVirtualController(OrgVirtualService orgVirtualService, OrgVirtualNodeService orgVirtualNodeService, ExportFileService exportFileService, ImportFileService importFileService) {
        this.orgVirtualService = orgVirtualService;
        this.orgVirtualNodeService = orgVirtualNodeService;
        this.exportFileService = exportFileService;
        this.importFileService = importFileService;
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:read"})
    public ResponseEntity<Page<OrgVirtual>> currentPage(OrgVirtualModel.Request.Query query, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTenantId(UserInfoHolder.currentUser().getTenantId());
        Page<OrgVirtual> page = orgVirtualService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:read"})
    public ResponseEntity<OrgVirtual> currentCreate(OrgVirtualModel.Request.Create model) {
        if (null == model.getStatus()) {
            model.setStatus(1);
        }
        model.setTenantId(UserInfoHolder.currentUser().getTenantId());
        OrgVirtual result = orgVirtualService.create(model);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:save"})
    public ResponseEntity<OrgVirtual> currentUpdate(Long orgVirtualId, OrgVirtualModel.Request.Update model) {
        model.setTenantId(UserInfoHolder.currentUser().getTenantId());
        model.setOrgVirtualId(orgVirtualId);
        OrgVirtual result = orgVirtualService.update(model);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:read"})
    public ResponseEntity<OrgVirtual> currentInfo(Long orgVirtualId) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        OrgVirtual result = orgVirtualService.currentInfo(tenantId, orgVirtualId);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:save"})
    public ResponseEntity<String> currentEnable(Long orgVirtualId) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        orgVirtualService.currentEnable(tenantId, orgVirtualId);
        return ResponseEntity.ok();
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:save"})
    public ResponseEntity<String> currentDisable(Long orgVirtualId) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        orgVirtualService.currentDisable(tenantId, orgVirtualId);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<List<OrgVirtualModel.Response.Model>> listByUserId(String tenantKey,
                                                                             Long typeId,
                                                                             Long userId,
                                                                             @ApiParam(value = "byTenantCode") Boolean byTenantCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        List<OrgVirtualModel.Response.Model> list = orgVirtualNodeService.listByUserId(typeId, userId, tenantId);
        return ResponseEntity.ok(list);
    }

    @Override
    public ResponseEntity<List<OrgVirtualModel.Response.Model>> listByNodeId(String tenantKey,
                                                                             Long nodeId,
                                                                             @ApiParam(value = "byTenantCode") Boolean byTenantCode,
                                                                             TreeModel.Scope scope) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        if(null == scope){
            scope = TreeModel.Scope.CHILDREN;
        }
        List<OrgVirtualModel.Response.Model> list = orgVirtualNodeService.listByNodeId(nodeId, tenantId,scope);
        return ResponseEntity.ok(list);
    }


    @Override
    public ResponseEntity<OrgVirtualModel.Response.Model> treeByNodeId(String tenantKey,
                                                                       Long nodeId,
                                                                       @ApiParam(value = "byTenantCode") Boolean byTenantCode,
                                                                       TreeModel.Scope scope) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        if(null == scope){
            scope = TreeModel.Scope.CHILDREN;
        }
        OrgVirtualModel.Response.Model model = orgVirtualNodeService.treeByNodeId(nodeId, tenantId,scope);
        return ResponseEntity.ok(model);
    }

    @Override
    public ResponseEntity<List<OrgVirtualModel.Response.Model>> listByKeys(String tenantKey,
                                                                           @ApiParam(value = "byTenantCode") Boolean byTenantCode,
                                                                           TreeModel.Scope scope,
                                                                           Integer nodeTypeId,
                                                                           boolean byId,
                                                                           String orgVirtualKeys) {

        String [] keys  = orgVirtualKeys.split(Separator.COMMA);
        if(keys.length > maxSizeOrgIds){
            throw new IllegalArgumentException("orgIds条数限制为"+maxSizeOrgIds);
        }

        Set<String> keySet = Arrays.stream(keys).collect(Collectors.toSet());
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        List<OrgVirtualModel.Response.Model> list = orgVirtualNodeService.listByKeys(tenantId,scope,nodeTypeId,byId,keySet);
        return ResponseEntity.ok(list);
    }

    @Override
    public ResponseEntity<OrgVirtual> tenantCreate(String tenantKey,
                                                   @ApiParam(value = "byTenantCode") Boolean byTenantCode,
                                                   OrgVirtualModel.Request.Create model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        if (null == model.getStatus()) {
            model.setStatus(1);
        }
        model.setTenantId(tenantId);
        OrgVirtual result = orgVirtualService.create(model);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<Page<OrgVirtual>> tenantPage(String tenantKey, Boolean byTenantCode, OrgVirtualModel.Request.Query query, Pageable pageable) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTenantId(tenantId);

        Page<OrgVirtual> page = orgVirtualService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @Override
    public ResponseEntity<OrgVirtual> tenantUpdate(String tenantKey, Boolean byTenantCode, Long orgVirtualId, OrgVirtualModel.Request.Update model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        model.setTenantId(tenantId);
        model.setOrgVirtualId(orgVirtualId);
        OrgVirtual result = orgVirtualService.update(model);
        return ResponseEntity.ok(result);
     }


    @ResponseBody
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:save"})
    @ApiOperation(value = "业务组织异步导出", notes = "业务组织异步导出")
    @RequestMapping(name = "业务组织异步导出", value = BATCH_EXPORT, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> exportCompanyConfig(@ApiParam(value = "request") @RequestBody OrgVirtualModel.Request.ExcelExportQuery query) {
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .build();
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), ORG_VIRTUAL_CONFIG_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(ORG_VIRTUAL_CONFIG_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:org:save"})
    @ApiOperation(value = "业务组织导入excel")
    @ResponseBody
    @RequestMapping(name = "业务组织导入excel", value = IMPORT, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> batchImportBind(@RequestParam("file") MultipartFile file) {
        IAuthorizedUser currentUser = UserInfoHolder.currentUser();
        Long tenantId = currentUser.getTenantId();
        Long userId = currentUser.getId();

        String businessType = ExcelConfigBusinessType.ORG_VIRTUAL_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = this.importFileService.createAndSaveFile(file, businessType, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }
}
