package com.xforceplus.business.org.virtual.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.current.org.virtual.OrgVirtualNodeApi;
import com.xforceplus.api.model.OrgVirtualNodeModel;
import com.xforceplus.api.model.OrgVirtualNodeModel.Request.Query;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.org.virtual.service.OrgVirtualNodeService;
import com.xforceplus.domain.orgVirtual.OrgVirtualTreeNode;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.OrgVirtualNode;
import com.xforceplus.entity.OrgVirtualNodeUserRel;
import com.xforceplus.entity.User;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.xforceplus.api.current.org.virtual.OrgVirtualNodeApi.Path.BATCH_EXPORT;
import static com.xforceplus.api.current.org.virtual.OrgVirtualNodeApi.Path.IMPORT;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.ORG_VIRTUAL_NODE_CONFIG_EXCEL_FILE;

/**
 * 版权：    上海云砺信息科技有限公司
 *
 * @author duanhy
 * 创建时间: 2020/12/22 16:17
 * 功能描述:
 * 修改历史:
 */
@Api(value = "虚拟组织树")
@Controller
@Slf4j
public class OrgVirtualNodeController extends AbstractController implements OrgVirtualNodeApi, com.xforceplus.api.tenant.org.virtual.OrgVirtualNodeApi {

    private final OrgVirtualNodeService orgVirtualNodeService;

    private final ImportFileService importFileService;
    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;

    public OrgVirtualNodeController(OrgVirtualNodeService orgVirtualNodeService, ImportFileService importFileService, ExportFileService exportFileService) {
        this.orgVirtualNodeService = orgVirtualNodeService;
        this.importFileService = importFileService;
        this.exportFileService = exportFileService;
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:read"})
    public ResponseEntity<Page<OrgVirtualNode>> currentPage(Long typeId, OrgVirtualNodeModel.Request.Query query, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTypeId(typeId);
        query.setTenantId(UserInfoHolder.currentUser().getTenantId());
        Page<OrgVirtualNode> page = orgVirtualNodeService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:read"})
    public ResponseEntity<OrgVirtualTreeNode> currentTree(Long typeId) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        OrgVirtualTreeNode node = orgVirtualNodeService.tree(typeId, tenantId);
        return ResponseEntity.ok(node);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:save"})
    public ResponseEntity<OrgVirtualNode> currentCreate(Long typeId, Long orgVirtualId) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        OrgVirtualNode result = orgVirtualNodeService.currentCreate(typeId, tenantId, orgVirtualId);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:save"})
    public ResponseEntity<OrgVirtualNode> currentAppend(OrgVirtualNodeModel.Request.Append append) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        append.setTenantId(tenantId);
        OrgVirtualNode result = orgVirtualNodeService.currentAppend(append);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:save"})
    public ResponseEntity<String> currentDelete(Long id) {
        Long tenantId = UserInfoHolder.currentUser().getTenantId();
        orgVirtualNodeService.currentDelete(tenantId, id);
        return ResponseEntity.ok();
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:read"})
    public ResponseEntity<Page> currentBindedTrees(Long nodeId, OrgVirtualNodeModel.Request.BindedUsers query, Pageable pageable) {

        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        query.setOrgVirtualNodeId(nodeId);
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<OrgVirtualNodeUserRel> result = orgVirtualNodeService.currentBindedTrees(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:user:save"})
    public ResponseEntity<String> currentBindTrees(Long nodeId, OrgVirtualNodeModel.Request.BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        orgVirtualNodeService.bindUsers(tenantId, nodeId, bindUsers);
        return ResponseEntity.ok("绑定成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:user:save"})
    public ResponseEntity<String> currentUnbindTrees(Long nodeId, OrgVirtualNodeModel.Request.UnbindUsers unbindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        orgVirtualNodeService.unbindUsers(tenantId, nodeId, unbindUsers);
        return ResponseEntity.ok("反绑定成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:user:save"})
    public ResponseEntity<String> currentUpdateUserScope(Long nodeId, OrgVirtualNodeModel.Request.UsersScope usersScope) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        orgVirtualNodeService.updateControlScope(tenantId, nodeId, usersScope);
        return ResponseEntity.ok();
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:admininfo:save"})
    public ResponseEntity<String> bindAdminInfoToNode(Long nodeId, OrgVirtualNodeModel.Request.BindAdminInfo bindAdminInfo) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        orgVirtualNodeService.bindAdminInfo(tenantId, nodeId, bindAdminInfo);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<String> markMainOrg(Long nodeId, OrgVirtualNodeModel.Request.BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        orgVirtualNodeService.markUserMainOrg(tenantId, nodeId, bindUsers);
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:read"})
    @ResponseBody
    @ApiOperation(value = "业务组织树异步导出", notes = "业务组织树异步导出")
    @RequestMapping(name = "业务组织树异步导出", value = BATCH_EXPORT, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> exportCompanyConfig(@ApiParam(value = "request") @RequestBody OrgVirtualNodeModel.Request.ExcelExportQuery query) {
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .build();
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), ORG_VIRTUAL_NODE_CONFIG_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(ORG_VIRTUAL_NODE_CONFIG_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:org-virtual:node:save"})
    @ApiOperation(value = "业务组织树导入excel")
    @ResponseBody
    @RequestMapping(name = "业务组织树导入excel", value = IMPORT, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> batchImportBind(@RequestParam("file") MultipartFile file, @RequestParam(name = "typeId") Long typeId) {
        Long tenantId = UserInfoHolder.get().getTenantId();
        Long userId = UserInfoHolder.get().getId();

        Map<String, Object> params = new HashMap<>();
        params.put("typeId", typeId);
        String businessType = ExcelConfigBusinessType.ORG_VIRTUAL_NODE_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file, businessType, userId, tenantId, params);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @Override
    public ResponseEntity<OrgVirtualNode> tenantCreate(String tenantKey, Boolean byTenantCode, Long typeId, Long orgVirtualId) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        OrgVirtualNode result = orgVirtualNodeService.currentCreate(typeId, tenantId, orgVirtualId);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<OrgVirtualNode> tenantAppend(String tenantKey, Boolean byTenantCode, OrgVirtualNodeModel.Request.Append append) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        append.setTenantId(tenantId);
        OrgVirtualNode result = orgVirtualNodeService.currentAppend(append);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<String> batchBindUserAndAdmin(Long typeId, Long nodeId, String tenantKey, Boolean byTenantCode, @Valid OrgVirtualNodeModel.Request.BindAdminInfo bindAdminInfo) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        OrgVirtualNodeModel.Request.BindUsers bindUsers = new OrgVirtualNodeModel.Request.BindUsers();
        List<Long> userIds = bindAdminInfo.getAdmins().stream().filter(Objects::nonNull).map(OrgVirtualNodeModel.Request.AdminInfo::getUserId).distinct().collect(Collectors.toList());
        bindUsers.setUserIds(userIds);
        orgVirtualNodeService.bindUsers(tenantId, nodeId, bindUsers);
        if (CollectionUtils.isEmpty(bindAdminInfo.getAdmins())) {
            return ResponseEntity.ok();
        }
        orgVirtualNodeService.bindAdminInfo(tenantId, nodeId, bindAdminInfo);
        return ResponseEntity.ok();
    }

    @Override
    public ResponseEntity<List<User>> queryUserInfoOfNode(String tenantKey, Boolean byTenantCode, Long virtualNodeId, Boolean isNodeId, Long typeId, Integer adminFlag) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        OrgVirtualNodeModel.Request.BindedUsers query = new OrgVirtualNodeModel.Request.BindedUsers();
        query.setOrgVirtualNodeId(virtualNodeId);
        query.setTenantId(tenantId);
        List<User> result = orgVirtualNodeService.currentBindedAdminInfo(query, adminFlag, isNodeId, typeId);
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<Page<OrgVirtualNode>> tenantVirtualTreePage(Long typeId,
                                                                      Query query,
                                                                      Pageable pageable,
                                                                      String tenantKey,
                                                                      Boolean byTenantCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);

        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTypeId(typeId);
        query.setTenantId(tenantId);
        Page<OrgVirtualNode> page = orgVirtualNodeService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @Override
    public ResponseEntity<String> tenantDelete(String tenantKey, Long id, Long typeId, Long userId, Boolean byTenantCode, Integer operationType) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        orgVirtualNodeService.tenantDelete(tenantId, id, typeId, userId, operationType);
        return ResponseEntity.ok("删除成功");
    }

}
