package com.xforceplus.business.org.virtual.service;

import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.xforceplus.business.tenant.service.WrapperOrgVirtualNodeUserService;
import lombok.Builder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Service;

import java.util.concurrent.ThreadPoolExecutor;


/**
 * 保存 虚拟虚拟组织树节点-用户关联表
 *
 * @author duanhy
 */
@Service
public class AsyncOrgVirtualNodeUserService implements InitializingBean, DisposableBean {
    private static final Logger logger = LoggerFactory.getLogger(AsyncOrgVirtualNodeUserService.class);

    private static final long SLEEP_MILLS = 1000L;


    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;

    /**
     * 创建线程池
     */
    private final ThreadPoolExecutor threadPoolExecutor;

    private final WrapperOrgVirtualNodeUserService wrapperOrgVirtualNodeUserService;

    public AsyncOrgVirtualNodeUserService(ThreadPoolExecutor threadPoolExecutor, WrapperOrgVirtualNodeUserService wrapperOrgVirtualNodeUserService) {
        this.threadPoolExecutor = threadPoolExecutor;
        this.wrapperOrgVirtualNodeUserService = wrapperOrgVirtualNodeUserService;
    }

    /**
     * 初始化Guava EventBus
     */
    @Override
    public void afterPropertiesSet() {
        //region 构建异步线程池
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, (exception, context) -> {
            logger.error(exception.getMessage(), exception);
        });
        //endregion
        //region 异步注册
        asyncEventBus.register(new AsyncProcessListener());
        //endregion
    }

    /**
     * 异步自动根据上级虚拟组织树关联的用户集合关联新建虚拟组织树
     * @param tenantId 租户id
     * @param nodeId   虚拟组织树id
     */
    public void autoBindParentUsers(Long tenantId, Long nodeId) {
        logger.info("tenantId:{}, nodeId: {}", tenantId, nodeId);
        if (tenantId == null || tenantId == 0 || nodeId == null || nodeId == 0) {
            logger.info("tenantId == 0 || nodeId == 0, return");
            return;
        }
        //region 构造上下文
        Context context = Context.builder()
                .tenantId(tenantId)
                .nodeId(nodeId)
                .build();
        //endregion

        //region 异步处理
        try {
            this.asyncEventBus.post(context);
        } catch (Exception e) {
            logger.error("asyncExcelProcess:" + e.getMessage(), e);
            throw new IllegalArgumentException("导入数据失败，当前服务器繁忙，请稍后重试");
        }
        //endregion
    }

    @Override
    public void destroy() {
        //region 关闭线程
//        this.threadPoolExecutor.shutdown();
        //endregion
    }

    @Builder
    static class Context {
        long tenantId;
        long nodeId;
    }

    /**
     * 事件监控处理器
     */
    protected class AsyncProcessListener {
        /**
         * 执行异步操作
         *
         * @param context 上下文信息
         */
        @Subscribe
        public void doProcess(Context context) {
            logger.info("AsyncProcessListener.doProcess, context.tenantId = {}, context.nodeId = {}", context.tenantId, context.nodeId);
            if (context.nodeId > 0) {
                try {
                    Thread.sleep(SLEEP_MILLS);
                } catch (InterruptedException e) {
                    logger.warn(e.getMessage());
                    Thread.currentThread().interrupt();
                } finally {
                    try {
                        wrapperOrgVirtualNodeUserService.autoBindUsers(context.tenantId, context.nodeId);
                    } catch (Exception e) {
                        logger.warn(e.getMessage());
                    }
                }
            }
        }
    }
}
