package com.xforceplus.business.org.virtual.service;

import com.xforceplus.api.model.OrgVirtualModel;
import com.xforceplus.api.model.OrgVirtualNodeModel;
import com.xforceplus.api.model.TreeModel;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.excel.ExcelValidator;
import com.xforceplus.business.org.virtual.dto.OrgVirtualNodeImportDto;
import com.xforceplus.constants.OperationType;
import com.xforceplus.dao.OrgVirtualDao;
import com.xforceplus.dao.OrgVirtualNodeDao;
import com.xforceplus.dao.OrgVirtualUserRelDao;
import com.xforceplus.dao.UserDao;
import com.xforceplus.domain.orgVirtual.OrgVirtualTreeNode;
import com.xforceplus.entity.OrgVirtual;
import com.xforceplus.entity.OrgVirtualNode;
import com.xforceplus.entity.OrgVirtualNodeUserRel;
import com.xforceplus.entity.User;
import com.xforceplus.query.OrgVirtualNodeUserRelQueryHelper;
import com.xforceplus.query.OrgVirtualTreeQueryHelper;
import com.xforceplus.utils.StringHelp;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.business.org.virtual.service.OrgVirtualNodeImportServiceImpl.TREE_ACTION_CREATE;
import static com.xforceplus.business.org.virtual.service.OrgVirtualNodeImportServiceImpl.TREE_ACTION_DELETE;

/**
 * 版权：    上海云砺信息科技有限公司
 * author:  duanhy
 * 创建时间: 2020/12/22 16:16
 * 功能描述:
 * 修改历史:
 */
@Service
@Slf4j
public class OrgVirtualNodeService {

    public OrgVirtualNodeService(OrgVirtualNodeDao orgVirtualNodeDao, OrgVirtualService orgVirtualService, OrgVirtualUserRelDao orgVirtualUserRelDao, UserDao userDao, OrgVirtualNodeTypeService orgVirtualNodeTypeService, AsyncOrgVirtualNodeUserService asyncOrgVirtualNodeUserService, OrgVirtualDao orgVirtualDao) {
        this.orgVirtualNodeDao = orgVirtualNodeDao;
        this.orgVirtualService = orgVirtualService;
        this.orgVirtualUserRelDao = orgVirtualUserRelDao;
        this.userDao = userDao;
        this.orgVirtualNodeTypeService = orgVirtualNodeTypeService;
        this.asyncOrgVirtualNodeUserService = asyncOrgVirtualNodeUserService;
        this.orgVirtualDao = orgVirtualDao;
    }

    private final OrgVirtualNodeDao orgVirtualNodeDao;
    private final OrgVirtualService orgVirtualService;
    private final OrgVirtualUserRelDao orgVirtualUserRelDao;
    private final UserDao userDao;
    private final OrgVirtualNodeTypeService orgVirtualNodeTypeService;
    private final AsyncOrgVirtualNodeUserService asyncOrgVirtualNodeUserService;
    private final OrgVirtualDao orgVirtualDao;


    public Page<OrgVirtualNode> page(OrgVirtualNodeModel.Request.Query query, Pageable pageable) {
        Specification<OrgVirtualNode> specification = OrgVirtualTreeQueryHelper.querySpecification(query);
        return orgVirtualNodeDao.findAll(specification, pageable, EntityGraphs.named(OrgVirtualNode.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public List<OrgVirtualNode> list(OrgVirtualNodeModel.Request.Query query) {
        Specification<OrgVirtualNode> specification = OrgVirtualTreeQueryHelper.querySpecification(query);
        return orgVirtualNodeDao.findAll(specification);
    }

    public OrgVirtualTreeNode tree(Long typeId, Long tenantId) {
        List<OrgVirtualTreeNode> list = orgVirtualNodeDao.treeNodeList(tenantId, typeId);
        return this.buildTreeForPage(list);
    }

    private OrgVirtualTreeNode buildTreeForPage(List<OrgVirtualTreeNode> orgs) {

        Map<Long, OrgVirtualTreeNode> datas = orgs.stream().collect(Collectors.toMap(OrgVirtualTreeNode::getKey, org -> org));

        OrgVirtualTreeNode root = null;

        for (OrgVirtualTreeNode ele : orgs) {
            if (null != ele.getPid()) {
                OrgVirtualTreeNode parent = datas.get(ele.getPid());
                if (CollectionUtils.isEmpty(parent.getChildren())) {
                    Set<OrgVirtualTreeNode> children = Stream.of(ele).collect(Collectors.toSet());
                    parent.setChildren(children);
                } else {
                    parent.getChildren().add(ele);
                }
            } else {
                root = ele;
            }
        }
        return root;
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgVirtualNode currentCreate(Long typeId, Long tenantId, Long orgVirtualId) {

        orgVirtualNodeTypeService.findByNodeTypeId(typeId);

        orgVirtualService.findByTenantIdAndId(tenantId, orgVirtualId);

        Optional<OrgVirtualNode> optionalOrgVirtualTree = orgVirtualNodeDao.findByTenantIdAndNodeTypeIdAndParentIdIsNull(tenantId, typeId);
        if (optionalOrgVirtualTree.isPresent()) {
            String message = "根组织已存在(" + optionalOrgVirtualTree.get().getOrgVirtualId() + ")";
            log.info(message);
            throw new IllegalArgumentException(message);
        }

        OrgVirtualNode tree = new OrgVirtualNode();
        tree.setTenantId(tenantId);
        tree.setOrgVirtualId(orgVirtualId);
        tree.setNodeTypeId(typeId);
        tree.setParentId(null);
        tree = orgVirtualNodeDao.save(tree);
        tree.setParentIds(tree.getOrgVirtualNodeId() + "#");
        tree = orgVirtualNodeDao.saveAndFlush(tree);
        return tree;
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgVirtualNode currentAppend(OrgVirtualNodeModel.Request.Append append) {

        OrgVirtualNode parent = this.findByTenantIdAndNodeId(append.getTenantId(), append.getParentId());

        orgVirtualService.findByTenantIdAndId(append.getTenantId(), parent.getOrgVirtualId());

        OrgVirtual current = orgVirtualService.findByTenantIdAndId(append.getTenantId(), append.getOrgVirtualId());
        if (1 != current.getStatus()) {
            String message = "该组织已禁用(" + append.getOrgVirtualId() + ")";
            log.info(message);
            throw new IllegalArgumentException(message);
        }

        OrgVirtualNodeModel.Request.Query query = new OrgVirtualNodeModel.Request.Query();
        query.setOrgVirtualId(append.getOrgVirtualId());
        query.setTenantId(append.getTenantId());
        query.setTypeId(parent.getNodeTypeId());
        List<OrgVirtualNode> existList = this.list(query);
        if (!CollectionUtils.isEmpty(existList)) {
            String message = "该业务组织已关联业务组织树(" + existList.get(0).getParentId() + ")";
            log.info(message);
            throw new IllegalArgumentException(message);
        }

        OrgVirtualNode tree = new OrgVirtualNode();
        tree.setTenantId(append.getTenantId());
        tree.setOrgVirtualId(append.getOrgVirtualId());
        tree.setNodeTypeId(parent.getNodeTypeId());
        tree.setParentId(parent.getOrgVirtualNodeId());
        tree = orgVirtualNodeDao.save(tree);
        tree.setParentIds(parent.getParentIds() + tree.getOrgVirtualNodeId() + Separator.BACKSLASH);
        tree = orgVirtualNodeDao.saveAndFlush(tree);

        //虚拟组织树增加下级自动关联 有权限的人员
        asyncOrgVirtualNodeUserService.autoBindParentUsers(append.getTenantId(), tree.getOrgVirtualNodeId());
        return tree;
    }

    public OrgVirtualNode findByTenantIdAndNodeId(Long tenantId, Long id) {
        return orgVirtualNodeDao.findByTenantIdAndOrgVirtualNodeId(tenantId, id).orElseThrow(() -> new IllegalArgumentException("未找到业务组织树（" + id + ")"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void currentDelete(Long tenantId, Long id) {

        OrgVirtualNode tree = this.findByTenantIdAndNodeId(tenantId, id);
        List<OrgVirtualNode> list = orgVirtualNodeDao.findByTenantIdAndParentId(tenantId, id);
        if (!CollectionUtils.isEmpty(list)) {
            String message = "该节点(" + id + ")包含下级组织，不能删除";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        List<OrgVirtualNodeUserRel> userRels = orgVirtualUserRelDao.findByOrgVirtualNodeId(id);
        if (!CollectionUtils.isEmpty(userRels)) {
            String message = "该节点(" + id + ")包含人员，不能删除";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        orgVirtualNodeDao.delete(tree);
    }

    @Transactional(rollbackFor = Exception.class)
    public void tenantDelete(Long tenantId, Long id, Long typeId, Long userId, Integer operationType) {
        Optional<OrgVirtualNode> orgVirtualNode = Optional.empty();
        List<Long> userIds = null;
        // 虚拟组织的删除范围：sys_org_virtual  - sys_org_virtual_node - sys_org_virtual_node_user_rel
        if (operationType <= OperationType.VIRTUAL) {
            OrgVirtual virtual = orgVirtualService.findByTenantIdAndId(tenantId, id);
            if (virtual == null || 0 == virtual.getStatus()) {
                String message = "该组织(" + id + ") 不存在";
                log.info(message);
                throw new IllegalArgumentException(message);
            }
            virtual.setStatus(0);
            orgVirtualDao.saveAndFlush(virtual);
            orgVirtualNode = orgVirtualNodeDao.findByNodeTypeIdAndOrgVirtualId(typeId, id);
            if (!orgVirtualNode.isPresent()) {
                log.info("virtualId :{} 查询 orgVirtualNode 为空  跳过执行", id);
                return;
            }
            //定义虚拟节点id，为删除虚拟节点做准备
            id = orgVirtualNode.get().getOrgVirtualNodeId();
        }
        // 虚拟节点的删除范围：  sys_org_virtual_node - sys_org_virtual_node_user_rel
        if (operationType <= OperationType.NODE) {
            if (!orgVirtualNode.isPresent()) {
                orgVirtualNode = orgVirtualNodeDao.findByTenantIdAndOrgVirtualNodeId(tenantId, id);
                if (!orgVirtualNode.isPresent()) {
                    log.info("nodeid :{} 查询 orgVirtualNode 为空  跳过执行", id);
                    throw new IllegalArgumentException("该节点(" + id + ") 不存在");
                }
            }
            List<OrgVirtualNode> list = orgVirtualNodeDao.findByTenantIdAndParentId(tenantId, id);
            if (!CollectionUtils.isEmpty(list)) {
                String message = "该组织或节点包含下级组织，不能删除";
                log.info(message);
                throw new IllegalArgumentException(message);
            }
            orgVirtualNodeDao.deleteByNodeId(tenantId, id);
            List<OrgVirtualNodeUserRel> orgVirtualNodeUserRels = orgVirtualUserRelDao.findByOrgVirtualNodeId(id);
            if (CollectionUtils.isEmpty(orgVirtualNodeUserRels)) {
                log.warn("nodeid :{} 查询 orgVirtualNodeUserRels 为空,跳过执行", id);
                return;
            }
            //定义关联的用户列表，为删除虚拟关系做准备
            userIds = orgVirtualNodeUserRels.stream().map(OrgVirtualNodeUserRel::getUserId).collect(Collectors.toList());
        }
        // 虚拟关系的删除范围：  sys_org_virtual_node_user_rel
        if (operationType <= OperationType.NODE_USER_REL) {
            if (CollectionUtils.isEmpty(userIds)) {
                if (Objects.isNull(userId)) {
                    log.warn("id {} operationType {}  userId 为空  跳过执行", id, operationType);
                    return;
                }
                userIds = Collections.singletonList(userId);
            }
            orgVirtualUserRelDao.deleteByUserIdsAndNodeId(tenantId, userIds, id);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindUsers(long tenantId, Long nodeId, OrgVirtualNodeModel.Request.UnbindUsers unbindUsers) {
        List<Long> userIds = unbindUsers.getUserIds();
        if (CollectionUtils.isEmpty(userIds)) {
            String message = "参数缺少用户id集合";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        if (!CollectionUtils.isEmpty(userIds)) {
            orgVirtualUserRelDao.deleteByUserIdsAndNodeId(tenantId, userIds, nodeId);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long tenantId, Long nodeId, OrgVirtualNodeModel.Request.BindUsers bindUsers) {

        List<Long> userIds = bindUsers.getUserIds();
        if (CollectionUtils.isEmpty(userIds)) {
            String message = "参数缺少用户id集合";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        OrgVirtualNode orgVirtualNode = this.findByTenantIdAndNodeId(tenantId, nodeId);
        List<OrgVirtualNodeUserRel> existRels = orgVirtualUserRelDao.findByOrgVirtualNodeId(orgVirtualNode.getOrgVirtualNodeId());

        Set<OrgVirtualNodeUserRel> insertingRels = userIds.stream().filter(Objects::nonNull).filter(userId -> existRels.stream().map(OrgVirtualNodeUserRel::getUserId).noneMatch(relUserId -> relUserId.equals(userId))).map(userId -> {
            Optional<User> userOptional = userDao.findById(userId);
            if (userOptional.isPresent()) {
                OrgVirtualNodeUserRel treeUserRel = new OrgVirtualNodeUserRel();
                treeUserRel.setTenantId(userOptional.get().getTenantId());
                treeUserRel.setOrgVirtualNodeId(nodeId);
                treeUserRel.setUserId(userId);
                treeUserRel.setTenantId(tenantId);
                treeUserRel.setControlFlag(0);
                return treeUserRel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        if (!insertingRels.isEmpty()) {
            insertingRels.forEach(orgVirtualUserRelDao::saveAndFlush);
        }
    }

    /**
     * 绑定组织管理员信息
     *
     * @param tenantId
     * @param nodeId
     * @param bindAdminInfo
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindAdminInfo(long tenantId, Long nodeId, OrgVirtualNodeModel.Request.BindAdminInfo bindAdminInfo) {
        if (Objects.isNull(bindAdminInfo) || CollectionUtils.isEmpty(bindAdminInfo.getAdmins())) {
            String message = "参数缺少管理员设置信息";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        OrgVirtualNode orgVirtualNode = this.findByTenantIdAndNodeId(tenantId, nodeId);
        List<OrgVirtualNodeModel.Request.AdminInfo> admins = bindAdminInfo.getAdmins();
        Map<Long, List<OrgVirtualNodeModel.Request.AdminInfo>> adminMap = admins.stream().filter(Objects::nonNull)
                .filter(admin -> Objects.nonNull(admin.getUserId()))
                .collect(Collectors.groupingBy(OrgVirtualNodeModel.Request.AdminInfo::getUserId));
        List<OrgVirtualNodeUserRel> existRels = orgVirtualUserRelDao.findByOrgVirtualNodeId(orgVirtualNode.getOrgVirtualNodeId());
        existRels.forEach(rel -> {
            if (adminMap.containsKey(rel.getUserId())) {
                //todo
                rel.setAdminFlag(adminMap.get(rel.getUserId()).get(0).getSort());
            }
        });
        if (!existRels.isEmpty()) {
            orgVirtualUserRelDao.saveBatch(existRels);
        }
    }

    /**
     * 绑定组织管理员信息
     * 同租户下的同组织树下，每个用户的主组织只能被设置一次
     * 如果users 用户之前标记过主组织，该操作会覆盖之前标记
     *
     * @param tenantId
     * @param nodeId
     * @param users
     */
    @Transactional(rollbackFor = Exception.class)
    public void markUserMainOrg(long tenantId, Long nodeId, OrgVirtualNodeModel.Request.BindUsers users) {
        if (Objects.isNull(users) || CollectionUtils.isEmpty(users.getUserIds())) {
            String message = "参数缺少有效的用户信息";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        OrgVirtualNode orgVirtualNode = this.findByTenantIdAndNodeId(tenantId, nodeId);
        Long typeId = orgVirtualNode.getNodeTypeId();
        List<Long> userIds = users.getUserIds().stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userIds)) {
            String message = "参数缺少有效的用户信息";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        orgVirtualUserRelDao.markMainOrg(nodeId, tenantId, typeId, userIds);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateControlScope(long tenantId, Long nodeId, OrgVirtualNodeModel.Request.UsersScope usersScope) {
        List<Long> userIds = usersScope.getUserIds();
        if (CollectionUtils.isEmpty(userIds)) {
            String message = "参数缺少用户id集合";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
        int controlFlag = usersScope.getControlFlag();

        OrgVirtualNode node = this.findByTenantIdAndNodeId(tenantId, nodeId);

        userIds.forEach(userId -> {
            Optional<OrgVirtualNodeUserRel> optionalOrgVirtualNodeUserRel = orgVirtualUserRelDao.findByOrgVirtualNodeIdAndTenantIdAndUserId(nodeId, tenantId, userId);

            if (!optionalOrgVirtualNodeUserRel.isPresent()) {
                return;
            }
            OrgVirtualNodeUserRel rel = optionalOrgVirtualNodeUserRel.get();

            if (controlFlag != rel.getControlFlag()) {
                switch (controlFlag) {
                    case 1:
                        this.choiceSub(node, userId);
                        break;
                    case 0:
                        this.unChoiceSub(node, userId);
                        break;
                    default:
                        break;
                }
                rel.setControlFlag(controlFlag);
                orgVirtualUserRelDao.saveAndFlush(rel);
            }

        });
    }

    public List<OrgVirtualNode> findSubByParentIds(OrgVirtualNode node) {
        OrgVirtualNodeModel.Request.Query query = new OrgVirtualNodeModel.Request.Query();
        query.setParentIds(node.getParentIds());
        query.setTenantId(node.getTenantId());
        query.setTypeId(node.getNodeTypeId());
        return this.list(query);
    }

    public void choiceSub(OrgVirtualNode node, Long userId) {
        List<OrgVirtualNode> nodes = this.findSubByParentIds(node);

        Set<OrgVirtualNodeUserRel> insertingRels = nodes.stream().filter(Objects::nonNull).map(orgVirtualNode -> {
            Optional<OrgVirtualNodeUserRel> rel = orgVirtualUserRelDao.findByOrgVirtualNodeIdAndUserId(orgVirtualNode.getOrgVirtualNodeId(), userId);
            if (!rel.isPresent()) {
                OrgVirtualNodeUserRel treeUserRel = new OrgVirtualNodeUserRel();
                treeUserRel.setTenantId(node.getTenantId());
                treeUserRel.setOrgVirtualNodeId(orgVirtualNode.getOrgVirtualNodeId());
                treeUserRel.setUserId(userId);
                treeUserRel.setTenantId(node.getTenantId());
                treeUserRel.setControlFlag(1);
                return treeUserRel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        if (!insertingRels.isEmpty()) {
            insertingRels.forEach(orgVirtualUserRelDao::saveAndFlush);
        }
    }

    public void unChoiceSub(OrgVirtualNode node, Long userId) {
        List<OrgVirtualNode> nodes = this.findSubByParentIds(node);
        nodes.stream().filter(orgVirtualNode -> !orgVirtualNode.getOrgVirtualNodeId().equals(node.getOrgVirtualNodeId())).forEach(orgVirtualNode -> {
            orgVirtualUserRelDao.deleteByTenantIdAndUserIdAndOrgVirtualNodeId(orgVirtualNode.getTenantId(), userId, orgVirtualNode.getOrgVirtualNodeId());
        });

    }

    public Page<OrgVirtualNodeUserRel> currentBindedTrees(OrgVirtualNodeModel.Request.BindedUsers query, Pageable currentPageable) {
        Specification<OrgVirtualNodeUserRel> specification = OrgVirtualNodeUserRelQueryHelper.querySpecification(query);
        return orgVirtualUserRelDao.findAll(specification, currentPageable, EntityGraphs.named(OrgVirtualNodeUserRel.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    /**
     * 查询绑定管理员
     *
     * @param query
     * @return
     */
    public List<User> currentBindedAdminInfo(OrgVirtualNodeModel.Request.BindedUsers query, Integer adminFlag, Boolean isNodeId, Long typeId) {
        if (!isNodeId) {
            Long virtualId = query.getOrgVirtualNodeId();
            Optional<OrgVirtualNode> optionalOrgVirtualNode = orgVirtualNodeDao.findByNodeTypeIdAndOrgVirtualIdAndTenantId(query.getTenantId(), typeId, virtualId);
            if (!optionalOrgVirtualNode.isPresent()) {
                String message = "无效的虚拟组织id: " + virtualId;
                log.info(message);
                throw new IllegalArgumentException(message);
            }
            query.setOrgVirtualNodeId(optionalOrgVirtualNode.get().getOrgVirtualNodeId());
        }
        List<OrgVirtualNodeUserRel> re = orgVirtualUserRelDao.findByOrgVirtualNodeIdAndAdminFlag(query.getOrgVirtualNodeId(), adminFlag);
        if (CollectionUtils.isEmpty(re)) {
            return Collections.emptyList();
        }

        return re.stream().map(rel -> {
            try {
                User user = rel.getUser();
                user.setAppResources(null);
                user.setAccount(null);
                user.setTenant(null);
                return user;
            } catch (Exception e) {
                log.error("获取用户信息异常{}，userid {}", e, rel.getUserId());
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public List<OrgVirtualNodeModel.Response.ExcelExportModel> listNodeToExcel(OrgVirtualNodeModel.Request.ExcelExportQuery query) {
        List<OrgVirtualTreeNode> list = orgVirtualNodeDao.treeNodeList(query.getTenantId(), query.getTypeId());
        return this.buildFlatList(list);
    }

    public List<OrgVirtualNodeModel.Response.ExcelExportModel> listNodeRelAccountToExcel(OrgVirtualNodeModel.Request.ExcelExportQuery query) {
        List<OrgVirtualNodeModel.Response.ExcelExportModel> list = orgVirtualNodeDao.listNodeRelAccount(query.getTenantId(), query.getTypeId());
        for (OrgVirtualNodeModel.Response.ExcelExportModel model : list) {
            if (null != model.getControlFlag()) {
                model.setControlFlagValue(1 == model.getControlFlag() ? "当前组织及下级组织" : "当前组织");
            }
            if (null != model.getAdminFlag()) {
                model.setAdminFlagValue(model.getAdminFlag().compareTo(0) > 0 ? "是" : "否");
            }
        }
        return list;
    }

    private List<OrgVirtualNodeModel.Response.ExcelExportModel> buildFlatList(List<OrgVirtualTreeNode> orgs) {

        Map<Long, OrgVirtualTreeNode> datas = orgs.stream().collect(Collectors.toMap(OrgVirtualTreeNode::getKey, org -> org));

        List<OrgVirtualNodeModel.Response.ExcelExportModel> list = new ArrayList<>();

        for (OrgVirtualTreeNode ele : orgs) {
            OrgVirtualNodeModel.Response.ExcelExportModel model = new OrgVirtualNodeModel.Response.ExcelExportModel();
            model.setOrgCode(ele.getOrgCode());
            model.setOrgName(ele.getOrgName());
            if (null != ele.getPid()) {
                OrgVirtualTreeNode parent = datas.get(ele.getPid());
                if (null != parent) {
                    model.setParentOrgCode(parent.getOrgCode());
                    model.setParentOrgName(parent.getOrgName());
                }
            }
            list.add(model);
        }
        return list;
    }

    @Transactional(rollbackFor = Exception.class)
    public String importSave(OrgVirtualNodeImportDto dto, Long tenantId, Long typeId) {

        List<String> errMsg = new ArrayList<>();

        ExcelValidator.validate(dto);

        if (StringUtils.isNotBlank(dto.getValidatedMessage())) {
            errMsg.add(dto.getValidatedMessage());
        }
        OrgVirtual orgVirtual = orgVirtualService.findByTenantIdAndOrgCode(tenantId, dto.getOrgCode());
        if (null == orgVirtual) {
            errMsg.add("不存在的业务组织[" + dto.getOrgCode() + "]");
        }
        OrgVirtual parentOrg = null;
        if (TREE_ACTION_CREATE.equals(dto.getAction())) {
            if (StringUtils.isNotBlank(dto.getParentOrgCode())) {
                parentOrg = orgVirtualService.findByTenantIdAndOrgCode(tenantId, dto.getParentOrgCode());
                if (null == parentOrg) {
                    errMsg.add("不存在的上级业务组织[" + dto.getParentOrgCode() + "]");
                }
            }
        }
        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }

        switch (dto.getAction()) {
            case TREE_ACTION_CREATE:
                try {
                    this.importBuildNode(tenantId, typeId, orgVirtual, parentOrg);
                } catch (Exception ex) {
                    errMsg.add(ex.getMessage());
                }
                break;
            case TREE_ACTION_DELETE:
                try {
                    this.importRemoveNode(tenantId, typeId, orgVirtual);
                } catch (Exception ex) {
                    errMsg.add(ex.getMessage());
                }
                break;
            default:
                break;
        }
        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }
        return null;
    }

    @Transactional(rollbackFor = Exception.class)
    public void importRemoveNode(Long tenantId, long typeId, OrgVirtual orgVirtual) {
        if (orgVirtual == null) {
            return;
        }
        Optional<OrgVirtualNode> temp = orgVirtualNodeDao.findByNodeTypeIdAndOrgVirtualId(typeId, orgVirtual.getOrgVirtualId());
        if (temp.isPresent()) {
            this.currentDelete(tenantId, temp.get().getOrgVirtualNodeId());
        } else {
            String message = "不存在的业务组织[" + orgVirtual.getOrgCode() + "]";
            log.info(message);
            throw new IllegalArgumentException(message);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void importBuildNode(Long tenantId, long typeId, OrgVirtual orgVirtual, OrgVirtual parentOrg) {
        if (null == parentOrg) {
            this.currentCreate(typeId, tenantId, orgVirtual.getOrgVirtualId());
        } else {
            Optional<OrgVirtualNode> temp = orgVirtualNodeDao.findByNodeTypeIdAndOrgVirtualId(typeId, parentOrg.getOrgVirtualId());
            if (!temp.isPresent()) {
                String message = "不存在的上级业务组织[" + orgVirtual.getOrgCode() + "]";
                log.info(message);
                throw new IllegalArgumentException(message);
            }
            OrgVirtualNodeModel.Request.Append append = new OrgVirtualNodeModel.Request.Append();
            append.setOrgVirtualId(orgVirtual.getOrgVirtualId());
            append.setTenantId(tenantId);
            append.setParentId(temp.get().getOrgVirtualNodeId());
            this.currentAppend(append);
        }
    }

    public List<OrgVirtualModel.Response.Model> listByUserId(Long typeId, Long userId, Long tenantId) {
        return orgVirtualNodeDao.listByUserId(typeId, userId, tenantId);
    }

    public List<OrgVirtualModel.Response.Model> listByNodeId(Long nodeId, Long tenantId, TreeModel.Scope scope) {
        OrgVirtualNode node = orgVirtualNodeDao.findByTenantIdAndOrgVirtualNodeId(tenantId, nodeId).orElseThrow(() -> new IllegalArgumentException("未找到业务组织树（" + nodeId + ")"));
        return orgVirtualNodeDao.listByNodeId(node, scope);
    }

    public List<OrgVirtualModel.Response.Model> listByKeys(Long tenantId, TreeModel.Scope scope, Integer nodeTypeId, boolean byId, Set<String> keySet) {
        List<OrgVirtualModel.Response.Model> models = orgVirtualNodeDao.listByKeys(tenantId, nodeTypeId, byId, keySet);
        if (TreeModel.Scope.SELF.equals(scope)) {
            return models;
        }
        for (OrgVirtualModel.Response.Model model : models) {
            OrgVirtualNode node = new OrgVirtualNode();
            node.setNodeTypeId(model.getNodeTypeId());
            node.setOrgVirtualNodeId(model.getOrgVirtualNodeId());
            node.setParentId(model.getParentId());
            node.setParentIds(model.getParentIds());
            node.setTenantId(model.getTenantId());

            List<OrgVirtualModel.Response.Model> list = orgVirtualNodeDao.listByNodeId(node, scope);
            list = list.stream().filter(m -> !m.getOrgVirtualNodeId().equals(model.getOrgVirtualNodeId())).collect(Collectors.toList());
            switch (scope) {
                case PARENTS:
                    model.setParents(list);
                    break;
                case CHILDREN:
                    model.setChildren(list);
                    break;
                default:
                    break;
            }
        }
        return models;
    }

    public OrgVirtualModel.Response.Model treeByNodeId(Long nodeId, Long tenantId, TreeModel.Scope scope) {
        List<OrgVirtualModel.Response.Model> list = this.listByNodeId(nodeId, tenantId, scope);
        return this.buildTree(list);
    }

    private OrgVirtualModel.Response.Model buildTree(List<OrgVirtualModel.Response.Model> orgs) {
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }

        Map<Long, OrgVirtualModel.Response.Model> datas = orgs.stream().collect(Collectors.toMap(OrgVirtualModel.Response.Model::getOrgVirtualNodeId, org -> org));

        OrgVirtualModel.Response.Model root = orgs.get(0);

        for (OrgVirtualModel.Response.Model ele : orgs) {
            int level = StringHelp.getCount(ele.getParentIds(), Separator.BACKSLASH) + 1;
            ele.setLevel(level);

            if (null == datas.get(ele.getParentId()) && ele.getParentIds().endsWith(ele.getOrgVirtualNodeId() + Separator.BACKSLASH)) {
                continue;
            }
            if (null != ele.getParentId()) {
                OrgVirtualModel.Response.Model parent = datas.get(ele.getParentId());
                if (CollectionUtils.isEmpty(parent.getChildren())) {
                    Set<OrgVirtualModel.Response.Model> children = Stream.of(ele).collect(Collectors.toSet());
                    parent.setChildren(children);
                } else {
                    parent.getChildren().add(ele);
                }
            } else {
                root = ele;
            }
        }
        return root;
    }

}
