package com.xforceplus.business.org.virtual.service;

import static com.xforceplus.business.org.virtual.service.OrgVirtualOrgStructImportServiceImpl.ACTION_CREATE;

import com.xforceplus.api.model.OrgVirtualOrgStructRelModel;
import com.xforceplus.api.model.OrgVirtualOrgStructRelModel.Request.ExcelExportQuery;
import com.xforceplus.api.model.OrgVirtualOrgStructRelModel.Request.OrgStructSimple;
import com.xforceplus.api.model.OrgVirtualOrgStructRelModel.Request.Query;
import com.xforceplus.api.model.OrgVirtualOrgStructRelModel.Request.Relate;
import com.xforceplus.api.model.OrgVirtualOrgStructRelModel.Response.ExcelExportModel;
import com.xforceplus.business.excel.ExcelValidator;
import com.xforceplus.business.org.virtual.dto.OrgVirtualOrgStructImportDto;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.dao.OrgVirtualDao;
import com.xforceplus.dao.OrgVirtualOrgStructDao;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.entity.OrgVirtual;
import com.xforceplus.entity.OrgVirtualOrgStructRel;
import com.xforceplus.query.OrgVirtualOrgStructRelQueryHelper;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;


/**
* @copyright © 上海云砺信息科技有限公司
* @author zhouxin
* @date 2021/8/4 11:01
*/
@Service
@Slf4j
public class OrgVirtualOrgStructService {

    @Autowired
    private OrgVirtualOrgStructDao orgVirtualOrgStructDao;

    @Autowired
    private OrgVirtualDao orgVirtualDao;

    @Autowired
    private OrgStructDao orgStructDao;


    public Page<OrgVirtualOrgStructRel> page(OrgVirtualOrgStructRelModel.Request.Query query, Pageable pageable) {
        Specification<OrgVirtualOrgStructRel> specification = OrgVirtualOrgStructRelQueryHelper.querySpecification(query);
        return orgVirtualOrgStructDao.findAll(specification, pageable);
    }

    public Boolean saveAllAndFlush(List<OrgVirtualOrgStructRel> list) {
        orgVirtualOrgStructDao.saveAllAndFlush(list);
        return true;
    }

    @Transactional(rollbackFor = Exception.class)
    public Boolean relate(OrgVirtualOrgStructRelModel.Request.Relate model) {
        this.validOrg(model.getTenantId(), model.getOrgStructs());
        if(Objects.isNull(model.getOrgVirtualId())) {
            throw new IllegalArgumentException("业务组织id不能为空!");
        }

        orgVirtualOrgStructDao.deleteByTenantIdAndOrgVirtualId(model.getTenantId(), model.getOrgVirtualId());

        List<OrgVirtualOrgStructRel> list = new ArrayList<>();
        model.getOrgStructs()
             .forEach(orgStructSimple -> {
                 OrgVirtualOrgStructRel rel = new OrgVirtualOrgStructRel();
                 rel.setTenantId(model.getTenantId());
                 rel.setOrgVirtualId(model.getOrgVirtualId());
                 rel.setOrgStructId(orgStructSimple.getOrgStructId());
                 rel.setOrgStructName(orgStructSimple.getOrgStructName());
                 rel.setCreateTime(new Date());
                 rel.setUpdateTime(new Date());
                 list.add(rel);
             });

        return this.saveAllAndFlush(list);
    }

    private void validOrg(Long tenantId, List<OrgStructSimple> orgStructs) {
        if (CollectionUtils.isEmpty(orgStructs)) {
            throw new IllegalArgumentException("行政组织不能为空");
        }
        orgStructs.forEach(orgStructSimple -> {
            if(Objects.isNull(orgStructSimple.getOrgStructId())) {
                throw new IllegalArgumentException("行政组织id不能为空!");
            }
            Long orgStructId = orgStructDao.findOrgIdByTenantIdAndOrgId(tenantId, orgStructSimple.getOrgStructId());
            if(Objects.isNull(orgStructId)) {
                throw new IllegalArgumentException("未找到组织实体(tenantId:" + tenantId + ", orgId:" + orgStructSimple.getOrgStructId());
            }
        });
    }

    @Transactional(rollbackFor = Exception.class)
    public String importSave(OrgVirtualOrgStructImportDto dto, Long tenantId) {
        List<String> errMsg = new ArrayList<>();

        ExcelValidator.validate(dto);

        if (StringUtils.isNotBlank(dto.getValidatedMessage())) {
            errMsg.add(dto.getValidatedMessage());
        }
        OrgStruct orgStruct = null;
        if (StringUtils.isNotBlank(dto.getOrgStructCode())) {
            List<OrgStruct> orgStructs = orgStructDao.findByTenantIdAndOrgCode(tenantId, dto.getOrgStructCode());
            if (CollectionUtils.isEmpty(orgStructs)) {
                errMsg.add("不存在的组织[" + dto.getOrgStructCode() + "]");
            } else {
                orgStruct = orgStructs.get(0);
                switch (orgStruct.getStatus()) {
                    case 0:
                        errMsg.add("该组织未启用[" + dto.getOrgStructCode() + "]");
                        break;
                    case 1:
                        break;
                    case 2:
                        errMsg.add("该组织已禁用[" + dto.getOrgStructCode() + "]");
                        break;
                    default:
                        break;
                }
            }
        }

        Optional<OrgVirtual> orgVirtualOpt = orgVirtualDao.findByTenantIdAndOrgCode(tenantId,dto.getOrgVirtualCode());
        if(!orgVirtualOpt.isPresent()) {
            errMsg.add("不存在的业务组织[" + dto.getOrgVirtualCode() + "]");
        }

        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }

        if (ACTION_CREATE.equals(dto.getAction())) {
            OrgVirtualOrgStructRelModel.Request.Relate model = new Relate();
            model.setTenantId(tenantId);
            OrgStructSimple orgStructSimple = new OrgStructSimple();
            orgStructSimple.setOrgStructId(orgStruct.getOrgId());
            orgStructSimple.setOrgStructName(orgStruct.getOrgName());
            model.setOrgStructs(Arrays.asList(orgStructSimple));
            model.setOrgVirtualId(orgVirtualOpt.get().getOrgVirtualId());

            try {
                this.relate(model);
            } catch (Exception e) {
                errMsg.add(e.getMessage());
            }

        }

        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }
        return null;
    }

    public List<ExcelExportModel> listToExcel(ExcelExportQuery excelQuery) {
        OrgVirtualOrgStructRelModel.Request.Query query = new Query();
        BeanUtils.copyProperties(excelQuery, query);
        Specification<OrgVirtualOrgStructRel> specification = OrgVirtualOrgStructRelQueryHelper.querySpecification(query);
        List<OrgVirtualOrgStructRel> list = orgVirtualOrgStructDao.findAll(specification);
        if(!CollectionUtils.isEmpty(list)) {
            List<ExcelExportModel> ret = new ArrayList<>();
            for(OrgVirtualOrgStructRel rel : list) {
                ExcelExportModel model = new ExcelExportModel();
                BeanUtils.copyProperties(rel, model);
                ret.add(model);
            }
            return ret;
        }
        return null;
    }

    @Async("threadPoolExecutor")
    public void asyncRelateParentOrgVirtual(OrgStruct orgStruct) {
        try{
            Long parentOrgId = orgStruct.getParentId();
            if(Objects.nonNull(parentOrgId)) {
                OrgVirtualOrgStructRelModel.Request.Query query = new Query();
                query.setTenantId(orgStruct.getTenantId());
                query.setOrgStructId(parentOrgId);
                Specification<OrgVirtualOrgStructRel> specification = OrgVirtualOrgStructRelQueryHelper.querySpecification(query);
                List<OrgVirtualOrgStructRel> rels = orgVirtualOrgStructDao.findAll(specification);
                if(!CollectionUtils.isEmpty(rels)) {
                    List<OrgVirtualOrgStructRel> list = new ArrayList<>();
                    for(OrgVirtualOrgStructRel rel : rels) {
                        OrgVirtualOrgStructRel oo = new OrgVirtualOrgStructRel();
                        oo.setOrgStructId(orgStruct.getOrgId());
                        oo.setTenantId(orgStruct.getTenantId());
                        oo.setOrgVirtualId(rel.getOrgVirtualId());
                        oo.setOrgStructName(orgStruct.getOrgName());
                        oo.setCreateTime(new Date());
                        oo.setUpdateTime(new Date());
                        list.add(oo);
                    }
                    orgVirtualOrgStructDao.saveAllAndFlush(list);
                }
            }
        } catch (Exception e) {
            log.error("error relate parent org virtual", e.getMessage());
        }
    }
}
