package com.xforceplus.business.org.virtual.service;

import com.xforceplus.api.model.OrgVirtualModel;
import com.xforceplus.business.excel.ExcelValidator;
import com.xforceplus.business.org.virtual.dto.OrgVirtualImportDto;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.dao.OrgVirtualCustomizedDao;
import com.xforceplus.dao.OrgVirtualDao;
import com.xforceplus.dao.OrgVirtualNodeDao;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.entity.OrgVirtual;
import com.xforceplus.entity.OrgVirtualNode;
import com.xforceplus.query.OrgVirtualQueryHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static com.xforceplus.business.org.virtual.service.OrgVirtualImportServiceImpl.ACTION_CREATE;
import static com.xforceplus.business.org.virtual.service.OrgVirtualImportServiceImpl.ACTION_UPDATE;


/**
 * 版权：    上海云砺信息科技有限公司
 * author:  duanhy
 * 创建时间: 2020/12/22 16:16
 * 功能描述:
 * 修改历史:
 */
@Service
@Slf4j
public class OrgVirtualService {

    private final OrgVirtualDao orgVirtualDao;

    private final OrgVirtualCustomizedDao customizedOrgVirtualDao;
    private final OrgService orgService;
    private final OrgVirtualNodeDao orgVirtualNodeDao;

    private static final Integer ORG_VIRTUAL_DISABLE = 0;

    public OrgVirtualService(OrgVirtualDao orgVirtualDao, OrgVirtualCustomizedDao customizedOrgVirtualDao, OrgService orgService, OrgVirtualNodeDao orgVirtualNodeDao) {
        this.orgVirtualDao = orgVirtualDao;
        this.customizedOrgVirtualDao = customizedOrgVirtualDao;
        this.orgService = orgService;
        this.orgVirtualNodeDao = orgVirtualNodeDao;
    }

    public Page<OrgVirtual> page(OrgVirtualModel.Request.Query query, Pageable pageable) {
        Specification<OrgVirtual> specification = OrgVirtualQueryHelper.querySpecification(query);
        return orgVirtualDao.findAll(specification, pageable);
    }

    public OrgVirtual findByTenantIdAndId(Long tenantId, Long virtualId) {
        return orgVirtualDao.findByTenantIdAndOrgVirtualId(tenantId, virtualId).orElseThrow(() -> new IllegalArgumentException("未找到业务组织（" + virtualId + ")"));
    }

    public OrgVirtual findByTenantIdAndOrgCode(Long tenantId, String orgCode) {
        return orgVirtualDao.findByTenantIdAndOrgCode(tenantId, orgCode).orElse(null);
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgVirtual create(OrgVirtualModel.Request.Create model) {
        this.validOrg(model);
        Optional<OrgVirtual> temp = orgVirtualDao.findByTenantIdAndOrgCode(model.getTenantId(), model.getOrgCode());
        if (temp.isPresent()) {
            throw new IllegalArgumentException("已存在业务组织代码(" + model.getOrgCode() + ")");
        }
        OrgVirtual orgVirtual = new OrgVirtual();
        BeanUtils.copyProperties(model, orgVirtual);
        return orgVirtualDao.saveAndFlush(orgVirtual);
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgVirtual update(OrgVirtualModel.Request.Update model) {
        OrgVirtual orgVirtual = this.findByTenantIdAndId(model.getTenantId(), model.getOrgVirtualId());
        this.validOrg(model);

        if (model.getStatus() != null) {
            if (ORG_VIRTUAL_DISABLE.equals(model.getStatus())) {
                //如果已经关联到了业务组织树上，节点不允许禁用
                List<OrgVirtualNode> orgVirtualNodeList = orgVirtualNodeDao.findByTenantIdAndOrgVirtualId(model.getTenantId(), model.getOrgVirtualId());
                if (!CollectionUtils.isEmpty(orgVirtualNodeList)) {
                    throw new IllegalArgumentException("该业务组织已关联组织树，不允许禁用(" + model.getOrgVirtualId() + ")");
                }
            }

            orgVirtual.setStatus(model.getStatus());
        }

        orgVirtual.setOrgName(model.getOrgName());
        orgVirtual.setOrgDesc(model.getOrgDesc());
        orgVirtual.setOrgStructId(model.getOrgStructId());
        orgVirtual.setShortName(model.getShortName());

        return orgVirtualDao.saveAndFlush(orgVirtual);
    }

    private void validOrg(OrgVirtualModel.Request.Save save) {
        if (null != save.getOrgStructId()) {
            orgService.findByTenantIdAndId(save.getTenantId(), save.getOrgStructId());
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgVirtual currentEnable(Long tenantId, Long orgVirtualId) {
        OrgVirtual orgVirtual = this.findByTenantIdAndId(tenantId, orgVirtualId);
        orgVirtual.setStatus(1);
        orgVirtual = orgVirtualDao.saveAndFlush(orgVirtual);
        return orgVirtual;
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgVirtual currentDisable(Long tenantId, Long orgVirtualId) {
        OrgVirtual orgVirtual = this.findByTenantIdAndId(tenantId, orgVirtualId);
        List<OrgVirtualNode> trees = orgVirtualNodeDao.findByTenantIdAndOrgVirtualId(tenantId, orgVirtualId);
        if (!CollectionUtils.isEmpty(trees)) {
            throw new IllegalArgumentException("该业务组织已关联组织树，不允许禁用(" + orgVirtualId + ")");
        }
        orgVirtual.setStatus(0);
        orgVirtual = orgVirtualDao.saveAndFlush(orgVirtual);
        return orgVirtual;
    }

    public OrgVirtual currentInfo(Long tenantId, Long orgVirtualId) {
        return this.findByTenantIdAndId(tenantId, orgVirtualId);
    }

    public List<OrgVirtualModel.Response.ExcelExportModel> listToExcel(OrgVirtualModel.Request.ExcelExportQuery query) {
        return customizedOrgVirtualDao.listToExcel(query);
    }

    @Transactional(rollbackFor = Exception.class)
    public String importSave(OrgVirtualImportDto dto, Long tenantId) {

        List<String> errMsg = new ArrayList<>();

        ExcelValidator.validate(dto);

        if (StringUtils.isNotBlank(dto.getValidatedMessage())) {
            errMsg.add(dto.getValidatedMessage());
        }
        Long orgId = null;
        if (StringUtils.isNotBlank(dto.getOrgStructCode())) {
            List<OrgStruct> orgStructs = orgService.findByTenantIdAndOrgCode(tenantId, dto.getOrgStructCode());
            if (CollectionUtils.isEmpty(orgStructs)) {
                errMsg.add("不存在的组织[" + dto.getOrgStructCode() + "]");
            } else {
                OrgStruct orgStruct = orgStructs.get(0);
                switch (orgStruct.getStatus()) {
                    case 0:
                        errMsg.add("该组织未启用[" + dto.getOrgStructCode() + "]");
                        break;
                    case 1:
                        orgId = orgStruct.getOrgId();
                        break;
                    case 2:
                        errMsg.add("该组织已禁用[" + dto.getOrgStructCode() + "]");
                        break;
                    default:
                        break;
                }
            }
        }

        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }

        if (ACTION_CREATE.equals(dto.getAction())) {
            OrgVirtualModel.Request.Create model = new OrgVirtualModel.Request.Create();
            model.setTenantId(tenantId);
            model.setOrgCode(dto.getOrgCode());
            model.setShortName(dto.getShortName());
            model.setOrgDesc(dto.getOrgDesc());
            model.setStatus(dto.getStatus());
            model.setOrgName(dto.getOrgName());
            model.setOrgStructId(orgId);
            try {
                this.create(model);
            } catch (Exception e) {
                errMsg.add(e.getMessage());
            }

        } else if (ACTION_UPDATE.equals(dto.getAction())) {
            Optional<OrgVirtual> optionalOrgVirtual = orgVirtualDao.findByTenantIdAndOrgCode(tenantId, dto.getOrgCode());
            if (optionalOrgVirtual.isPresent()) {
                boolean isAdd = true;
                OrgVirtual orgVirtual = optionalOrgVirtual.get();
                if (dto.getStatus() == 0) {
                    List<OrgVirtualNode> trees = orgVirtualNodeDao.findByTenantIdAndOrgVirtualId(tenantId, orgVirtual.getOrgVirtualId());
                    if (!CollectionUtils.isEmpty(trees)) {
                        errMsg.add("该业务组织已关联组织树，不允许禁用");
                        isAdd = false;
                    }
                }
                if (isAdd) {
                    orgVirtual.setShortName(dto.getShortName());
                    orgVirtual.setOrgDesc(dto.getOrgDesc());
                    orgVirtual.setStatus(dto.getStatus());
                    orgVirtual.setOrgName(dto.getOrgName());
                    orgVirtual.setOrgCode(dto.getOrgCode());
                    orgVirtual.setOrgStructId(orgId);
                    orgVirtualDao.saveAndFlush(orgVirtual);
                }
            } else {
                errMsg.add("不存在的业务组织[" + dto.getOrgCode() + "]");
            }
        }

        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }
        return null;
    }

}
