/*
 * Copyright (c)  2015~2021, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: PubSubAsyncService.java   2021-01-13 15-33-21
 * Author: Evan
 */
package com.xforceplus.business.pub.service;

import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.eventbus.SubscriberExceptionHandler;
import com.xforceplus.business.pub.dto.PubContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * <p>
 * Title: 异步发送消息 MQ
 * </p>
 * <p>
 * Description: 异步发送消息 MQ
 * </p>
 * <p>
 * Copyright: 2015~2021
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2021-01-13 15-33-21
 * @since V1.0
 */
@Slf4j
@Service
public class PubSubAsyncService implements InitializingBean, DisposableBean {

    /**
     * EventBus Name:{@value}
     */
    public static final String ASYNC_PUS_SUB_EVENT_BUS = "ASYNC_PUS_SUB_EVENT_BUS";
    @SuppressWarnings("all")
    @Autowired(required = false)
    private PubsubService pubsubService;
    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;
    /**
     * 创建线程池
     */
    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    /**
     * Invoked by the containing {@code BeanFactory} on destruction of a bean.
     *
     * @throws Exception in case of shutdown errors. Exceptions will get logged
     *                   but not rethrown to allow other beans to release their resources as well.
     */
    @Override
    public void destroy() throws Exception {
//        if (this.threadPoolExecutor != null) {
//            this.threadPoolExecutor.shutdown();
//        }
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        SubscriberExceptionHandler subscriberExceptionHandler = new SubscriberExceptionHandlerImpl();
        //构建异步线程池
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, subscriberExceptionHandler);
        //异步注册
        asyncEventBus.register(new AsyncPubSubProcessListener());
    }

    /**
     * 发送消息
     *
     * @param pubContext 上下文
     */
    @Async("threadPoolExecutor")
    public void send(PubContext pubContext) {
        if (this.pubsubService == null) {
            log.warn("PubSubAsyncService.pubsub.enable:{}", false);
        }
        this.asyncEventBus.post(pubContext);
    }


    /**
     * 事件监控处理器
     */
    protected class AsyncPubSubProcessListener {
        @Subscribe
        public void doProcess(PubContext context) {
            try {
                String messageId = pubsubService.sendMessage(context.getCode(),
                        context.getContent(),
                        context.getParam());
                log.info("pubsub code:{},messageId:{}", context.getCode(), messageId);
            } catch (Exception e) {

            }
        }
    }

    public void sendMessage(String code, Map<String, String> tagkvs, String content) {
        PubContext pubContext = PubContext.builder()
                .code(code)
                .param(tagkvs)
                .content(content)
                .build();
        try {
            this.send(pubContext);
        } catch (Exception e) {
            log.error("xforce.pub.bus.error,requestName={}",code, e);
        }
    }
}
